import java.awt.image.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.util.*;
import java.awt.geom.*;
import java.io.*;

public class VisionTrain extends JFrame {
  int height=700, width=700;
  public ColorControlPanel controlPanel;
  public static ColorConverter colorConverter;

  public static void main(String args[]) {
    if (args.length<1) {
			usageAndExit();
    }
    colorConverter=new ColorConverter() {
      public final float[] getColor(int r, int g, int b) {
        float[] hsb=new float[3];
        Color.RGBtoHSB(r, g, b, hsb);
        float[] res=new float[2];
        res[0]=hsb[0];
        res[1]=hsb[1];
        return res;
      }
    };
    
		boolean isRGB=true;
		if(args[0].equals("-isRGB"))
			isRGB=true;
		else if(args[0].equals("-isYUV"))
			isRGB=false;
		else {
			System.out.println(args[0]+" is not valid color mode");
			usageAndExit();
		}
		
		String files[]=new String[args.length-1];
		for(int i=0; i<files.length; i++)
			files[i]=args[i+1];

    VisionTrain visionTrain=new VisionTrain(isRGB,files);
    visionTrain.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent e) { System.exit(0); } });
    visionTrain.controlPanel.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent e) { System.exit(0); } });
  }

	public static void usageAndExit() {
		System.out.println("usage: java VisionTrain (-isRGB|-isYUV) filename [filename ..]");
		System.out.println("       Using RGB images will minimize color space conversions.");
		System.out.println("       A mode must be specified.");
		System.exit(1);
	}
 
  public VisionTrain(boolean isRGB, String files[]) {
		System.out.println("Interpreting images as "+(isRGB?"RGB":"YUV")+" colorspace");
    ImageShow imageShow=new ImageShow(isRGB,files);
    
    setBackground(Color.black);
    setSize(new Dimension(width, height));
    Container root=this.getContentPane();
    root.setLayout(new BorderLayout());
    TrainCanvas trainCanvas=new TrainCanvas();
    root.add(trainCanvas);
    addKeyListener(trainCanvas);
    show();

    controlPanel=new ColorControlPanel(trainCanvas, imageShow);

    ImageData imageData=new ImageData();

		//Thanks to martin.mueller at mni.fh-giessen.de for the bug fix:
		if(isRGB)
			imageData.loadFullRGBFilesAsRGB(files);
		else 
			imageData.loadFullYUVFilesAsRGB(files);
    trainCanvas.plotImage(imageData.getHS(), imageData.getPixels());
  }
}

class ColorControlPanel extends JFrame implements ActionListener,
    ListSelectionListener {
  Container root;
  JTextField colorname;
  JList colorlist;
  DefaultListModel list;
  JScrollPane colorlistscroll;
  JButton remove, clear, save, imageview, invert;
  int curcolor;

  TrainCanvas trainCanvas;
  ImageShow imageShow;

  public ColorControlPanel (TrainCanvas trainCanvas, ImageShow imageShow) {
    this.trainCanvas=trainCanvas;
    this.imageShow=imageShow;
    imageShow.addMouseMotionListener(trainCanvas);
    imageShow.addMouseListener(trainCanvas);

    setSize(new Dimension (120,350));
    setLocation(800,50);

    setResizable(false);
    root=this.getContentPane();
    root.setLayout(new FlowLayout());

    colorname=new JTextField(10);
    colorname.addActionListener(this);
    root.add(colorname);

    list=new DefaultListModel();
    colorlist=new JList(list);
    colorlist.setFixedCellWidth(90);
    colorlist.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    colorlist.addListSelectionListener(this);

    colorlistscroll=new JScrollPane(colorlist,
                       JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
                       JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
    root.add(colorlistscroll);

    remove=new JButton("remove");
    remove.addActionListener(this);
    root.add(remove);

    clear=new JButton("clear");
    clear.addActionListener(this);
    root.add(clear);

    invert=new JButton("invert");
    invert.addActionListener(this);
    root.add(invert);

    imageview=new JButton("image view");
    imageview.addActionListener(this);
    root.add(imageview);

    save=new JButton("save");
    save.addActionListener(this);
    root.add(save);

    setCurColor(-1);
    show();
  }

  public void actionPerformed(ActionEvent e) {
    if (e.getSource()==save) {
      JFileChooser chooser=new JFileChooser();
      chooser.setSelectedFile(new File("default.tm"));
      int returnval=chooser.showSaveDialog(save);
      if (returnval==JFileChooser.APPROVE_OPTION) {
        trainCanvas.save(chooser.getSelectedFile().getName());   
      }
    } else if (e.getSource()==clear) {
      trainCanvas.clear();
    } else if (e.getSource()==invert) {
      trainCanvas.invert();
    } else if (e.getSource()==remove) {
      trainCanvas.remove((String)list.get(curcolor));
      list.remove(curcolor);
      setCurColor(-1);
    } else if (e.getSource()==imageview) {
      imageShow.show();
    } else if (e.getSource()==colorname) {
      String s=e.getActionCommand();
      if (!s.equals("")) {
        int i=0;
        while (i<list.getSize() && !list.get(i).equals(s)) {
          i++;
        }
        if (i==list.getSize()) {
          list.addElement(s);
          colorname.setText("");
          colorlist.setSelectedIndex(i);
        }
      }
    }
  }

  public void valueChanged(ListSelectionEvent e) {
    if (!e.getValueIsAdjusting()) {
      setCurColor(colorlist.getSelectedIndex());
    }
  }

  void setCurColor(int index) {
    curcolor=index;

    if (index<0) {
      remove.setEnabled(false);
      clear.setEnabled(false);
      trainCanvas.setCurColor(null);
    } else {
      remove.setEnabled(true);
      clear.setEnabled(true);
      trainCanvas.setCurColor((String)list.get(curcolor));
    }
  }
}

class TrainCanvas extends Canvas implements MouseListener,
    MouseMotionListener, KeyListener {
  BufferedImage image;
  BufferedImage cachedplot;
  BufferedImage inv_cachedplot;

  Graphics2D graphics;
  Polygon curPoly;
  int lastx, lasty;
  float f_width, f_height, f_offset;

  Area curArea;
  String curColor;
  int curSelectModifier;
  int curSelectModified;

  Point imageHint;

  Hashtable areasHash;

  boolean inverted;

  public static final int MODIFIER_NONE=0;
  public static final int MODIFIER_SHIFT=1;
  public static final int MODIFIER_CTRL=2;

  float[] xy;
  int[] rgb;

  public TrainCanvas() {
    setBackground(Color.black);
    addMouseListener(this);
    addMouseMotionListener(this);
    curSelectModifier=MODIFIER_NONE;
    curSelectModified=MODIFIER_NONE;
    addKeyListener(this);
    areasHash=new Hashtable();
    inverted=false;
    imageHint=null;
  }

  public void plotImage(float[] xy, int[] rgb) {
    this.xy=xy;
    this.rgb=rgb;
    Dimension d=getSize();
    f_width=(float)d.width-41;
    f_height=(float)d.height-41;
    f_offset=20;
    cachedplot=new BufferedImage(d.width, d.height, BufferedImage.TYPE_INT_RGB);
    inv_cachedplot=
               new BufferedImage(d.width, d.height, BufferedImage.TYPE_INT_RGB);

    graphics=(Graphics2D)cachedplot.getGraphics();
    Graphics2D inv_graphics=(Graphics2D)inv_cachedplot.getGraphics();

    inv_graphics.setColor(Color.white);
    inv_graphics.fillRect(0,0,d.width, d.height);

    for (int i=0; i<rgb.length; i++) {
      //      draws thin
      //      cachedplot.setRGB((int)(xy[i*2]*f_width),
      //                        (int)(xy[i*2+1]*f_height), rgb[i]);

      // draws thick
      graphics.setColor(new Color(rgb[i]));
      graphics.drawRect((int)(xy[i*2]*f_width+f_offset),
                        (int)(xy[i*2+1]*f_height+f_offset), 1,1);

      inv_graphics.setColor(new Color(rgb[i]));
      inv_graphics.drawRect((int)(xy[i*2]*f_width+f_offset),
                        (int)(xy[i*2+1]*f_height+f_offset), 1,1);
    }

    image=new BufferedImage(d.width, d.height, BufferedImage.TYPE_INT_RGB);
    graphics=(Graphics2D)image.getGraphics();
//    graphics.drawImage(cachedplot,0,0,this);
    redrawScene();
    repaint();
  }

  public void mousePressed(MouseEvent e) {
    if (curArea==null) return;
    if (e.getButton()!=MouseEvent.BUTTON1) return;

    curSelectModified=curSelectModifier;

    if (curSelectModified!=MODIFIER_SHIFT
        && curSelectModified!=MODIFIER_CTRL)
      curArea.reset();

    curPoly=new Polygon();
    lastx=e.getX();
    lasty=e.getY();
    curPoly.addPoint(e.getX(), e.getY());
    repaint();
  }

  public void mouseDragged(MouseEvent e) {
    if (curArea==null) return;

    int x=e.getX();
    int y=e.getY();
    if ((Math.abs(x-lastx)+Math.abs(y-lasty))>2) {
      curPoly.addPoint(e.getX(), e.getY());
      lastx=x;
      lasty=y;
      repaint();
    }
  }

  public void mouseReleased(MouseEvent e) {
    if (curArea==null) return;
    if (e.getButton()!=MouseEvent.BUTTON1) return;

    curPoly.addPoint(e.getX(), e.getY());
    if (curPoly.npoints>=3) {
      if (curSelectModified==MODIFIER_CTRL) curArea.subtract(new Area(curPoly));
      else curArea.add(new Area(curPoly));
    }
    curPoly=null;
    repaint();
  }

  public void mouseMoved(MouseEvent e){
    Object source=e.getSource();
    if (source instanceof ImageShow) {
      ImageShow imageShow=(ImageShow)source;
      int rgb=imageShow.getPixel(e.getX(), e.getY());
      float[] xy=VisionTrain.colorConverter.getColor(rgb);
      int x=(int)(xy[0]*f_width+f_offset);
      int y=(int)(xy[1]*f_height+f_offset);
      imageHint=new Point(x, y);
      repaint();
    }
  }

  public void mouseClicked(MouseEvent e){}
  public void mouseExited(MouseEvent e){
    if (e.getSource() instanceof ImageShow) {
      imageHint=null;
      repaint();
    }
  }
  public void mouseEntered(MouseEvent e){}
  public void updateLocation(MouseEvent e){}

  public void paint(Graphics g) {
    update(g);
  }

  public void update(Graphics g) {
    Graphics2D g2d=(Graphics2D) g;
    if (image!=null) g.drawImage(image, 0, 0, this);
    if (inverted)
      g2d.setColor(Color.black);
    else
      g2d.setColor(Color.white);

    if (curArea!=null)
      g2d.draw(curArea);

    if (curPoly!=null)
      g2d.draw(curPoly);

    if (imageHint!=null) {
      g2d.drawRect(imageHint.x-2, imageHint.y-2, 5, 5);
    }
  }

  public void setCurColor(String color) {
    curColor=color;
    if (color==null) {
      curArea=null;
      return;
    }

    curArea=(Area)areasHash.get(color);
    if (curArea==null) {
      curArea=new Area();
      areasHash.put(color, curArea);
    }
    redrawScene();
  }

  public void clear () {
    if (curArea==null) return;
    curArea.reset();
    redrawScene();
  }

  public void invert () {
    inverted=!inverted;
    redrawScene();
  }

  public void remove(String color) {
    curColor=null;
    curArea=null;
    areasHash.remove(color);
    redrawScene();
  }

  public void redrawScene() {
    if (inverted)
      graphics.drawImage(inv_cachedplot,0,0,this);
    else 
      graphics.drawImage(cachedplot,0,0,this);
    graphics.setColor(Color.gray);
    Collection values=areasHash.values();
    for (Iterator i=values.iterator(); i.hasNext(); ) {
      Area area=(Area)i.next();
      if (area!=curArea)
        graphics.draw(area);
    }
    repaint();  
  }

  public void keyPressed(KeyEvent e) {
    if (e.getKeyCode()==KeyEvent.VK_SHIFT) {
      curSelectModifier=MODIFIER_SHIFT;
    } else if (e.getKeyCode()==KeyEvent.VK_CONTROL) {
      curSelectModifier=MODIFIER_CTRL;
    }
  }

  public void keyReleased(KeyEvent e) {
    if (e.getKeyCode()==KeyEvent.VK_SHIFT
        ||e.getKeyCode()==KeyEvent.VK_CONTROL) {
      curSelectModifier=MODIFIER_NONE;
    }
  }

  public void keyTyped(KeyEvent e) { }

  public void save (String filename) {
    int dotpos=filename.lastIndexOf('.');
    if (dotpos>0) filename=filename.substring(0,dotpos);
    
    Enumeration colors=areasHash.keys();
    ArrayList colornames=new ArrayList(20);
    while (colors.hasMoreElements()) colornames.add(colors.nextElement());
    Collections.sort(colornames);
    ArrayList areas=new ArrayList(20);

    for (Iterator i=colornames.iterator(); i.hasNext();
         areas.add(areasHash.get(i.next())));
    
    int size_y=16, size_u=64, size_v=64;
    byte[] tmap=new byte[size_y*size_u*size_v];

    float[] hsb=new float[3];
    int y, u, v, r, g, b, h, s;
    int iy, iu, iv, i;

    for (iy=0; iy<16; iy++) {
      for (iu=0; iu<64; iu++) {
        for (iv=0; iv<64; iv++) {
          y=iy<<4; u=iu<<2; v=iv<<2;
          u=u*2-255;
          v=v*2-255;
          r=y+u;
          b=y+v;
          u=u>>1;
          v=(v>>2)-(v>>4);
          g=y-u-v;
          if (r<0) r=0; if (g<0) g=0; if (b<0) b=0;
          if (r>255) r=255; if (g>255) g=255; if (b>255) b=255;
          float[] xy=VisionTrain.colorConverter.getColor(r, g, b);
          h=(int)(xy[0]*f_width+f_offset);
          s=(int)(xy[1]*f_height+f_offset);
          Point point=new Point(h, s);
          for (i=0; i<areas.size(); i++) {
            Area area = (Area) areas.get(i);
            if (area.contains(point)) {
              tmap[(iy*size_u+iu)*size_v+iv]=(byte)(i+1);
              break;
            }
          }
        }
      }
    }

    try {
      FileOutputStream file_tm_fos=new FileOutputStream(filename + ".tm");
      OutputStreamWriter file_tm_osw=new OutputStreamWriter(file_tm_fos);
      file_tm_osw.write("TMAP\nYUV8\n" +
          size_y + " " + size_u + " " + size_v + "\n");
      file_tm_osw.flush();
      file_tm_fos.write(tmap);
      file_tm_osw.close();
    } catch (Exception ex) {
      System.out.println("Error saving to "+filename +".tm: " + ex);
    }

    int[][] avgcolors=new int[areas.size()][4]; 

    int skip=(rgb.length/10000)+1;
    for (i=0; i<rgb.length; i+=skip) {
      for (int a=0; a<areas.size(); a++) {
        Area area=(Area)areas.get(a);
        Point p=new Point((int)(xy[i*2]*f_width+f_offset),
                          (int)(xy[i*2+1]*f_height+f_offset));
        if (area.contains(p)) {
          avgcolors[a][0]+=(rgb[i]>>16)&0xff;
          avgcolors[a][1]+=(rgb[i]>>8)&0xff;
          avgcolors[a][2]+=rgb[i]&0xff;
          avgcolors[a][3]++;
        }
      }
    }

    for (i=0; i<avgcolors.length; i++) {
      if (avgcolors[i][3]>0) {
        avgcolors[i][0]=avgcolors[i][0]/avgcolors[i][3];
        avgcolors[i][1]=avgcolors[i][1]/avgcolors[i][3];
        avgcolors[i][2]=avgcolors[i][2]/avgcolors[i][3];
      } else {
        avgcolors[i][0]=0;
        avgcolors[i][1]=0;
        avgcolors[i][2]=0;
      }
    }
 
    try {
      FileWriter file_col_fw=new FileWriter(filename + ".col");
      file_col_fw.write("0 (128 128 128) \"unclassified\" 8 1.00\n");
      for (i=0; i<areas.size(); i++) {
        file_col_fw.write((i+1)+ " (" +
                          avgcolors[i][0] + " " +
                          avgcolors[i][1] + " " +
                          avgcolors[i][2] + ") " +
                          "\"" + colornames.get(i)+ "\" 8 0.75\n");
      }

      file_col_fw.close();
    } catch (Exception ex) {
      System.out.println("Error saving to "+filename + ".col: " + ex);
    }
  }
}
