#include "StartupBehavior.h"

#include "Shared/ProjectInterface.h"
#include "Shared/Config.h"

#include "Behaviors/Controls/BehaviorSwitchControl.h"

#include "Vision/RawCameraGenerator.h"
#include "Vision/InterleavedYUVGenerator.h"
#include "Vision/JPEGGenerator.h"
#include "Vision/SegmentedColorGenerator.h"
#include "Vision/RLEGenerator.h"
#include "Vision/RegionGenerator.h"
#include "Vision/BallDetectionGenerator.h"
#include "Vision/CDTGenerator.h"

using namespace ProjectInterface;

BallDetectionGenerator * pball=NULL;
BallDetectionGenerator * bball=NULL;
BallDetectionGenerator * handball=NULL;

/*! We set the default vision generators and source IDs here.
 *
 * The vis*SID and def*Generator variables are defined in
 * ProjectInterface.  You can reassign their values as you see fit,
 * which allows you to reorganize and adapt the vision pipeline, but
 * still retain the ability of the included demo behaviors to access
 * vision information.
 */
void
StartupBehavior::initVision() {
	//The value you set here will define the number of resolution layers available throughout the vision pipeline
	//If you change this value, you should modify the *Layer variables in ProjectInterface (if you want to use them)
	unsigned int numLayers=6;

	//The top level layer will be double resolution
	//This will set how many real layers are provided by the system (3 in Aperios)
	//Any other layers will be subsamples of the lowest resolution system resolution
	unsigned int numSystemLayers=3;

	defRawCameraGenerator = new RawCameraGenerator(numSystemLayers,numLayers,EventBase::visOFbkEGID,0,visRawCameraSID);
	defRawCameraGenerator->DoStart();
	spawned.push_back(defRawCameraGenerator);

	defInterleavedYUVGenerator = new InterleavedYUVGenerator(EventBase::visRawCameraEGID,visRawCameraSID,visInterleaveSID,RawCameraGenerator::CHAN_Y,RawCameraGenerator::CHAN_U,RawCameraGenerator::CHAN_V);
	defInterleavedYUVGenerator->DoStart();
	spawned.push_back(defInterleavedYUVGenerator);

	defColorJPEGGenerator = new JPEGGenerator(EventBase::visInterleaveEGID,visInterleaveSID,visColorJPEGSID,JPEGGenerator::SRC_AUTO);
	defColorJPEGGenerator->DoStart();
	defColorJPEGGenerator->setName("ColorJPEGGenerator");
	spawned.push_back(defColorJPEGGenerator);

	defGrayscaleJPEGGenerator = new JPEGGenerator(EventBase::visRawCameraEGID,visRawCameraSID,visGrayscaleJPEGSID,JPEGGenerator::SRC_AUTO);
	defGrayscaleJPEGGenerator->DoStart();
	defGrayscaleJPEGGenerator->setName("GrayscaleJPEGGenerator");
	spawned.push_back(defGrayscaleJPEGGenerator);

	// the hardware level CDT generator allows faster, but less flexible
	// segmenting it still needs a little work though before it can be
	// hooked up to the CMVision RLE generator.  See CDTGenerator class notes.
	// defSegmentedColorGenerator = new CDTGenerator(numSystemLayers,numLayers,EventBase::visOFbkEGID,0,visSegmentSID);
	SegmentedColorGenerator * segcol = new SegmentedColorGenerator(EventBase::visRawCameraEGID,visRawCameraSID,visSegmentSID);
	defSegmentedColorGenerator = segcol;
	segcol->loadColorInfo(config->vision.colors);
	for(unsigned int i=0; i<config->vision.thresh.size(); i++)
		segcol->loadThresholdMap(config->vision.thresh[i]);
	if(config->vision.colors!="" && config->vision.thresh.size()>0) {
		defSegmentedColorGenerator->DoStart();
		spawned.push_back(defSegmentedColorGenerator);
	}

	defRLEGenerator = new RLEGenerator(EventBase::visSegmentEGID,visSegmentSID,visRLESID);
	defRLEGenerator->DoStart();
	spawned.push_back(defRLEGenerator);
	
	defRegionGenerator = new RegionGenerator(EventBase::visRLEEGID,visRLESID,visRegionSID);
	defRegionGenerator->DoStart();
	spawned.push_back(defRegionGenerator);
	
	// for lack of a better idea, the object recognizers below will all
	// use the config->vision.rlecam_channel for picking the threshold
	// file to use
	// note that this is done here just once with the initial value, but
	// the detectors won't get reloaded if you change the rlecam_channel
	// later on

	// these names match up with /ms/config/default.col - the default
	// color information... if that changes, these should change too
	unsigned int threshChan=config->vision.rlecam_channel;

	// higher value reduce false events, but increase reaction time [0-inf]
	unsigned int noiseFiltering=1;

	// lower values increase sensitivity (and noise) [0-1]
	float confidenceThreshold=.8;

	unsigned int pinkIdx=segcol->getColorIndex("red");
	if(pinkIdx!=-1U) {
		pball = new BallDetectionGenerator(EventBase::visRegionEGID,visRegionSID,visPinkBallSID,pinkIdx,threshChan,noiseFiltering,confidenceThreshold);
		pball->setName("PinkBallDetectionGenerator");
		pball->DoStart();
		spawned.push_back(pball);
	}
	unsigned int blueIdx=segcol->getColorIndex("blue");
	if(blueIdx!=-1U) {
		bball = new BallDetectionGenerator(EventBase::visRegionEGID,visRegionSID,visBlueBallSID,blueIdx,threshChan,noiseFiltering,confidenceThreshold);
		pball->setName("BlueBallDetectionGenerator");
		bball->DoStart();
		spawned.push_back(bball);
	}
	unsigned int skinIdx=segcol->getColorIndex("brown");
	if(skinIdx!=-1U) {
		handball = new BallDetectionGenerator(EventBase::visRegionEGID,visRegionSID,visHandSID,skinIdx,threshChan,noiseFiltering,confidenceThreshold);
		pball->setName("HandBallDetectionGenerator");
		handball->DoStart();
		spawned.push_back(handball);
	}
}

ControlBase*
StartupBehavior::SetupVision() {
	addItem(new ControlBase("Vision Pipeline","Start/Stop stages of the vision pipeline"));
	startSubMenu();
	{ 
		addItem(new BehaviorSwitchControlBase(defRawCameraGenerator));
		addItem(new BehaviorSwitchControlBase(defInterleavedYUVGenerator));
		addItem(new BehaviorSwitchControlBase(defColorJPEGGenerator));
		addItem(new BehaviorSwitchControlBase(defGrayscaleJPEGGenerator));
		addItem(new BehaviorSwitchControlBase(defSegmentedColorGenerator));
		addItem(new BehaviorSwitchControlBase(defRLEGenerator));
		addItem(new BehaviorSwitchControlBase(defRegionGenerator));
		addItem(new BehaviorSwitchControlBase(pball));
		addItem(new BehaviorSwitchControlBase(bball));
		addItem(new BehaviorSwitchControlBase(handball));
	}
	return endSubMenu();
}
