//-*-c++-*-
#ifndef INCLUDED_StartupBehavior_h_
#define INCLUDED_StartupBehavior_h_
#include "Behaviors/BehaviorBase.h"
#include "Motion/MotionManager.h"
#include <vector>
#include <stack>

class Controller; // defined in Controller.h, don't actually need header file yet
class ControlBase; // defined in ControlBase.h, don't actually need header file yet

//! This Behavior is the only hardcoded behavior by the framework to start up once all the data structures and such are set up

/*! This is done by assigning an instance of StartupBehavior to the
 *  global ::startupBehavior variable.  Note that there's nothing
 *  special about the type of this class, it's just another
 *  BehaviorBase - ::startupBehavior could be any behavior.  If you
 *  want some other behavior to run automatically at startup, simply
 *  remove StartupBehavior from the project and assign an instance of
 *  your own class to ::startupBehavior instead.
 *
 *  By default, this behavior is similar in idea to the init process
 *  in unix/linux. */
class StartupBehavior : public BehaviorBase {
public:
	//! Constructor
	StartupBehavior();
	//! Destructor
	virtual ~StartupBehavior();

	//! @name Inherited
	virtual void DoStart();
	virtual void DoStop();
	virtual void processEvent(const EventBase&);
	virtual std::string getName() const { return "StartupBehavior"; }
	static std::string getClassDescription() { return "The initial behavior, when run, sets up everything else"; }
	//@}
protected:
	//! Initializes the Controller menu structure - calls each of the other Setup functions in turn
	/*! Also adds Shutdown and Help items at the end */
	virtual ControlBase* SetupMenus();

	virtual ControlBase* SetupModeSwitch();          //!< sets up Mode Switch menu
	virtual ControlBase* SetupBackgroundBehaviors(); //!< sets up Background Behaviors menu
	virtual ControlBase* SetupTekkotsuMon();         //!< sets up TekkotsuMon menu
	virtual ControlBase* SetupStatusReports();       //!< sets up Status Reports menu
	virtual ControlBase* SetupFileAccess();          //!< sets up File Access menu
	virtual ControlBase* SetupWalkEdit();            //!< sets up Walk Editing menus (contributed by PGSS 2003 team project - see StartupBehavior_SetupWalkEdit.cc )
	virtual ControlBase* SetupVision();              //!< sets up the Vision menu

	virtual void initVision(); //!< Sets up the vision pipelines (held in StartupBehavior_SetupVision.cc)

	virtual void startSubMenu(); //!< future calls to addItem() will be placed under the most recently added item
	virtual void addItem(ControlBase * control); //!< inserts a control at the end of the menu
	virtual ControlBase* endSubMenu();  //!< closes out a submenu so that future calls to addItem() will be added to the enclosing menu, returns the menu being closed out

	std::vector<BehaviorBase*> spawned; //!< Behaviors spawned from DoStart, so they can automatically be stopped on DoStop
	std::stack<ControlBase*> setup;     //!< only used during setup - easier than passing it around all the functions
	MotionManager::MC_ID stop_id; //!< the main EmergencyStopMC
	MotionManager::MC_ID pid_id; //!< used to fade in the PIDs up to full strength (from initial zero) This is so the joints don't jerk on startup.
};

extern BehaviorBase& startupBehavior;

#endif
