/* Implements a stateful behavior specifically designed to feed data to the
 * second WorldModel.
 *
 * Started 13 February 2003, tss
 */

#include "WorldModel2Behavior.h"

#include "Behaviors/StateNode.h"
#include "Behaviors/Transitions/TimeOutTrans.h"
#include "Motion/MotionManager.h"
#include "Motion/WalkMC.h"
#include "Motion/HeadPointerMC.h"
#include "Motion/MotionSequenceMC.h"
#include "Motion/OutputCmd.h"
#include "Shared/SharedObject.h"
#include "Events/EventRouter.h"
#include "Shared/WorldState.h"
#include "Shared/RobotInfo.h"
#include "Shared/get_time.h"

// for afsParticle
#include "WorldModel2/FastSLAM/afsParticle.h"
// for AFS_NUM_LANDMARKS
#include "WorldModel2/FastSLAM/Configuration.h"
// for stopgap observational poses
#include "WorldModel2/Poses.h"

#include <math.h>

#ifdef DEBUG_WM2B
#include <iostream>
using namespace std;
#endif

WorldModel2Behavior::WorldModel2Behavior()		// Constructor
  : StateNode("WorldModel2Behavior", NULL), start(NULL), WM2(),
    standUp(), standUp_id(MotionManager::invalid_MC_ID)
{
#ifdef WANT_GLOBAL_MAP
  // Enable the LazyFastSLAM kludge
  WM2.enableKludge(WM2Kludge::LazyFastSLAM);

  // Initialize WM2 with landmark locations
  WM2.setLandmark(0, 120, 800, 50);
  WM2.setLandmark(1, 470, 685, 50);
  WM2.setLandmark(2, 730, 410, 50);
  WM2.setLandmark(3, 800, 60, 50);
  WM2.setLandmark(4, 660, -285, 50);
  WM2.setLandmark(5, 375, -505, 50);
  WM2.setLandmark(6, 20, -550, 50);
  WM2.setLandmark(7, -320, -415, 50);
  WM2.setLandmark(8, -560, -140, 50);
  WM2.setLandmark(9, -630, 215, 50);
  WM2.setLandmark(10, -500, 550, 50);
  WM2.setLandmark(11, -195, 770, 50);

  // Spread FastSLAM particles around landmarks
  WM2.fsDistribute(-630, 800, 800, -550);
#endif

  // Set up motion sequence for standing erect
  standUp->setPlayTime(700);	// a procedure it must do in 700 milliseconds
  standUp->setOutputCmd(LFrLegOffset+RotatorOffset,  OutputCmd(SP_LFR_JOINT, 1));
  standUp->setOutputCmd(LFrLegOffset+ElevatorOffset, OutputCmd(SP_LFR_SHLDR, 1));
  standUp->setOutputCmd(LFrLegOffset+KneeOffset,     OutputCmd(SP_LFR_KNEE, 1));
  standUp->setOutputCmd(RFrLegOffset+RotatorOffset,  OutputCmd(SP_RFR_JOINT, 1));
  standUp->setOutputCmd(RFrLegOffset+ElevatorOffset, OutputCmd(SP_RFR_SHLDR, 1));
  standUp->setOutputCmd(RFrLegOffset+KneeOffset,     OutputCmd(SP_RFR_KNEE, 1));
  standUp->setOutputCmd(LBkLegOffset+RotatorOffset,  OutputCmd(SP_LBK_JOINT, 1));
  standUp->setOutputCmd(LBkLegOffset+ElevatorOffset, OutputCmd(SP_LBK_SHLDR, 1));
  standUp->setOutputCmd(LBkLegOffset+KneeOffset,     OutputCmd(SP_LBK_KNEE, 1));
  standUp->setOutputCmd(RBkLegOffset+RotatorOffset,  OutputCmd(SP_RBK_JOINT, 1));
  standUp->setOutputCmd(RBkLegOffset+ElevatorOffset, OutputCmd(SP_RBK_SHLDR, 1));
  standUp->setOutputCmd(RBkLegOffset+KneeOffset,     OutputCmd(SP_RBK_KNEE, 1));

  // Insert into motman
  standUp_id = motman->addMotion(standUp, false);
}

WorldModel2Behavior::~WorldModel2Behavior()		// Destructor
{
  // nothing yet
}

void WorldModel2Behavior::setup()			// Initial setup
{
  // Create state machine nodes
  StateNode *Halt = addNode(new WaitNode("WM2B/Wait", this, standUp_id));
  StateNode *Gawk = addNode(new GawkNode("WM2B/Gawk", this, &WM2, standUp_id));
  StateNode *Walk = addNode(new WalkNode("WM2B/Walk", this, standUp_id));

  // Indicate transitions among states--all of these are timer driven
  Halt->addTransition(new TimeOutTrans(Gawk, 1000));
  Gawk->addTransition(new TimeOutTrans(Walk, 5000));
  Walk->addTransition(new TimeOutTrans(Halt, 4000));

  // Denote start state
  start = Halt;

  // Superclass state machine setup
  StateNode::setup();
}

void WorldModel2Behavior::DoStart()			// Behavior startup
{
  StateNode::DoStart();	// superclass startup stuff
#ifdef WANT_GLOBAL_MAP
  WM2.enableMarkers();  // start marker detection
#endif
  start->DoStart();	// now start the state machine!
}

void WorldModel2Behavior::DoStop()			// Behavior stop
{
  // nothing yet except...
  StateNode::DoStop();	// superclass stopping stuff
#ifdef WANT_GLOBAL_MAP
  WM2.disableMarkers(); // stop marker detection
#endif
}




  // Private classes for specific behavior states
  // This one does a random walk
WorldModel2Behavior::WalkNode::WalkNode(const char *title, StateNode *poppa,
					MotionManager::MC_ID su)
  : StateNode(title, poppa),
    walker_id(MotionManager::invalid_MC_ID),
    head_id(MotionManager::invalid_MC_ID),
    stand_id(su), walkstate(AHEAD)
{
  // get head moving controls
  head_id = motman->addMotion(SharedObject<HeadPointerMC>());
  // get walking controls
  walker_id = motman->addMotion(SharedObject<WalkMC>());
}

WorldModel2Behavior::WalkNode::~WalkNode()		// Destructor
{
  // release walking controls
  motman->removeMotion(walker_id);
  // release head motion controls
  motman->removeMotion(head_id);
}

// IR obst. avoidance
void WorldModel2Behavior::WalkNode::processEvent(const EventBase&)
{
  WalkMC *walker;

  // If we're turning, see if we can stop turning
  if(walkstate == TURN) {
    if(state->sensors[IRDistOffset] < THR_AHEAD) return; // Still too close
    // clear water, all ahead full!
    walker = (WalkMC*)motman->checkoutMotion(walker_id);
    //walker->setTargetVelocity(0, 0, 0);
    walker->setTargetVelocity(180, 0, 0);
    motman->checkinMotion(walker_id);
    walkstate = AHEAD;
  }
  // If we're going forward, see if we have to turn.
  else {
    if(state->sensors[IRDistOffset] > THR_TURN) return; // no obstacles near
    // iceberg ahead! hard a'port!
    walker = (WalkMC*)motman->checkoutMotion(walker_id);
    //walker->setTargetVelocity(0, 0, 0);
    walker->setTargetVelocity(0, 0, 1);
    motman->checkinMotion(walker_id);
    walkstate = TURN;
  }
}

void WorldModel2Behavior::WalkNode::DoStart()		// Behavior startup
{
#ifdef DEBUG_WM2B
  cout << "Walk: start at " << get_time() / 1000 << endl;
#endif

  StateNode::DoStart();
  // point head forward
  HeadPointerMC *headptr = (HeadPointerMC*)motman->checkoutMotion(head_id);
  headptr->setJoints(0.0, 0.0, 0.0);
  motman->checkinMotion(head_id);
  // let's start moving!
  walkstate = AHEAD;
  WalkMC *walker = (WalkMC*)motman->checkoutMotion(walker_id);
  walker->setTargetVelocity(180, 0, 0);
  motman->checkinMotion(walker_id);
  // turn on IR sensors
  erouter->addListener(this, EventBase::sensorEGID, SensorSourceID::UpdatedSID);
}

void WorldModel2Behavior::WalkNode::DoStop()		// Behavior stop
{
#ifdef DEBUG_WM2B
  cout << "Walk: stop at " << get_time() / 1000 << endl;
#endif

  // turn off IR sensors
  erouter->removeListener(this, EventBase::sensorEGID,
				SensorSourceID::UpdatedSID);
  // stop moving
  // THIS CODE HAS NO EFFECT--motion is stopped before DoStop is called!!!
  WalkMC *walker = (WalkMC*)motman->checkoutMotion(walker_id);
  walker->setTargetVelocity(0, 0, 0);
  motman->checkinMotion(walker_id);
  StateNode::DoStop();
}




// This one wags the head around
WorldModel2Behavior::GawkNode::GawkNode(const char *title, StateNode *poppa,
					WorldModel2 *WM,
					MotionManager::MC_ID su)
  : StateNode(title, poppa),
    WM2ref(WM),
    head_id(MotionManager::invalid_MC_ID),
    stand_id(su)
{
  // nothing yet
}

// Head wag timer
void WorldModel2Behavior::GawkNode::processEvent(const EventBase&)
{
  // This routine should only be called by the timer event, so we won't
  // bother checking who rang us...
  // these two lines cause sinusoidal head oscillation
  double now = ((double) get_time()) / 1000.0;
  double pan = sin(now) * (80.0*M_PI/180.0);

  HeadPointerMC *headptr = (HeadPointerMC*)motman->checkoutMotion(head_id);
  headptr->setJoints(20*M_PI/180, pan, 0);
  motman->checkinMotion(head_id);
}

void WorldModel2Behavior::GawkNode::DoStart()		// Behavior startup
{
#ifdef DEBUG_WM2B
  cout << "Gawk: start" << endl;
#endif

  StateNode::DoStart();
  // stand up now
  // STANDING IS DISABLED FOR NOW
  //MotionSequenceMC<MotionSequence::SizeSmall> *stand = (MotionSequenceMC<MotionSequence::SizeSmall>*)motman->checkoutMotion(stand_id);
  //stand->play();
  //motman->checkinMotion(stand_id);
  // get head moving controls
  head_id = motman->addMotion(SharedObject<HeadPointerMC>());
  // turn on head wag timer.
  erouter->addTimer(this, 0, 5, true);
  // Head starts wagging in processEvent
}

void WorldModel2Behavior::GawkNode::DoStop()		// Behavior stop
{
#ifdef DEBUG_WM2B
  cout << "Gawk: stop" << endl;
#endif

  // turn off head timer
  erouter->removeTimer(this);
  // point head forward
  // THIS CODE HAS NO EFFECT--motion is stopped before DoStop is called!!!
  //HeadPointerMC *headptr = (HeadPointerMC*)motman->checkoutMotion(head_id);
  //headptr->setJoints(0.0, 0.0, 0.0);
  //motman->checkinMotion(head_id);
  // release head moving controls
  motman->removeMotion(head_id);
  StateNode::DoStop();

#ifdef DEBUG_WM2B
#ifdef WANT_GLOBAL_MAP
  // Generate MATLAB code onto the screen to see where we are.
  afsParticle p;
  WM2ref->getFastSLAMMap(p);

  cout << endl;
  cout << "plot(" << p.pose.x << ',' << p.pose.y << ",'g+')" << endl;

  for(int i=0; i<AFS_NUM_LANDMARKS; ++i) {
    if(p.landmarks[i].state == PRIMING) continue;

     cout << "plot(" << p.landmarks[i].mean.x << ','
		     << p.landmarks[i].mean.y << ",'b*')" << endl;

     cout << "plot_ellipse([" << p.landmarks[i].variance.x << ' '
			      << p.landmarks[i].variance.xy << ';'
			      << p.landmarks[i].variance.xy << ' '
			      << p.landmarks[i].variance.y << "],"
			      << p.landmarks[i].mean.x << ','
			      << p.landmarks[i].mean.y << ");" << endl;
  }
#endif
#endif
}




// This one lets the AIBO have a reflective pause
WorldModel2Behavior::WaitNode::WaitNode(const char *title, StateNode *poppa,
					MotionManager::MC_ID su)
  : StateNode(title, poppa),
    head_id(MotionManager::invalid_MC_ID),
    stand_id(su)
{
  // nothing yet
}

void WorldModel2Behavior::WaitNode::DoStart()		// Behavior startup
{
#ifdef DEBUG_WM2B
  cout << "Wait: start" << endl;
#endif

  StateNode::DoStart();
  // get head moving controls
  head_id = motman->addMotion(SharedObject<HeadPointerMC>());
  // stare dead ahead (har har)
  HeadPointerMC *headptr = (HeadPointerMC*)motman->checkoutMotion(head_id);
  headptr->setJoints(0, 0, 0);
  motman->checkinMotion(head_id);
}

void WorldModel2Behavior::WaitNode::DoStop()		// Behavior stop
{
#ifdef DEBUG_WM2B
  cout << "Wait: stop" << endl;
#endif

  // release head moving controls
  motman->removeMotion(head_id);
  StateNode::DoStop();
}
