/*
 * Motion resampling code for AIBO FastSLAM
 */

#ifdef __cplusplus
extern "C" {
#endif

#include "afsMotionResample.h"
#include "afsParticle.h"
#include "afsUtility.h"
#include "Configuration.h"

/* Needed for AIBO physical parameters */
#include "../Maps/Configuration.h"

#include <stdlib.h>
#include <math.h>

#ifdef UNIT_TEST
#define USE_OWN_MOTION_MODEL
#include <stdio.h>
#include <time.h>
#endif

#ifndef USE_OWN_MOTION_MODEL
#include "../Common/WalkMotionModel.h"
#endif

/* The resampling routine itself. See the header file for more details */
/* This routine has its own motion model included so that AIBO FastSLAM
 * can be used on a standalone basis. In the context of the WorldModel2
 * project, it's best to use the common motion model used by the entire
 * codebase. Define USE_OWN_MOTION_MODEL if you'd like to use the local
 * motion model--and beware of it falling out of date! */
void afsMotionResample(afsParticle *p,
			double dx, double dy, double da, unsigned int time)
{
#ifdef USE_OWN_MOTION_MODEL
  double x_motion, y_motion, theta_motion, arc_length;
  double xpos_before_angles, ypos_before_angles;
  double chord_length, total_motion_angle;
#endif

  /* If there was no motion at all, just don't do anything. */
  if((dx == 0) && (dy == 0) && (da == 0)) return;

  /* There was motion. Since the robot's motion is highly unpredictable,
   * we add Gaussian noise to the motion commands. This approach has
   * drawbacks. Most of the rotational error on the AIBO comes from the
   * robot's very first and very last motions, when balancing while
   * starting and stopping can tip the robot in strange ways. On long
   * treks, the perturbation added here can result in large deviations,
   * which don't quite match the real behavior of the robot. */
  dx += normRand() * AFS_PERTURB_DX_VARIANCE + AFS_PERTURB_DX_MEAN;
  dy += normRand() * AFS_PERTURB_DY_VARIANCE + AFS_PERTURB_DY_MEAN;
  da += normRand() * AFS_PERTURB_DA_VARIANCE + AFS_PERTURB_DA_MEAN;

  /* The motion model code used here predicts the location of the
   * AIBO's heading pivot point, NOT the location of the AIBO's chest,
   * which is where all the other FastSLAM code tracks the AIBO's
   * location. This makes the math easier for the motion model but means
   * also that we'll have to convert the AIBO pose to pivot centric
   * coordinates before invoking it. Later, at the end of this routine,
   * we'll convert back. */
  p->pose.x -= AIBO_TURN_PIVOT_DIST * cos(p->pose.theta);
  p->pose.y -= AIBO_TURN_PIVOT_DIST * sin(p->pose.theta);

#ifndef USE_OWN_MOTION_MODEL

  WalkMotionModel(&p->pose.x, &p->pose.y, &p->pose.theta, dx, dy, da, time);

#else /* not USE_OWN_MOTION_MODEL */
#warning "Using innate (and possibly obsolete) walking motion model!"

  /* Integrate dx, dy, and da */
  x_motion = dx * ((double) time)/1000.0;
  y_motion = dy * ((double) time)/1000.0;
  theta_motion = da * ((double) time)/1000.0;

  /* Compute linear distance */
  arc_length = sqrt(x_motion*x_motion + y_motion*y_motion);

  /* Compute relative motion along arc. This part pretends we start out going
   * straight forward, even though our dy might mean we're not. Don't worry,
   * it all gets sorted out. */
  if(fabs(theta_motion) < 0.000001) { /* avoids divide by zero */
    xpos_before_angles = arc_length;
    ypos_before_angles = 0;
  }
  else if(theta_motion < 0) {
    double hypotenuse = arc_length/theta_motion;
    xpos_before_angles = sin(theta_motion)*hypotenuse;
    ypos_before_angles = hypotenuse - cos(theta_motion)*hypotenuse;
  }
  else { /* theta_motion > 0 -- note negative sign */
    double hypotenuse = arc_length/theta_motion;
    xpos_before_angles = sin(theta_motion)*hypotenuse;
    ypos_before_angles = -(hypotenuse - cos(theta_motion)*hypotenuse);
  }

  /* Find the length of the chord between our current position and our
   * new position along the arc. */
  chord_length = sqrt(xpos_before_angles*xpos_before_angles +
		      ypos_before_angles*ypos_before_angles);

  /* At last we construct the final position of the robot */
  total_motion_angle =	atan2(y_motion, x_motion) +
			atan2(ypos_before_angles, xpos_before_angles) +
			p->pose.theta;

  /* We place the new position in the particle */
  p->pose.x += cos(total_motion_angle) * chord_length;
  p->pose.y += sin(total_motion_angle) * chord_length;
  p->pose.theta += theta_motion;

  /* Make sure that the angle is in [0, 2pi). */
  while(p->pose.theta >= 2*M_PI) p->pose.theta -= 2*M_PI;
  while(p->pose.theta < 0)       p->pose.theta += 2*M_PI;

#endif /* USE_OWN_MOTION_MODEL */

  /* Here we restore pose coordinates from the turn pivot point to the
   * neck point. */
  p->pose.x += AIBO_TURN_PIVOT_DIST * cos(p->pose.theta);
  p->pose.y += AIBO_TURN_PIVOT_DIST * sin(p->pose.theta);

  /* We are finished */
}

#ifdef UNIT_TEST

int main(int argc, char **argv)
{
  afsParticle p;
  int i;

  afsParticleInit(&p);
  srand(time(NULL));

  puts("hold off");
  puts("plot(0,0,'bx')");
  puts("hold on");
  puts("axis equal");
  for(i=0; i<10; ++i) {
    afsMotionResample(&p, 70, 0, 0, 1000);
    printf("plot (%f, %f, 'bx')\n", p.pose.x, p.pose.y);
  }
  for(i=0; i<10; ++i) {
    afsMotionResample(&p, -70, 0, 0, 1000);
    printf("plot (%f, %f, 'rx')\n", p.pose.x, p.pose.y);
  }
  for(i=0; i<10; ++i) {
    afsMotionResample(&p, 0, 30, 0, 1000);
    printf("plot (%f, %f, 'b*')\n", p.pose.x, p.pose.y);
  }
  for(i=0; i<10; ++i) {
    afsMotionResample(&p, 0, -30, 0, 1000);
    printf("plot (%f, %f, 'r*')\n", p.pose.x, p.pose.y);
  }
  for(i=0; i<20; ++i) {
    afsMotionResample(&p, 0, 30, 0.3, 1000);
    printf("plot (%f, %f, 'gx')\n", p.pose.x, p.pose.y);
  }

  return 0;
}

#endif

#ifdef __cplusplus
}
#endif
