/*
 * Finds the proper orientation of the landmark and position information
 * within a particle based on the known locations of landmarks.
 */

#ifdef __cplusplus
extern "C" {
#endif

#include "afsParticle.h"
#include "afsUtility.h"
#include "afsFindBestPose.h"

#include "Configuration.h"

#include <math.h>
#include <stdio.h>

/* This routine determines the best orientation of the particle p given the
 * supplied robot pose and landmark positions. The latter are supplied in an
 * array pointed to by the last argument (note: the landmark ID numbers will
 * be used to index the array. */
afsPose afsFindBestPose(afsParticle *p, afsPose pose, afsXY *real_landmarks)
{
  afsPose Best;
  /* Decode as (real|particle) center of mass (x|y) */
  double r_cm_x, r_cm_y, p_cm_x, p_cm_y;
  /* A tally of how many primed landmarks this particle has */
  int i, tally = 0;
  /* Vector components we use for angle averaging */
  double dtheta_sum_x, dtheta_sum_y;
  /* The averaged angle, which indicates how much we should rotate the
   * particle's map to align it to the real map. */
  double dtheta;

  /* initialize Best to current pose */
  Best = p->pose;

  /* First we find the centers of mass of both the real landmarks and the
   * ones known to the particle */
  r_cm_x = r_cm_y = p_cm_x = p_cm_y = 0;
  /* summing step */
  for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
    r_cm_x += real_landmarks[i].x;
    r_cm_y += real_landmarks[i].y;

    if(p->landmarks[i].state == PRIMING) continue;
    p_cm_x += p->landmarks[i].mean.x;
    p_cm_y += p->landmarks[i].mean.y;
    ++tally;
  }
  /* special case: if tally is 0, there's no way to best fit this particle.
   * just return the current one. */
  if(tally == 0) return Best;
  /* division step */
  r_cm_x /= (double) AFS_NUM_LANDMARKS;
  r_cm_y /= (double) AFS_NUM_LANDMARKS;
  p_cm_x /= (double) tally;
  p_cm_y /= (double) tally;

  /* Now we compute how much the particle's map should be rotated to match
   * the real map best */
  dtheta_sum_x = dtheta_sum_y = 0;
  for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
    /* x and y distances from the centers of mass to the landmark */
    /* use the same decoding scheme as above */
    double r_dist_x, r_dist_y, p_dist_x, p_dist_y;
    /* The angle difference between the particle and the map account of
     * the landmark location */
    double instance_dtheta;

    /* skip this landmark if unprimed */
    if(p->landmarks[i].state == PRIMING) continue;

    /* find distances */
    r_dist_x = real_landmarks[i].x - r_cm_x;
    r_dist_y = real_landmarks[i].y - r_cm_y;
    p_dist_x = p->landmarks[i].mean.x - p_cm_x;
    p_dist_y = p->landmarks[i].mean.y - p_cm_y;

    /* find angle distance */
    instance_dtheta =
      find_dtheta(atan2(p_dist_y, p_dist_x), atan2(r_dist_y, r_dist_x));
    /* angle averaging summing step */
    dtheta_sum_x += cos(instance_dtheta);
    dtheta_sum_y += sin(instance_dtheta);
  }
  /* angle averaging division step */
  dtheta_sum_x /= (double) tally;
  dtheta_sum_y /= (double) tally;
  /* the angle, at last */
  dtheta = atan2(dtheta_sum_y, dtheta_sum_x);

  /* We change the angle in the suggested pose first */
  Best.theta += dtheta;
  /* Now we rotate the position around the center of mass */
  {
    double p_dist_from_cm, p_theta_from_cm, pdfc_x, pdfc_y;

    pdfc_x = p->pose.x - p_cm_x;
    pdfc_y = p->pose.y - p_cm_y;
    p_dist_from_cm = sqrt(pdfc_x*pdfc_x + pdfc_y*pdfc_y);
    p_theta_from_cm = atan2(pdfc_y, pdfc_x);

    Best.x = r_cm_x + p_dist_from_cm*cos(p_theta_from_cm + dtheta);
    Best.y = r_cm_y + p_dist_from_cm*sin(p_theta_from_cm + dtheta);
  }

  return Best;
}

/* This routine takes the location and orientation in the particle p and
 * places the same information in new_p after applying the translation and
 * rotation recommended in best_pose (which you can get from
 * afsFindBestPose. You can make p and new_p point to the same thing
 * safely, but why would you want to? */
void afsApplyBestPose(afsParticle *p, afsParticle *new_p, afsPose best_pose)
{
  int i;
  afsParticle work_p = *p; /* copy! */
  double dtheta = find_dtheta(work_p.pose.theta, best_pose.theta);

  /* Reposition the landmark locations */
  for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
    double rel_dist, theta;

    /* shorthand */
    afsLandmarkLoc *l = &(work_p.landmarks[i]);

    /* Ignore unprimed landmarks */
    if(l->state == PRIMING) continue;

    /* Subtract off the robot x,y from the landmark positions */
    l->mean.x -= work_p.pose.x;
    l->mean.y -= work_p.pose.y;

    /* Rotate landmark positions */
    rel_dist = sqrt(l->mean.x*l->mean.x + l->mean.y*l->mean.y);
    theta = atan2(l->mean.y, l->mean.x) + dtheta;
    l->mean.x = rel_dist*cos(theta);
    l->mean.y = rel_dist*sin(theta);

    /* Add the new robot x,y */
    l->mean.x += best_pose.x;
    l->mean.y += best_pose.y;
  }

  /* Now just change the robot pose information */
  work_p.pose = best_pose;

  /* copy! */
  *new_p = work_p;
}

/* This routine finds the error value for a particle given real map
 * information. The function we use to calculate the error here is the
 * same one we minimized in afsFindBestPose, only there we were just
 * concerned with its derivative. */
/* NOTE: The more unprimed landmarks a particle has, the lower its error is
 * likely to be. This does not affect the performance of the algorithm */
double afsParticleError(afsParticle *p, afsPose pose, afsXY *real_landmarks)
{
  int i;
  double xerr, yerr;
  double error=0;

  for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
    /* shorthand */
    afsLandmarkLoc *l = &(p->landmarks[i]);

    /* Skip this landmark if it's not primed */
    if(l->state == PRIMING) continue;

    /* OK, find the error */
    xerr = real_landmarks[i].x - l->mean.x;
    yerr = real_landmarks[i].y - l->mean.y;
    error += xerr*xerr + yerr*yerr;
  }

  /* Add location error too. */
  xerr = pose.x - p->pose.x;
  yerr = pose.y - p->pose.y;
  error += xerr*xerr + yerr*yerr;

  return error;
}

/* This routine makes an estimated map of landmarks in the environment and
 * comes up with an estimated robot pose for situations where ground truth
 * is not known. It does all of this by averaging. The first argument is
 * a collection of all the particles in the filter. g in the arguments
 * stands for "guess". */
void afsGuessState(afsParticle *P, afsPose *g_pose, afsXY *g_landmarks)
{
  int i, p;
  double hdgvect_x, hdgvect_y;

  /* initialize location to 0, since we're averaging */
  g_pose->x = g_pose->y = 0;

  /* Find the average location and heading */
  /* Since heading is modular, we instead convert the headings into unit
   * vectors, average them, then take the arc tangent (or set it to 0 if
   * all of the vectors cancel). */
  hdgvect_x = hdgvect_y = 0;
  /* averaging summing step */
  for(p=0; p<AFS_NUM_PARTICLES; ++p) {
    g_pose->x += P[p].pose.x;
    g_pose->y += P[p].pose.y;
    hdgvect_x += cos(P[p].pose.theta);
    hdgvect_y += sin(P[p].pose.theta);
  }
  /* averaging division step */
  g_pose->x /= (double) AFS_NUM_PARTICLES;
  g_pose->y /= (double) AFS_NUM_PARTICLES;
  hdgvect_x /= (double) AFS_NUM_PARTICLES;
  hdgvect_y /= (double) AFS_NUM_PARTICLES;
  g_pose->theta = atan2(hdgvect_y, hdgvect_x); /* takes care of 0,0 itself */

  /* That was the easy part. Now we have to find the averaged locations of
   * all of the landmarks too. Whee! We transform them to relative radial
   * coordinates first, which makes things a lot easier. */
  for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
    double avg_rel_dist, avg_rel_theta;
    int tally=0;

    avg_rel_dist = hdgvect_x = hdgvect_y = 0;

    /* averaging summing step */
    for(p=0; p<AFS_NUM_PARTICLES; ++p) {
      /* shorthand */
      afsLandmarkLoc *l = &(P[p].landmarks[i]);
      double rel_theta, rel_dist_x, rel_dist_y;

      /* Skip this landmark if it's not primed */
      if(l->state == PRIMING) continue;

      /* Otherwise, up the tally and start summin'! */
      tally++;
      rel_dist_x = l->mean.x - P[p].pose.x;
      rel_dist_y = l->mean.y - P[p].pose.y;
      avg_rel_dist += sqrt(rel_dist_x*rel_dist_x + rel_dist_y*rel_dist_y);

      /* We do the same vector trick for the bearing that we did before.
       * First, though, we have to find the relative bearing of the
       * landmark. */
      rel_theta = find_dtheta(P[p].pose.theta, atan2(l->mean.y, l->mean.x));
      hdgvect_x += cos(rel_theta);
      hdgvect_y += sin(rel_theta);
    }

    /* If none of the particles had landmark i primed, then we can't very
     * well compute an average location. Oh well, set it to 0,0 */
    if(tally == 0) {
      g_landmarks[i].x = g_landmarks[i].y = 0;
      continue;
    }

    /* averaging division step */
    avg_rel_dist /= (double) tally;
    hdgvect_x /= (double) tally;
    hdgvect_y /= (double) tally;

    avg_rel_theta = atan2(hdgvect_x, hdgvect_y);

    /* the location, computed at last! */
    g_landmarks[i].x = avg_rel_dist*cos(avg_rel_theta + g_pose->theta);
    g_landmarks[i].y = avg_rel_dist*sin(avg_rel_theta + g_pose->theta);
  }
}

#ifdef __cplusplus
}
#endif
