/*
 * Walking motion model for the AIBO using walk routines from RoboSoccer.
 */

/* This extern c thingy, usually reserved for header files, is here too so
 * that it's easy to compile unit test code for the C++ programs. Yep. */
#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include <math.h>

/* The motion model routine itself. See the header file for more details */
void WalkMotionModel(double *x, double *y, double *theta,
		     double dx, double dy, double da, unsigned int time)
{
  double x_motion, y_motion, theta_motion, arc_length;
  double xpos_before_angles, ypos_before_angles;
  double chord_length, total_motion_angle;

  /* If there was no motion at all, just don't do anything. */
  if((dx == 0) && (dy == 0) && (da == 0)) return;

  /* Integrate dx, dy, and da */
  x_motion = dx * ((double) time)/1000.0;
  y_motion = dy * ((double) time)/1000.0;
  theta_motion = da * ((double) time)/1000.0;

  /* Compute linear distance */
  arc_length = sqrt(x_motion*x_motion + y_motion*y_motion);

  /* Compute relative motion along arc. This part pretends we start out going
   * straight forward, even though our dy might mean we're not. Don't worry,
   * it all gets sorted out. */
  if(fabs(theta_motion) < 0.000001) { /* avoids divide by zero */
    xpos_before_angles = arc_length;
    ypos_before_angles = 0;
  }
  else if(theta_motion < 0) {
    double hypotenuse = arc_length/theta_motion;
    xpos_before_angles = sin(theta_motion)*hypotenuse;
    ypos_before_angles = hypotenuse - cos(theta_motion)*hypotenuse;
  }
  else { /* theta_motion > 0 -- note negative sign */
    double hypotenuse = arc_length/theta_motion;
    xpos_before_angles = sin(theta_motion)*hypotenuse;
    ypos_before_angles = -(hypotenuse - cos(theta_motion)*hypotenuse);
  }

  /* Find the length of the chord between our current position and our
   * new position along the arc. */
  chord_length = sqrt(xpos_before_angles*xpos_before_angles +
		      ypos_before_angles*ypos_before_angles);

  /* At last we construct the final position of the robot */
  total_motion_angle =	atan2(y_motion, x_motion) +
			atan2(ypos_before_angles, xpos_before_angles) + *theta;

  /* We place the new position in the particle */
  *x += cos(total_motion_angle) * chord_length;
  *y += sin(total_motion_angle) * chord_length;
  *theta += theta_motion;

  /* Make sure that the angle is in [0, 2pi). */
  while(*theta >= 2*M_PI) *theta -= 2*M_PI;
  while(*theta < 0)       *theta += 2*M_PI;

  /* We are finished */
}

#ifdef __cplusplus
}
#endif
