//-*-c++-*-
#ifndef INCLUDED_SampleBehavior_h_
#define INCLUDED_SampleBehavior_h_

#include "Behaviors/BehaviorBase.h"
#include "SampleMC.h"
#include "Motion/LedMC.h"
#include "Motion/MotionManager.h"
#include "Motion/MMAccessor.h"
#include "Shared/SharedObject.h"
#include "Shared/WorldState.h"
#include "Events/EventRouter.h"

// Just a quick little demo behavior - when a button is pressed, this will
// light up the LED which happens to have the same index number as the
// button that was pressed

//! just a quick little demo behavior which lights up the LED with the same index number as the button which is pushed
class SampleBehavior : public BehaviorBase {
public:
	SampleBehavior()
		: BehaviorBase(), leds_id(MotionManager::invalid_MC_ID),
			mirror_id(MotionManager::invalid_MC_ID)
	{}
	
	virtual void DoStart() {
		//call superclass first for housekeeping:
		BehaviorBase::DoStart();

		//now do your code:
		leds_id=motman->addMotion(SharedObject<LedMC>());    // creates an LED controller
		mirror_id=motman->addMotion(SharedObject<SampleMC>()); // creates your sample MC
		erouter->addListener(this,EventBase::buttonEGID); // subscribes to all button events
	}
	
	virtual void DoStop() {
		//do your code:
		motman->removeMotion(leds_id);    // removes the LED controller
		motman->removeMotion(mirror_id);  // removes your sample MC
		erouter->forgetListener(this); // stops getting events (and timers, if we had any)

		//but don't forget to call superclass:
		BehaviorBase::DoStop();
	}
	
	virtual void processEvent(const EventBase& event) {
		// to be more general, let's check that it's the right event:
		if(event.getGeneratorID()==EventBase::buttonEGID) {

			// if it's an activate or status event, turn the LED on, otherwise turn it off
			float value = (event.getTypeID()!=EventBase::deactivateETID)?1:0;

			//if it's a paw button, select that paw as the source in the sample MC
			//normally i'd do this with one line since SIDs and Offsets happen
			//to be mapped together for convenience, but for example of 'correct' code:
			switch(event.getSourceID()) {
			case ButtonSourceID::LFrPawSID:
				{ MMAccessor<SampleMC>(mirror_id)->setSource(LFrLegOrder); }
				break;
			case ButtonSourceID::RFrPawSID:
				{ MMAccessor<SampleMC>(mirror_id)->setSource(RFrLegOrder); }
				break;
			case ButtonSourceID::LBkPawSID:
				{ MMAccessor<SampleMC>(mirror_id)->setSource(LBkLegOrder); }
				break;
			case ButtonSourceID::RBkPawSID:
				{ MMAccessor<SampleMC>(mirror_id)->setSource(RBkLegOrder); }
				break;
			}

			// make a bitmask corresponding to the button's source ID
			//   LEDBitMask_t is just an unsigned int, defined in RobotInfo.h
			LEDBitMask_t bitmask = (1<<event.getSourceID());

			// "checks out" the motion primitive from MotionManager
			MMAccessor<LedMC> leds_mc(leds_id);

			// now send the bitmask and value to the LED controller
			leds_mc->set(bitmask,value);

			// notice that there's no "check in"
			// MMAccessor's destructor does this automatically

		} else {
			//should never happen
			cout << "Bad Event:" << event.getName() << endl;
		}
	}
	
	virtual std::string getName() const {
		// Name is used for menus, or debugging.  We've been using the convention
		// that the first character should show if the behavior is currently active
		return "SampleBehavior";
	}

protected:
	MotionManager::MC_ID leds_id; //id of LED controller
	MotionManager::MC_ID mirror_id; //id of the SampleMC, which mirrors the legs
};

#endif
