//-*-c++-*-
#ifndef INCLUDED_SoundManager_h_
#define INCLUDED_SoundManager_h_

#include <OPENR/RCRegion.h>
#include <OPENR/ODataFormats.h>

#include <string>
#include <vector>
#include "Shared/ListMemBuf.h"
#include "Shared/MutexLock.h"
#include "SoundManagerMsg.h"
#include "Shared/ProcessID.h"

class OSubject;
class ONotifyEvent;

//! Provides sound effects and caching services, as well as mixing buffers for the SoundPlay process
/*! Provides easy methods for playing back sounds, either from files on the memory stick,
 *  or from dynamically generated buffers.  You can chain playback commands so that when
 *  one sound finishes, another picks up automatically.  This might be handy if, say, someone
 *  wants to write an MP3 player ;)  The sounds would be too large to load into memory all at
 *  once, but you could load a block at a time and chain them so it seamlessly moves from one
 *  to the other.
 *  
 *  All functions will attempt to lock the SoundManager.
 *
 *  @todo Volume control, variable playback speed, support more wav file formats (all go together)
 *  
 *  @todo Add functions to hand out regions to be filled out to avoid copying into the buffer.
 */
class SoundManager {
public:
	//!constructor
	SoundManager();

	//!This is used for referring to sound data so you can start playing it or release it
	typedef SoundManagerMsg::Snd_ID Snd_ID;
	static const Snd_ID invalid_Snd_ID=(Snd_ID)-1; //!< for reporting errors
	static const Snd_ID MAX_SND=50; //!< the number of sounds that can be loaded at any given time
	
	//!This is for referring to instances of the play command so you can stop, pause, or monitor progress (later versions will send events upon completion)
	typedef unsigned short Play_ID;
	static const Play_ID invalid_Play_ID=(Play_ID)-1; //!< for reporting errors
	static const Play_ID MAX_PLAY=256; //!< the number of sounds that can be enqueued at the same time (see MixMode_t)

	static const unsigned int MAX_NAME_LEN=64;   //!<maximum length of a path

	//!Used to set the mode for mixing multiple sound channels
	/*!Feel free to add a higher quality mixer if you're an audiophile - I'm pretty new to sound processing*/
	enum MixMode_t {
		Fast,    //!< uses bit shifting trick, but can result in reduced volume with more active channels, best if you set max_channels to a power of 2
		Quality  //!< uses real division to maintain volume level, although still a rather naive (but relatively fast) algorithm
	};

	enum QueueMode_t {
		Enqueue,        //!< newer sounds are played when a channel opens up (when old sound finishes)
		Pause,          //!< newer sounds pause oldest sound, which continues when a channel opens
		Stop,           //!< newer sounds stop oldest sound
		Override,       //!< older sounds have play heads advanced, but don't get mixed until a channel opens
	};

	//!Needed to send sounds to the SoundPlay process
	void InitAccess(OSubject* subj);

	//!loads a wav file (if it matches Config::sound_config settings - can't do resampling yet)
	/*!Since the SoundManager does the loading, if the same file is being played more than once, only once copy is stored in memory 
	 * @param name can be either a full path, or a partial path relative to Config::sound_config::root
	 * @return ID number for future references (can also use name)
	 * The sound data will be cached until ReleaseFile() or Release() is called a matching number of times*/
	Snd_ID LoadFile(const char* name);

	//!loads raw samples from a buffer (assumes matches Config::sound_config settings)
	/*!The sound data will be cached until Release() is called a matching number of times.\n
	 * This function is useful for dynamic sound sources.  A copy will be made. */
	Snd_ID LoadBuffer(const char buf[], unsigned int len);
	
	//!Marks the sound buffer to be released after the last play command completes (or right now if not being played)
	void ReleaseFile(const char* name);

	//!Marks the sound buffer to be released after the last play command completes (or right now if not being played)
	void Release(Snd_ID id);
	
	//!play a wav file (if it matches Config::sound_config settings - can't do resampling yet)
	/*!Will do a call to LoadFile() first, and then automatically release the sound again when complete.
	 * @param name can be either a full path, or a partial path relative to Config::sound_config::root
	 * @return ID number for future references
	 * The sound data will not be cached after done playing unless a call to LoadFile is made*/
	Play_ID PlayFile(const char* name);

	//!loads raw samples from a buffer (assumes buffer matches Config::sound_config settings)
	/*!The sound data will be released after done playing*/
	Play_ID PlayBuffer(const char buf[], unsigned int len);
	
	//!plays a previously loaded buffer or file
	Play_ID Play(Snd_ID id);
	
	//!allows automatic queuing of sounds - good for dynamic sound sources!
	/*!if you chain more than once to the same base, the new buffers are appended
	 * to the end of the chain - the new buffer doesn't replace the current chain
	 * @return @a base - just for convenience of multiple calls*/
	Play_ID ChainFile(Play_ID base, const char* next);

	//!allows automatic queuing of sounds - good for dynamic sound sources!
	/*!if you chain more than once to the same base, the new buffers are appended
	 * to the end of the chain - the new buffer doesn't replace the current chain
	 * @return @a base - just for convenience of multiple calls*/
	Play_ID ChainBuffer(Play_ID base, const char buf[], unsigned int len);

	//!allows automatic queuing of sounds - good for dynamic sound sources!
	/*!if you chain more than once to the same base, the new buffers are appended
	 * to the end of the chain - the new buffer doesn't replace the current chain
	 * @return @a base - just for convenience of multiple calls*/
	Play_ID Chain(Play_ID base, Snd_ID next);
	
	//!Lets you stop playback of all sounds
	void StopPlay();

	//!Lets you stop playback of a sound
	void StopPlay(Play_ID id);
	
	//!Lets you pause playback
	void PausePlay(Play_ID id);
	
	//!Lets you resume playback
	void ResumePlay(Play_ID id);
	
	//!Lets you control the maximum number of channels (currently playing sounds), method for mixing (applies when max_chan>1), and queuing method (for when overflow channels)
	void SetMode(unsigned int max_channels, MixMode_t mixer_mode, QueueMode_t queuing_mode);

	//!Gives the time until the sound finishes, in milliseconds.  Subtract 32 to get guarranteed valid time for this ID.
	/*!You should be passing the beginning of a chain to get proper results...\n
	 * May be slightly conservative (will report too small a time) because this
	 * does not account for delay until SoundPlay picks up the message that a
	 * sound has been added.\n
	 * However, it is slighly optimistic (will report too large a time) because
	 * it processes a buffer all at one go, so it could mark the sound as finished
	 * (and cause the ID to go invalid) up to RobotInfo::SoundBufferTime (32 ms)
	 * before the sound finishes.  So subtract SoundBufferTime if you want to be
	 * absolutely sure the ID will still valid. */
	unsigned int GetRemainTime(Play_ID id) const;
	
	//!Copies the sound data to the OPENR buffer, ready to be passed to the system, only called by SoundPlay
	/*!@return the number of active sounds */
	unsigned int CopyTo(OSoundVectorData* data);

	//!updates internal data structures on the SoundPlay side - you shouldn't be calling this
	void ReceivedMsg(const ONotifyEvent& event);
	
	//! returns number of sounds currently playing
	unsigned int GetNumPlaying() { return chanlist.size(); }
	
protected:
	//!Sets up a shared region to hold a sound - rounds to nearest page size
	static RCRegion* initRegion(unsigned int size);

	//!Looks to see if @a name matches any of the sounds in sndlist
	Snd_ID lookup(const char* name) const;
	//!Looks to see if @a name matches any of the sounds in sndlist (assumes is absolute path)
	Snd_ID lookupPath(const char* path) const;

	//!prepends config.sound.root to the name if necessary
	static const char* makePath(const char* name, char tmp[MAX_NAME_LEN]);

	//!selects which of the channels are actually to be mixed together, depending on queue_mode
	void selectChannels(std::vector<Play_ID>& mix);

	//!update the offsets of sounds which weren't mixed (when needed depending on queue_mode)
	void updateChannels(const std::vector<Play_ID>& mixs,size_t used);

	//!called when a buffer end is reached, may reset buffer to next in chain, or just StopPlay()
	bool endPlay(Play_ID id);

	//!Holds data about the loaded sounds
	struct SoundData {
		SoundData();                             //!<constructor
		RCRegion * rcr;                          //!<shared region - don't need to share among processes, just collect in SoundPlay
		byte* data;                              //!<point to data in region (for convenience, only valid in SoundPlay)
		unsigned int len;                        //!<size of the sound
		unsigned int ref;                        //!<reference counter
		char name[SoundManager::MAX_NAME_LEN];   //!<stores the path to the file, empty if from a buffer
	private:
		SoundData(const SoundData&);             //!< don't call
		SoundData operator=(const SoundData&);   //!< don't call
	};
	//!For convenience
	typedef ListMemBuf<SoundData,MAX_SND,Snd_ID> sndlist_t;
	//!Holds a list of all currently loaded sounds
	sndlist_t sndlist;
	
	//!Holds data about sounds currently being played
	struct PlayState {
		PlayState();            //!<constructor
		Snd_ID snd_id;          //!<index of sound
		unsigned int offset;    //!<position in the sound
		unsigned int cumulative;//!<total time of playing (over queued sounds)
		Play_ID next_id;        //!<lets you queue for continuous sound, or loop
	};
	//!For convenience
	typedef ListMemBuf<PlayState,MAX_PLAY,Play_ID> playlist_t;
	//!Holds a list of all sounds currently enqueued
	playlist_t playlist;
	//!For convenience
	typedef ListMemBuf<Play_ID,MAX_PLAY,Play_ID> chanlist_t;
	//!Holds a list of all currently playing sounds, ordered newest (front) to oldest(back)
	chanlist_t chanlist;
	
	//!Current mixing mode, set by SetMode();
	MixMode_t mix_mode;

	//!Current queuing mode, set by SetMode();
	QueueMode_t queue_mode;

	//!Current maximum number of sounds to mix together
	unsigned int max_chan;

	//!Prevents multiple processes from accessing at the same time
	mutable MutexLock<ProcessID::NumProcesses> lock;
	
	//!For automatic transmission of shared regions to SoundPlay
	OSubject * subjs[ProcessID::NumProcesses];

	SoundManager(const SoundManager&);           //!< don't call
	SoundManager operator=(const SoundManager&); //!< don't call
};

//! lets you play a sound from anywhere in your code - just a one liner!
extern SoundManager * sndman;

/*! @file
 * @brief Describes SoundManager, which provides sound effects and caching services, as well as mixing buffers for the SoundPlay process
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0_1 $
 * $Revision: 1.10 $
 * $State: Exp $
 * $Date: 2004/01/18 10:16:58 $
 */

#endif
