//-*-c++-*-
#ifndef INCLUDED_PostureEngine_h
#define INCLUDED_PostureEngine_h

#include "OutputCmd.h"
#include "Shared/RobotInfo.h"
#include "Shared/LoadSave.h"

//! A class for storing a set of positions and weights for all the outputs
/*! Handy for any class which wants to deal with setting joints and postures without writing a custom class
 *  @see PostureMC */
class PostureEngine : public LoadSave {
public:
	//!constructor
	PostureEngine() : LoadSave() {}
	//!constructor, loads a position from a file - not necessarily quick!
	/*! @todo might want to make a library of common positions so they don't have to be loaded repeatedly from memstick */
	PostureEngine(const char * filename) : LoadSave() { LoadFile(filename); }
	//! destructor
	virtual ~PostureEngine();

	//! sets the internal #cmds to the current state of the outputs
	virtual void takeSnapshot();

	//! sets all joints to unused
	virtual void clear();

	//! sets joints of this to all joints of @a pe which are not equal to unused (layers @a pe over this) stores into this
	virtual PostureEngine& setOverlay(const PostureEngine& pe);
	//! sets joints of this to all joints of @a pe which are not equal to unused (layers @a pe over this) returns new PostureEngine
	virtual PostureEngine createOverlay(const PostureEngine& pe) const;

	//! sets joints of this which are equal to unused to @a pe, (layers this over @a pe) stores into this
	virtual PostureEngine& setUnderlay(const PostureEngine& pe);
	//! sets joints of this which are equal to unused to @a pe, (layers this over @a pe) returns new PostureEngine
	virtual PostureEngine createUnderlay(const PostureEngine& pe) const;

	//! computes a weighted average of this vs. @a pe, @a w being the weight towards @a pe (so @a w==1 just copies @a pe)
	virtual PostureEngine& setAverage(const PostureEngine& pe,float w=0.5);
	//! computes a weighted average of this vs. @a pe, @a w being the weight towards @a pe (so @a w==1 just copies @a pe)
	virtual PostureEngine createAverage(const PostureEngine& pe,float w=0.5) const;

	//! computes a weighted average of this vs. @a pe, using the weight values of the joints, storing the total weight in the result's weight value
	virtual PostureEngine& setCombine(const PostureEngine& pe);
	//! computes a weighted average of this vs. @a pe, using the weight values of the joints, storing the total weight in the result's weight value
	virtual PostureEngine createCombine(const PostureEngine& pe) const;

	//! returns the sum squared error between this and pe's output values, but only between outputs which are both not unused
	/*! @todo create a version which looks at weights?  This doesn't use them. */
	float diff(const PostureEngine& pe) const;
	
	//! returns the average sum squared error between this and pe's output values for outputs which are both not unused
	/*! @todo create a version which looks at weights?  This doesn't use them. */
	float avgdiff(const PostureEngine& pe) const;
	
	//! returns the max sum squared error between this and pe's output values for outputs which are both not unused
	/*! @todo create a version which looks at weights?  This doesn't use them. */
	float maxdiff(const PostureEngine& pe) const;
	
	//! NOT VIRTUAL! You should be able to call this to set outputs without checking out, just a peekMotion().  Theoretically.
	//!@name Output Accessors
	inline PostureEngine& setOutputCmd(unsigned int i, const OutputCmd& c) { cmds[i]=c; return *this; } //!<sets output @a i to OutputCmd @a c, returns @c *this so you can chain them
	inline OutputCmd& getOutputCmd(unsigned int i) { return cmds[i]; } //!< returns output @a i, returns a reference so you can also set "through" this call.
	inline const OutputCmd& getOutputCmd(unsigned int i) const { return cmds[i]; } //!< returns output @a i
	//@}

	//!Uses LoadSave interface so you can load/save to files, uses a human-readable storage format
	//!@name LoadSave
	virtual unsigned int getBinSize() const;
	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);
	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;
	//@}

protected:
	//!used by LoadBuffer()/SaveBuffer(), checks to see if the amount read/written (@a res) is nonzero, increments @a buf, decrements @a len, or displays @a msg if @a is zero
	static bool ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg);
	//!used by LoadBuffer()/SaveBuffer(), checks to see if the amount read/written (@a res) is nonzero, increments @a buf, decrements @a len, or displays @a msg with @a arg1 if @a is zero
	static bool ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg, int arg1);
	
	//!the table of outputs' values and weights, can be accessed through setOutputCmd() and getOutputCmd()
	OutputCmd cmds[NumOutputs];
};

/*! @file
 * @brief Describes PostureEngine, a base class for managing the values and weights of all the outputs
 * @todo write a binary version of Load/Save commands for faster access
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0_1 $
 * $Revision: 1.3 $
 * $State: Rel $
 * $Date: 2003/09/25 15:27:23 $
 */

#endif
