//-*-c++-*-
#ifndef INCLUDED_EventGeneratorBase_h_
#define INCLUDED_EventGeneratorBase_h_

#include "Behaviors/BehaviorBase.h"

//! A simple convenience class for event generators
/*! Note that you don't need to inherit from this class to be able to
 *  send events!  Any code can send any event any time, just by calling 
 *  one of the EventRouter::postEvent() functions.
 *  
 *  Uses a BehaviorBase base class so that you can start and stop it.
 *
 *  Allows variable settings of the generator id and source id for
 *  outgoing events as well as automatic handling of
 *  listening/unlistening for a single optional event source.  If you
 *  want something more fancy that that though, you'll have to
 *  override DoStart/DoStop yourself. (or extend/modify this class...)
 *
 */
class EventGeneratorBase : public BehaviorBase {
public:
	//!@name Constructors
	//!
	EventGeneratorBase()
		: BehaviorBase(), myGenID(EventBase::unknownEGID), mySourceID(0), autoListen(false), srcGenID(EventBase::unknownEGID), srcSourceID(0), myName()
	{}
	EventGeneratorBase(const std::string& name, EventBase::EventGeneratorID_t mgid, unsigned int msid)
		: BehaviorBase(), myGenID(mgid), mySourceID(msid), autoListen(false), srcGenID(EventBase::unknownEGID), srcSourceID(0), myName(name)
	{}
	EventGeneratorBase(const std::string& name, EventBase::EventGeneratorID_t mgid, unsigned int msid,EventBase::EventGeneratorID_t srcgid, unsigned int srcsid)
		: BehaviorBase(), myGenID(mgid), mySourceID(msid), autoListen(true), srcGenID(srcgid), srcSourceID(srcsid), myName(name)
	{}
	//@}
	
	virtual void DoStart();
	
	virtual void DoStop();
	
	//! return the generator ID that will be broadcast from
	virtual EventBase::EventGeneratorID_t getGeneratorID() { return myGenID; }
	//! set the generator ID that will be broadcast from (typically it's a bad idea to call this...)
	virtual void setGeneratorID(EventBase::EventGeneratorID_t gid) { myGenID=gid; }

	//! return the source ID that will be broadcast on
	virtual unsigned int getSourceID() { return mySourceID; }
	//! set the source ID that will be broadcast on
	virtual void setSourceID(unsigned int sid) { mySourceID=sid; }

	//! returns the source ID that will be listened for (not the source of the FilterBankEvent to be created - that depends on the subclass)
	virtual unsigned int getListenSourceID() const { return srcSourceID; }
	//! returns the generator ID that will be listened for (not the generator of the FilterBankEvent to be created - that depends on the subclass)
	virtual EventBase::EventGeneratorID_t getListenGeneratorID() const { return srcGenID; }
	//! turns on auto listening to make it easier to set up dependancies between vision filters
	virtual void setAutoListen(EventBase::EventGeneratorID_t gid, unsigned int sid) { autoListen=true; srcGenID=gid; srcSourceID=sid; }

	//! turns off auto listening
	virtual void unsetAutoListen() { autoListen=false; }

	//! returns current name
	virtual std::string getName() const {
		return myName;
	}

	//! sets a name (overriding automatically generated one)
	virtual void setName(const std::string& name) {
		myName=name;
	}

protected:
	EventBase::EventGeneratorID_t myGenID; //!< the generator ID to broadcast on
	unsigned int mySourceID;     //!< the source ID to broadcast on
	bool autoListen;          //!< if true, will automatically start listening for EventBase(genID,sourceID) events
	EventBase::EventGeneratorID_t srcGenID; //!< the generator ID to listen for (typically the source that this filter works on)
	unsigned int srcSourceID;    //!< the source ID to listen for
	std::string myName;  //!< the name to report (handy for debugging output)
};

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0_1 $
 * $Revision: 1.4 $
 * $State: Exp $
 * $Date: 2004/01/18 10:16:57 $
 */

#endif
