//-*-c++-*-
#ifndef INCLUDED_SegCamBehavior_h_
#define INCLUDED_SegCamBehavior_h_

#include "Behaviors/BehaviorBase.h"
#include "Shared/Config.h"

class Socket;
class FilterBankGenerator;
class FilterBankEvent;

//! Forwards segmented images from camera over wireless
/*! The format used for serialization is basically defined by the
 *  subclass of FilterBankGenerator being used.  I suggest looking at
 *  that classes's documentation to determine the format used.
 *
 *  However, SegCamBehavior will add a few fields at the beginning of
 *  each packet to assist in processing the image stream.
 *
 *  I emphasize: <i>beginning</i> of each Vision packet, <i>before</i> the FilterBankGenerator header. 
 *  - <@c string:"TekkotsuImage">
 *  - <<tt>Config::vision_config::encoding_t</tt>: Config::vision_config::ENCODE_SINGLE_CHANNEL> <i>(always just sends a single channel)</i>
 *  - <<tt>Config::vision_config::compression_t</tt>: Config::vision_config::COMPRESS_RLE> <i>(This is misleading - may actually be uncompressed, but this signals it's a segmented color image)</i>
 *  - <@c unsigned @c int: width> <i>(this is the width of the largest channel - note different channels can be sent at different resolutions!  Provides cheap "compression" of chromaticity channels)</i>
 *  - <@c unsigned @c int: height> <i>(similarly, height of largest channel)</i>
 *  - <@c unsigned @c int: timestamp> <i>(time image was taken, milliseconds since boot)</i>
 *  - <@c unsigned @c int: framenumber> <i>(incremented for each frame, so we can tell if/when we drop one)</i>
 *
 *  This is exactly the same protocol that is followed by the
 *  RawCamBehavior as well - the same code can parse either stream.
 *
 *  However, odd bit - since the RLEGenerator doesn't save the color
 *  information itself, SegCamBehavior will do it instead.  So, if
 *  SegCamBehavior is using RLE compression, it will tack a footer at
 *  the end of the packet: (from SegmentedColorGenerator::encodeColors())
 *  - <@c unsigned @c int: num_cols> <i>(number of different colors available)</i>
 *  - for each of num_col:
 *    - <@c char: red> <i>red color to use for display of this index</i>
 *    - <@c char: green> <i>green color to use for display of this index</i>
 *    - <@c char: blue> <i>blue color to use for display of this index</i>
 */ 
class SegCamBehavior : public BehaviorBase {
public:
	//! constructor
	SegCamBehavior();

	static const unsigned int WIRELESS_BUFFER_SIZE=85000; //!< 85000 bytes for use up to 416x320 pixels / 8 min expected runs * 5 bytes per run + some padding

	virtual void DoStart();

	virtual void DoStop();

	virtual void processEvent(const EventBase& e);

	virtual std::string getName() const { return "SegCamServer"; }

	static std::string getClassDescription() {
		char tmp[20];
		sprintf(tmp,"%d",config->vision.rle_port);
		return std::string("Forwards segmented images from camera over port ")+tmp;
	}
	
protected:
	//! opens a new packet, writes header info; returns true if open, false if otherwise open (check cur==NULL for error)
	/*! see the class documentation for SegCamBehavior for the protocol documentation */
	bool openPacket(FilterBankGenerator& fbkgen, unsigned int time, unsigned int layer); 
	bool writeRLE(const FilterBankEvent& fbke); //!< writes a color image
	bool writeSeg(const FilterBankEvent& fbke); //!< writes a color image
	void closePacket(); //!< closes and sends a packet, does nothing if no packet open

	Socket * visRLE; //!< socket to send image stream over
	char * packet; //!< buffer being filled out to be sent
	char * cur; //!< current location in #packet
	unsigned int avail; //!< number of bytes remaining in #packet

private:
	SegCamBehavior(const SegCamBehavior&); //!< don't call
	SegCamBehavior& operator=(const SegCamBehavior&); //!< don't call
};

/*! @file
 * @brief Describes SegCamBehavior, which forwards segmented images from camera over wireless
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-2_0_1 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2004/02/05 23:33:41 $
 */

#endif
