#ifndef Socket_h_DEFINED
#define Socket_h_DEFINED

#include <ant.h>
#include <Types.h>
#include <stdarg.h>

namespace SocketNS {
  //! Specifies transport type. TCP is usually a good idea
  enum TransportType_t {
    SOCK_STREAM=0, //!< TCP: guaranteed delivery, higher overhead
    SOCK_DGRAM     //!< UDP: no guarantees, low overhead
  };

  //! Internal TCP/UDP Connection State
  enum ConnectionState {
    CONNECTION_CLOSED,
    CONNECTION_CONNECTING,
    CONNECTION_CONNECTED,
    CONNECTION_LISTENING,
    CONNECTION_CLOSING,
    CONNECTION_ERROR
  };

  //! Chooses between blocking and non-blocking Wireless Input, Output. Blocking wireless output from the main process will affect the performance of the Aibo, and should only be used for debugging purposes
  enum FlushType_t {
    FLUSH_NONBLOCKING=0, //!< Writes and Reads return immediately, and are processed by another process, so Main can continue to run. Non-blocking reads require specifying a callback function to handle data received
    FLUSH_BLOCKING       //!< Blocking writes are a good idea for debugging - a blocking write will be transmitted before execution continues to the next statement. Blocking reads should be avoided, since they'll cause a significant slow down in the main process
  };

};

using namespace SocketNS;

class Wireless;

//! Tekkotsu wireless Socket class
/*! 
 * For more information on using wireless, please read the following tutorials:
 * - <a href="../AiboMon.html">TekkotsuMon</a>
 * - <a href="../Wireless.html">TCP/IP</a>
 * - <a href="../RemoteProcess.html">Remote Processing OPENR</a>
 * Tekkotsu Wireless and Remote Processing OPENR provide different
 * interfaces to comparable wireless functionality.
 */

class Socket {
  friend class Wireless;

public:
  int sock; //!< unique non-negative integer representing socket. Serves as index into socket Objects array

public:
  //! constructor
  Socket(int sockn)
		: sock(sockn), trType(), flType(FLUSH_NONBLOCKING), verbosity(0), 
			endpoint(), state(CONNECTION_CLOSED), sendBufSize(), recvBufSize(),
			sendSize(0), recvSize(0), writeSize(0), readSize(0),
			tx(false), rx(false), sendBuffer(), recvBuffer(), sendData(NULL),
			recvData(NULL), readData(NULL), writeData(NULL), server_port(0),
      rcvcbckfn(NULL), textForward(false), textForwardBuf(NULL),
      forwardSock(NULL), daemon(false)
	{ }
  virtual ~Socket() {} //!< destructor

  //! use getWriteBuffer to get a memory address to write bytes to, for
  //! subsequent writing to a connection.
  /*!
   * The getWriteBuffer-write(int) combo eliminates one buffer copy. You
   * don't need to use getWriteBuffer with write(byte*, int)
   * @return pointer to the current position in the current write buffer for this socket or NULL on error
   * @param bytesreq maximum number of bytes the caller intends to set before the write method is called */
  byte* getWriteBuffer(int bytesreq);

  //! writes the specified number of bytes starting at the pointer returned.
  /*!
   * in a (prior) call to getWriteBufer
   * @param size number of bytes to be sent from the current write buffer
   */
  void write(int size);

  //! Blocking read.
  /*!
   * Tries to read upto receive buffer size worth of data from this socket.
   *
   * Blocking read is currently broken - it will be fixed in the next release
   * @return number of bytes read or -1 on error
   */
  int read();

  //! getReadBuffer is used with blocking read's
  /*!
   * The read(void) and getReadBuffer combo eliminates one buffer copy. You
   * don't need to use getReadBuffer with read(byte*, int)
   *
   * Blocking read is currently broken - it will be fixed in the next release
   * @return pointer to the buffer the previous call to blocking read wrote into or NULL if no data was read
   */
  byte* getReadBuffer();

  //! initialize socket member variables. This is different from the constructor since sockets are reused
  void init(); 

  //! Chooses between blocking and non-blocking input, output.
  /*! This function
   * can only be called when a socket is disconnected, since it is a bad
   * idea to mix blocking and non-blocking input, output.
   * The default for a socket is non-blocking
   * @return 0 on success
   */
  int setFlushType(FlushType_t fType);

	//!causes this socket to forward output to stdout if it is not connected, call setForward(NULL) to unset
  void setTextForward() { textForward=true; forwardSock=NULL; }

	//!causes this socket to forward output to @a sock if it is not connected, pass NULL to unset
  void setForward(Socket * forsock) { forwardSock=forsock; textForward=false; }

  //! Picks a level of verbosity for filtering pprintf commands.
  /*! The higher the
   * verbosity, the more the number of messages printed. This is useful
   * for filtering out non-important messages with very little processor
   * cost. Default is 0.
   * @param verbose the higher the value of verbose, the more the output
   */
  void setVerbosity(int verbose) { verbosity=verbose; }

  //! Standard write - writes specified amount of data from a buffer to a
  //! connection
  /*! You might want to consider the getWriteBuffer-write(int) combo if you
   * call this often
   * @param buf buffer to write from
   * @param size number of bytes to write
   * @return the number of bytes actually written or -1 on error
   */
  int write(const byte *buf, int size);

  //! Blocking read.
  /*! You might want to consider the read(void) and getReadBuffer combo if you
   * call this often
   *
   * Blocking read is currently broken - it will be fixed in the next release
   * @param buf buffer to write from
   * @param size number of bytes to write
   * @return number of bytes actually read
   */
  
  int read(byte *buf, int size);

  //! It's standard stuff. man 3 printf on most systems should give you more
  //! information
  int printf(const char *fmt, ...);

  //! It's standard stuff. man 3 printf on most systems should give you more
  //! information
  int vprintf(const char *fmt, va_list al);

  //! Similar to printf, except it takes an extra first argument.
  /*! If vlevel is
   * than or equal to the current verbosity level, the string will be printed
   * else it will be ignored
   * @param vlevel if (vlevel<=verbosity) print, else ignore
	 * @param fmt same as the standard printf's format string
   */
  int pprintf(int vlevel, const char *fmt, ...);
  
  //! Initiate blocking or nonblocking write transfer depending on the type
  //! of socket.
  /*! All write commands on the socket will implicity call this. You
   * don't need to call it, unless you're implementing your own write
   */
  void flush();

protected:
	//@{
	//!private ALOKL_TODO
  TransportType_t trType;
  FlushType_t flType;

  int verbosity;

  antModuleRef endpoint;
  ConnectionState state;

  int sendBufSize, recvBufSize, sendSize, recvSize, writeSize, readSize;
  bool tx, rx;

  antSharedBuffer sendBuffer, recvBuffer;
  byte *sendData, *recvData;
  byte *readData, *writeData;
  int server_port;
  int (*rcvcbckfn) (char*, int);

  bool textForward;
	char* textForwardBuf;
	Socket * forwardSock;

  bool daemon;
	//@}
 
protected:
  Socket(const Socket&); //!< copy constructor, don't call
  Socket& operator= (const Socket&); //!< assignment operator, don't call
};

extern Socket* sout;  //!< the standard tekkotsu in/out console (default port 10001)
extern Socket* serr;  //!< the standard tekkotsu error output (default port 10002)

/*! @file
 * @brief Defines Tekkotsu wireless Socket class, also sout and serr
 * @author alokl (Creator)
 * 
 * $Author: ejt $
 * $Name: tekkotsu-1_5 $
 * $Revision: 1.15 $
 * $State: Rel $
 * $Date: 2003/09/25 15:32:14 $
 */

#endif
