#ifdef PLATFORM_APERIOS
#include <OPENR/core_macro.h>
#include <OPENR/ObjcommTypes.h>
#include <OPENR/OPENR.h>
#include <OPENR/OPENRAPI.h>
#include <OPENR/OPENRMessages.h>
#include <OPENR/OPower.h>
#endif

#include "WorldState.h"
#include "Shared/get_time.h"
#include "Events/EventRouter.h"
#include "ERS210Info.h"
#include "ERS220Info.h"

#define GETD(cpc) (((float)sensor.GetData(cpc)->frame[0].value) / 1.0E6f) //!<@return value from OPEN-R, converted from micro in int to base in float
#define GETB(cpc) ((bool)sensor.GetData(cpc)->frame[0].value) //!<@return value from OPEN-R, as bool
#define GETSIG(cpc) ((word)sensor.GetData(cpc)->frame[0].signal) //!<@return signal from OPEN-R as word
#define GETDUTY(cpc) ((float)((OJointValue*)&sensor.GetData(cpc)->frame[0])->pwmDuty/512.0f) //!<@return duty cycle from OPEN-R as float; -1 (full reverse) to 0 (idle) to 1 (full forward)

const double WorldState::g=9.80665;
const double WorldState::IROORDist = 900.0;

WorldState * state=NULL;

WorldState::WorldState()
	: alwaysGenerateStatus(false), robotStatus(0), batteryStatus(0),
		lastSensorUpdateTime(0), mainProfile(), motionProfile(),
#ifdef PLATFORM_APERIOS
		wsser(NULL), 
#endif
		robotDesign(0), curtime(get_time())
{
	for(unsigned int i=0; i< NumOutputs; i++)
		outputs[i]=0;
	for(unsigned int i=0; i< NumButtons; i++)
		buttons[i]=0;
	for(unsigned int i=0; i< NumSensors; i++)
		sensors[i]=0;
	for(unsigned int i=0; i< NumPIDJoints; i++)
		for(unsigned int j=0; j<3; j++)
			pids[i][j]=0;
	for(unsigned int i=0; i< NumPIDJoints; i++)
		pidduties[i]=0;
	memset(powerFlags,0,sizeof(unsigned int)*PowerSourceID::NumPowerSIDs);
	memset(button_times,0,sizeof(unsigned int)*NumButtons);
#ifdef PLATFORM_APERIOS
	wsser=new WorldStateSerializer();
#endif
	//Thanks Daishi:
	char robotDesignStr[orobotdesignNAME_MAX];
	memset(robotDesignStr, 0, sizeof(robotDesignStr));
	if (OPENR::GetRobotDesign(robotDesignStr) != oSUCCESS) {
		cout << "OPENR::GetRobotDesign() failed." << endl;
	} else {
		if(strcmp(robotDesignStr,"ERS-210")==0)
			robotDesign=ERS210Mask;
		else if(strcmp(robotDesignStr,"ERS-220")==0)
			robotDesign=ERS220Mask;
		else {
			cout << "ERROR: Unrecognized model: "<<robotDesignStr<<"\nTrying ERS-210..."<<endl;
			robotDesign=ERS210Mask;
		}
	}
}

#ifdef PLATFORM_APERIOS

/*! This will cause events to be posted
 *  @todo change to use most recent instead of oldest - is a buffer! */
void WorldState::read(OSensorFrameVectorData& sensor, EventRouter* er) {
	curtime=get_time();

	if(robotDesign&ERS210Mask) {
		outputs[LFrLegOffset + RotatorOffset   ] = GETD(ERS210Info::CPCJointLFRotator);
		outputs[LFrLegOffset + ElevatorOffset  ] = GETD(ERS210Info::CPCJointLFElevator);
		outputs[LFrLegOffset + KneeOffset      ] = GETD(ERS210Info::CPCJointLFKnee);
		pidduties[LFrLegOffset + RotatorOffset ] = GETDUTY(ERS210Info::CPCJointLFRotator);
		pidduties[LFrLegOffset + ElevatorOffset] = GETDUTY(ERS210Info::CPCJointLFElevator);
		pidduties[LFrLegOffset + KneeOffset    ] = GETDUTY(ERS210Info::CPCJointLFKnee);
	
		outputs[RFrLegOffset + RotatorOffset   ] = GETD(ERS210Info::CPCJointRFRotator);
		outputs[RFrLegOffset + ElevatorOffset  ] = GETD(ERS210Info::CPCJointRFElevator);
		outputs[RFrLegOffset + KneeOffset      ] = GETD(ERS210Info::CPCJointRFKnee);
		pidduties[RFrLegOffset + RotatorOffset ] = GETDUTY(ERS210Info::CPCJointRFRotator);
		pidduties[RFrLegOffset + ElevatorOffset] = GETDUTY(ERS210Info::CPCJointRFElevator);
		pidduties[RFrLegOffset + KneeOffset    ] = GETDUTY(ERS210Info::CPCJointRFKnee);
	
		outputs[LBkLegOffset + RotatorOffset   ] = GETD(ERS210Info::CPCJointLHRotator);
		outputs[LBkLegOffset + ElevatorOffset  ] = GETD(ERS210Info::CPCJointLHElevator);
		outputs[LBkLegOffset + KneeOffset      ] = GETD(ERS210Info::CPCJointLHKnee);
		pidduties[LBkLegOffset + RotatorOffset ] = GETDUTY(ERS210Info::CPCJointLHRotator);
		pidduties[LBkLegOffset + ElevatorOffset] = GETDUTY(ERS210Info::CPCJointLHElevator);
		pidduties[LBkLegOffset + KneeOffset    ] = GETDUTY(ERS210Info::CPCJointLHKnee);

		outputs[RBkLegOffset + RotatorOffset   ] = GETD(ERS210Info::CPCJointRHRotator);
		outputs[RBkLegOffset + ElevatorOffset  ] = GETD(ERS210Info::CPCJointRHElevator);
		outputs[RBkLegOffset + KneeOffset      ] = GETD(ERS210Info::CPCJointRHKnee);
		pidduties[RBkLegOffset + RotatorOffset ] = GETDUTY(ERS210Info::CPCJointRHRotator);
		pidduties[RBkLegOffset + ElevatorOffset] = GETDUTY(ERS210Info::CPCJointRHElevator);
		pidduties[RBkLegOffset + KneeOffset    ] = GETDUTY(ERS210Info::CPCJointRHKnee);

		// Get head tilt,pan,roll joint angles
		outputs[HeadOffset+TiltOffset] = GETD(ERS210Info::CPCJointNeckTilt);
		outputs[HeadOffset+PanOffset ] = GETD(ERS210Info::CPCJointNeckPan);
		outputs[HeadOffset+RollOffset] = GETD(ERS210Info::CPCJointNeckRoll);
		pidduties[HeadOffset+TiltOffset] = GETDUTY(ERS210Info::CPCJointNeckTilt);
		pidduties[HeadOffset+PanOffset ] = GETDUTY(ERS210Info::CPCJointNeckPan);
		pidduties[HeadOffset+RollOffset] = GETDUTY(ERS210Info::CPCJointNeckRoll);

		outputs[ERS210Info::TailOffset+TiltOffset] = GETD(ERS210Info::CPCJointTailTilt);
		outputs[ERS210Info::TailOffset+PanOffset]  = GETD(ERS210Info::CPCJointTailPan);
		pidduties[ERS210Info::TailOffset+TiltOffset] = GETDUTY(ERS210Info::CPCJointTailTilt);
		pidduties[ERS210Info::TailOffset+PanOffset]  = GETDUTY(ERS210Info::CPCJointTailPan);
		
		outputs[ERS210Info::MouthOffset] = GETD(ERS210Info::CPCJointMouth);
		pidduties[ERS210Info::MouthOffset] = GETDUTY(ERS210Info::CPCJointMouth);

		// Get foot switches
		chkEvent(er,LFrPawOffset,GETB(ERS210Info::CPCSensorLFPaw),"LFrPaw");
		chkEvent(er,RFrPawOffset,GETB(ERS210Info::CPCSensorRFPaw),"RFrPaw");
		chkEvent(er,LBkPawOffset,GETB(ERS210Info::CPCSensorLHPaw),"LBkPaw");
		chkEvent(er,RBkPawOffset,GETB(ERS210Info::CPCSensorRHPaw),"RBkPaw");

		// Get buttons
		chkEvent(er,ChinButOffset,  ((OSwitchValue)sensor.GetData(ERS210Info::CPCSensorChinSwitch)->frame[0].value == oswitchON)?1:0,"ChinBut");
		chkEvent(er,BackButOffset,  ((OSwitchValue)sensor.GetData(ERS210Info::CPCSensorBackSwitch)->frame[0].value == oswitchON)?1:0,"BackBut");
		chkEvent(er,HeadFrButOffset,GETD(ERS210Info::CPCSensorHeadFrontPressure),"HeadFrBut");
		chkEvent(er,HeadBkButOffset,GETD(ERS210Info::CPCSensorHeadBackPressure),"HeadBkBut");

		// Get IR distance sensor
		sensors[IRDistOffset]=((float)sensor.GetData(ERS210Info::CPCSensorPSD)->frame[0].value) / 1000.0;

		// Get acceleration sensors
		sensors[BAccelOffset] = GETD(ERS210Info::CPCSensorAccelFB);
		sensors[LAccelOffset] = GETD(ERS210Info::CPCSensorAccelLR);
		sensors[DAccelOffset] = GETD(ERS210Info::CPCSensorAccelUD);

		sensors[ThermoOffset] = GETD(ERS210Info::CPCSensorThermoSensor);
	}

	// (ERS-220 only)
	if(robotDesign&ERS220Mask) {
		outputs[LFrLegOffset + RotatorOffset   ] = GETD(ERS220Info::CPCJointLFRotator);
		outputs[LFrLegOffset + ElevatorOffset  ] = GETD(ERS220Info::CPCJointLFElevator);
		outputs[LFrLegOffset + KneeOffset      ] = GETD(ERS220Info::CPCJointLFKnee);
		pidduties[LFrLegOffset + RotatorOffset ] = GETDUTY(ERS220Info::CPCJointLFRotator);
		pidduties[LFrLegOffset + ElevatorOffset] = GETDUTY(ERS220Info::CPCJointLFElevator);
		pidduties[LFrLegOffset + KneeOffset    ] = GETDUTY(ERS220Info::CPCJointLFKnee);
	
		outputs[RFrLegOffset + RotatorOffset   ] = GETD(ERS220Info::CPCJointRFRotator);
		outputs[RFrLegOffset + ElevatorOffset  ] = GETD(ERS220Info::CPCJointRFElevator);
		outputs[RFrLegOffset + KneeOffset      ] = GETD(ERS220Info::CPCJointRFKnee);
		pidduties[RFrLegOffset + RotatorOffset ] = GETDUTY(ERS220Info::CPCJointRFRotator);
		pidduties[RFrLegOffset + ElevatorOffset] = GETDUTY(ERS220Info::CPCJointRFElevator);
		pidduties[RFrLegOffset + KneeOffset    ] = GETDUTY(ERS220Info::CPCJointRFKnee);
	
		outputs[LBkLegOffset + RotatorOffset   ] = GETD(ERS220Info::CPCJointLHRotator);
		outputs[LBkLegOffset + ElevatorOffset  ] = GETD(ERS220Info::CPCJointLHElevator);
		outputs[LBkLegOffset + KneeOffset      ] = GETD(ERS220Info::CPCJointLHKnee);
		pidduties[LBkLegOffset + RotatorOffset ] = GETDUTY(ERS220Info::CPCJointLHRotator);
		pidduties[LBkLegOffset + ElevatorOffset] = GETDUTY(ERS220Info::CPCJointLHElevator);
		pidduties[LBkLegOffset + KneeOffset    ] = GETDUTY(ERS220Info::CPCJointLHKnee);

		outputs[RBkLegOffset + RotatorOffset   ] = GETD(ERS220Info::CPCJointRHRotator);
		outputs[RBkLegOffset + ElevatorOffset  ] = GETD(ERS220Info::CPCJointRHElevator);
		outputs[RBkLegOffset + KneeOffset      ] = GETD(ERS220Info::CPCJointRHKnee);
		pidduties[RBkLegOffset + RotatorOffset ] = GETDUTY(ERS220Info::CPCJointRHRotator);
		pidduties[RBkLegOffset + ElevatorOffset] = GETDUTY(ERS220Info::CPCJointRHElevator);
		pidduties[RBkLegOffset + KneeOffset    ] = GETDUTY(ERS220Info::CPCJointRHKnee);

		// Get head tilt,pan,roll joint angles
		outputs[HeadOffset+TiltOffset] = GETD(ERS220Info::CPCJointNeckTilt);
		outputs[HeadOffset+PanOffset ] = GETD(ERS220Info::CPCJointNeckPan);
		outputs[HeadOffset+RollOffset] = GETD(ERS220Info::CPCJointNeckRoll);
		pidduties[HeadOffset+TiltOffset] = GETDUTY(ERS220Info::CPCJointNeckTilt);
		pidduties[HeadOffset+PanOffset ] = GETDUTY(ERS220Info::CPCJointNeckPan);
		pidduties[HeadOffset+RollOffset] = GETDUTY(ERS220Info::CPCJointNeckRoll);

	 	chkEvent(er, ERS220Info::TailLeftButOffset, GETB(ERS220Info::CPCSensorTailLeftSwitch),  "LBkBut");
	 	chkEvent(er, ERS220Info::TailCenterButOffset, GETB(ERS220Info::CPCSensorTailCenterSwitch),"CBkBut");
	 	chkEvent(er, ERS220Info::TailRightButOffset, GETB(ERS220Info::CPCSensorTailRightSwitch), "RBkBut");

		// Get foot switches
		chkEvent(er,LFrPawOffset,GETB(ERS220Info::CPCSensorLFPaw),"LFrPaw");
		chkEvent(er,RFrPawOffset,GETB(ERS220Info::CPCSensorRFPaw),"RFrPaw");
		chkEvent(er,LBkPawOffset,GETB(ERS220Info::CPCSensorLHPaw),"LBkPaw");
		chkEvent(er,RBkPawOffset,GETB(ERS220Info::CPCSensorRHPaw),"RBkPaw");

		// Get buttons
		chkEvent(er,ChinButOffset,  ((OSwitchValue)sensor.GetData(ERS220Info::CPCSensorChinSwitch)->frame[0].value == oswitchON)?1:0,"ChinBut");
		chkEvent(er,BackButOffset,  ((OSwitchValue)sensor.GetData(ERS220Info::CPCSensorBackSwitch)->frame[0].value == oswitchON)?1:0,"BackBut");
		chkEvent(er,HeadFrButOffset,GETD(ERS220Info::CPCSensorHeadFrontPressure),"HeadFrBut");
		chkEvent(er,HeadBkButOffset,GETD(ERS220Info::CPCSensorHeadBackPressure),"HeadBkBut");

		// Get IR distance sensor
		sensors[IRDistOffset]=((float)sensor.GetData(ERS220Info::CPCSensorPSD)->frame[0].value) / 1000.0;

		// Get acceleration sensors
		sensors[BAccelOffset] = GETD(ERS220Info::CPCSensorAccelFB);
		sensors[LAccelOffset] = GETD(ERS220Info::CPCSensorAccelLR);
		sensors[DAccelOffset] = GETD(ERS220Info::CPCSensorAccelUD);

		sensors[ThermoOffset] = GETD(ERS220Info::CPCSensorThermoSensor);
	}

	unsigned int dif=curtime-lastSensorUpdateTime;
	lastSensorUpdateTime=curtime;
	er->postEvent(EventBase::sensorEGID,SensorSourceID::UpdatedSID,EventBase::statusETID,dif);
}

/*! This will cause events to be posted */
void WorldState::read(const OPowerStatus& power, EventRouter* er) {
	std::string actnames[PowerSourceID::NumPowerSIDs];
	std::string denames[PowerSourceID::NumPowerSIDs];
	unsigned int actmasks[PowerSourceID::NumPowerSIDs];
	memset(actmasks,0,sizeof(unsigned int)*PowerSourceID::NumPowerSIDs);

	//RobotStatus
	chkPowerEvent(PowerSourceID::PauseSID,          power.robotStatus,orsbPAUSE,                        "Pause",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::MotorPowerSID,     power.robotStatus,orsbMOTOR_POWER,                  "MotorPower",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::VibrationSID,      power.robotStatus,orsbVIBRATION_DETECT,             "Vibration",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ExternalPortSID,   power.robotStatus,orsbEX_PORT_CONNECTED,            "ExternalPort",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::StationConnectSID, power.robotStatus,orsbSTATION_CONNECTED,            "StationConnect",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ExternalPowerSID,  power.robotStatus,orsbEX_POWER_CONNECTED,           "ExternalPower",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryConnectSID, power.robotStatus,orsbBATTERY_CONNECTED,            "BatteryConnect",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ChargingSID,       power.robotStatus,orsbBATTERY_CHARGING,             "BatteryCharging",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryFullSID,    power.robotStatus,orsbBATTERY_CAPACITY_FULL,        "BatteryFull",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::LowPowerWarnSID,   power.robotStatus,orsbBATTERY_CAPACITY_LOW,         "BatteryLow",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverChargedSID,    power.robotStatus,orsbBATTERY_OVER_CURRENT,         "BatteryOverCurrent",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverheatingSID,    power.robotStatus,orsbBATTERY_OVER_TEMP_DISCHARGING,"BatteryOverTempDischarge",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverheatingSID,    power.robotStatus,orsbBATTERY_OVER_TEMP_CHARGING,   "BatteryOverTempCharge",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbBATTERY_ERROR_OF_CHARGING,    "BatteryChargeError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbERROR_OF_PLUNGER,             "PlungerError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::PowerGoodSID,      power.robotStatus,orsbOPEN_R_POWER_GOOD,            "PowerGood",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbERROR_OF_FAN,                 "FanError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::DataFromStationSID,power.robotStatus,orsbDATA_STREAM_FROM_STATION,     "DataFromStation",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RegisterUpdateSID, power.robotStatus,orsbREGISTER_UPDATED_BY_STATION,  "RegisterUpdate",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbRTC_ERROR,                    "RTCError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RTCSID,            power.robotStatus,orsbRTC_OVERFLOW,                 "RTCOverflow",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RTCSID,            power.robotStatus,orsbRTC_RESET,                    "RTCReset",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RTCSID,            power.robotStatus,orsbRTC_SET,                      "RTCSet",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::SpecialModeSID,    power.robotStatus,orsbSPECIAL_MODE,                 "SpecialMode",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BMNDebugModeSID,   power.robotStatus,orsbBMN_DEBUG_MODE,               "BMNDebugMode",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ChargerStatusSID,  power.robotStatus,orsbCHARGER_STATUS,               "ChargerStatus",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::PlungerSID,        power.robotStatus,orsbPLUNGER,                      "Plunger",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::SuspendedSID,      power.robotStatus,orsbSUSPENDED,                    "Suspended",actnames,denames,actmasks);

	//BatteryStatus
	chkPowerEvent(PowerSourceID::ErrorSID,        power.batteryStatus,obsbERROR_CODE_MASK,             "BatteryError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryEmptySID, power.batteryStatus,obsbFULLY_DISCHARGED,            "FullyDischarged",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryFullSID,  power.batteryStatus,obsbFULLY_CHARGED,               "FullyCharged",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::DischargingSID,  power.batteryStatus,obsbDISCHARGING,                 "Discharging",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryInitSID,  power.batteryStatus,obsbINITIALIZED,                 "BatteryInit",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::LowPowerWarnSID, power.batteryStatus,obsbREMAINING_TIME_ALARM,        "RemainingTimeAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::LowPowerWarnSID, power.batteryStatus,obsbREMAINING_CAPACITY_ALARM,    "RemainingCapacityAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::TermDischargeSID,power.batteryStatus,obsbTERMINATED_DISCHARGING_ALARM,"TermDischargeAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverheatingSID,  power.batteryStatus,obsbOVER_TEMP_ALARM,             "OverTempAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::TermChargeSID,   power.batteryStatus,obsbTERMINATED_CHARGING_ALARM,   "TermChargeAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverChargedSID,  power.batteryStatus,obsbOVER_CHARGED_ALARM,          "OverChargeAlarm",actnames,denames,actmasks);
	
	sensors[PowerRemainOffset] = power.remainingCapacity/100.0;
	sensors[PowerThermoOffset] = power.temperature/100.0;
	sensors[PowerCapacityOffset] = power.fullyChargedCapacity;
	sensors[PowerVoltageOffset] = power.voltage/1000.0;
	sensors[PowerCurrentOffset] = power.current;

	//only generate status events when a change happens
	for(unsigned int i=0; i<PowerSourceID::NumPowerSIDs; i++) {
		if(actmasks[i]) { //now on
			if(!powerFlags[i]) //was off: activation
				er->postEvent(EventBase::powerEGID,i,EventBase::activateETID,0,actnames[i],1);
			else if(actmasks[i]!=powerFlags[i]) //already on - change? : status
				er->postEvent(EventBase::powerEGID,i,EventBase::statusETID,0,actnames[i],1);
		} else { // now off
			if(powerFlags[i]) //was on: deactivation
				er->postEvent(EventBase::powerEGID,i,EventBase::deactivateETID,0,denames[i],0);
		}
		powerFlags[i]=actmasks[i];
	}

	er->postEvent(EventBase::powerEGID,PowerSourceID::UpdatedSID,EventBase::statusETID,0);
}

#endif //PLATFORM_APERIOS

void WorldState::chkEvent(EventRouter* er, unsigned int off, float newval, const char* name) {
	ButtonSourceID::ButtonSourceID_t sid=static_cast<ButtonSourceID::ButtonSourceID_t>(off); //should be a direct conversion
	if(newval>=0.1) { //now on
		if(buttons[sid]<0.1) { //was off: activation
			er->postEvent(EventBase::buttonEGID,sid,EventBase::activateETID,0,name,newval);
			button_times[sid]=curtime;
		} else if(alwaysGenerateStatus || buttons[sid]!=newval) { //already on - always or change? : status
			unsigned int dur=curtime-button_times[sid];
			er->postEvent(EventBase::buttonEGID,sid,EventBase::statusETID,dur,name,newval);
		}
	} else { //now off
		if(buttons[sid]>=0.1) { //was on: deactivation
			unsigned int dur=curtime-button_times[sid];
			button_times[sid]=0;
			er->postEvent(EventBase::buttonEGID,sid,EventBase::deactivateETID,dur,name,0);
		}
	}
	//update value
	buttons[sid]=newval;
}

/*! @file
 * @brief Implements WorldState, maintains information about the robot's environment, namely sensors and power status
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_5 $
 * $Revision: 1.16 $
 * $State: Rel $
 * $Date: 2003/09/25 15:31:53 $
 */
