//-*-c++-*-
#ifndef INCLUDED_EventBase_h
#define INCLUDED_EventBase_h

#include "Shared/get_time.h"
#include "Shared/LoadSave.h"
#include <string>

//! The basis of events passed around the high level system
/*! Contains the list of 'known' event generators in EventGeneratorID_t
 *  and EventGeneratorNames[].  If you want to make a new generator, all
 *  you have to do is add a new entry to the ID list (EventGeneratorID_t)
 *  and then put it's name in the EventGeneratorNames[] array.
 *
 *  Alternatively, there is an 'unlicensed spectrum' available under
 *  the unknownEGID.  You can send out events from that generator just
 *  like any other, but it should only be used for quick tests and hacking
 *  around...
 *
 *  The SourceID is generator specific.  A SID of @a i from the button
 *  generator will refer to a particular button, whereas a SID from vision
 *  refers to seeing a particular object.  These SIDs are usually defined
 *  in the generators themselves.  See the EventGeneratorID_t list for
 *  links to the generators.
 *
 *  The duration field is also generator specific - some may refer to
 *  the time since the last activation event (e.g. button events) where 
 *  as others refer to time since last status (e.g. sensors updates)
 *
 */
class EventBase : public LoadSave {
 public:
	//! Lists all possible event generator ids
	/*! An event generator is a abstract source of events, used for listening to and parsing certain classes of events
	 *
	 *  IF YOU ADD AN EVENT GENERATOR, DON'T FORGET TO NAME IT (EventBase::EventGeneratorNames, below)*/
	enum EventGeneratorID_t {
		unknownEGID=0,    //!< default EGID, used if you forget to set it, probably not a good idea to use this for anything except errors or testing quick hacks
		visionEGID,       //!< vision processing from camera
		buttonEGID,       //!< button up, down, and still down, @see ButtonSourceID::ButtonSourceID_t
		worldModelEGID,   //!< not being used, yet (for when objects are detected/lost?)
		aiEGID,           //!< not being used, yet (might use this when AI makes decisions?)
		audioEGID,        //!< Sends an event when a sound starts/ends playback, status events as chained sounds end
		sensorEGID,       //!< currently only used to alert of new sensor readings, may also want to generate IR proximity warnings? @see SensorSourceID::SensorSourceID_t
		powerEGID,        //!< used to generate low power warnings, temperature, etc. @see PowerSourceID::PowerSourceID_t
		timerEGID,        //!< EGID from which timers are sent, you set timers, but you don't have to also listen @see EventRouter::setTimer()
		stateMachineEGID, //!< Sends an event upon entering and leaving a StateNode.
		locomotionEGID,   //!< Sends events regarding transportation in the world; you can/should assume these will all be LocomotionEvent classes
    textmsgEGID,      //!< Sends events when a text msg is received on console
		estopEGID,        //!< Sends an event when the estop is turned on or off
		motmanEGID,       //!< Sends events when a MotionCommand is added or removed
		wmVarEGID,        //!< Sends an event when a watched memory is changed; source id is pointer to WMEntry
		numEGIDs          //!< the number of generators available
	};

	//! Holds string versions of each of the generator's names, handy for debugging so you can output the events as readable strings
	static const char* const EventGeneratorNames[numEGIDs];
	
	//! an event type id is used to denote whether it's the first in a sequence (button down), in a sequence (button still down), or last (button up)
	enum EventTypeID_t {
		activateETID,   //!< e.g. button down
		statusETID,     //!< e.g. button still down
		deactivateETID, //!< e.g. button up
		numETIDs        //!< the number of different event types
	};

	/*! @name Constructors/Destructors */
	//! constructor
	/*! @see EventRouter::postEvent() */
	EventBase(); 
	EventBase(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur=0);
	EventBase(EventGeneratorID_t gid, unsigned int sid, EventTypeID_t tid, unsigned int dur, const std::string& n, float mag);
	virtual ~EventBase() {} //!< destructor
	//@}

	/*! @name Methods */
	virtual const std::string& getName() const { return stim_id; } //!< gets the name of the event - useful for debugging, outputs
	virtual EventBase& setName(const std::string& n) { nameisgen=false; stim_id=n; return *this; } //!< sets name to a given string, prevents overwriting by generated names

	virtual float getMagnitude() const { return magnitude; } //!< gets "strength" of event - you might have useful values... used by AI
	virtual EventBase& setMagnitude(float m) { magnitude=m; return *this; }//!< sets "strength" of event - but you might have useful values... used by AI

	virtual unsigned int getTimeStamp() const { return timestamp; } //!< time event was created

	virtual EventGeneratorID_t getGeneratorID() const { return genID; } /*!< @brief gets the generator ID for this event @see EventGeneratorID_t */
	virtual EventBase& setGeneratorID(EventGeneratorID_t gid) { genID=gid; genName(); return *this; } /*!< @brief sets the generator ID for this event @see EventGeneratorID_t */
	
	virtual unsigned int getSourceID() const { return sourceID; } /*!< @brief gets the source ID for this event @see sourceID */
	virtual EventBase& setSourceID(unsigned int gid) { sourceID=gid; genName(); return *this; } /*!< @brief sets the source ID for this event @see sourceID */
	
	virtual EventTypeID_t getTypeID() const { return typeID; } /*!< @brief gets the type ID @see EventTypeID_t */
	virtual EventBase& setTypeID(EventTypeID_t tid) { typeID=tid; genName(); return *this; } /*!< @brief sets the type ID @see EventTypeID_t */

	virtual unsigned int getDuration() const { return duration; } /*!< @brief OPTIONAL gets the time since the beginning of this sequence (the timestamp of the activate event) @see duration */
	virtual EventBase& setDuration(unsigned int d) { duration = d; return *this; }/*!< @brief OPTIONAL gets the time since the beginning of this sequence (the timestamp of the activate event) @see duration */

	virtual const std::string& resetName() { nameisgen=true; genName(); return stim_id; } //!< resets name to generated form, overwriting any previous name
	virtual bool isCustomName() const { return !nameisgen; } //!< returns true if not using the generated name

	inline bool operator<(const EventBase& e) const { return timestamp<e.timestamp; }

	//! is true if the genID, typeID, and sourceID's all match
	virtual bool operator==(const EventBase& eb) const {
		return (sourceID==eb.sourceID && genID==eb.genID && typeID==eb.typeID);
	}
	//!tests to see if events have the same generator and source IDs
	bool sameGenSource(const EventBase eb) const { return genID==eb.genID && sourceID==eb.sourceID; }
	
	bool longerThan(const EventBase eb) const { return duration>eb.duration && operator==(eb); } //!< compares event duration and ensures same event generator, source, and type - useful for event masks
	bool shorterThan(const EventBase eb) const { return duration<eb.duration && operator==(eb); }//!< compares event duration and ensures same event generator, source, and type - useful for event masks
	bool equalOrLongerThan(const EventBase eb) const { return duration>=eb.duration && operator==(eb); }//!< compares event duration and ensures same event generator, source, and type - useful for event masks
	bool equalOrShorterThan(const EventBase eb) const { return duration<=eb.duration && operator==(eb); }//!< compares event duration and ensures same event generator, source, and type - useful for event masks
	//@}

	//! Useful for serializing events to send between processes
	/*! @name LoadSave interface */
	virtual unsigned int getBinSize() const;
	virtual unsigned int LoadBuffer(const char buf[], unsigned int len);
	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const;
	//@}
 protected:
  std::string stim_id; //!< the name of the event, use the same name consistently or else will be seen as different stimuli
  float magnitude; //!< the current "strength" of the event/stimuli... MAKE SURE this gets set to ZERO IF event is DEACTIVATE
  unsigned int timestamp; //!< the time the event was created - set automatically by constructor

	bool nameisgen; //!< tracks whether the current name (stim_id) is generated or set
	virtual void genName(); //!< This does the actual generation of names based on genID, sourceID, and typeID

	EventGeneratorID_t genID; //!< generator ID, @see EventGeneratorID_t
	EventTypeID_t typeID; //!< type ID, @see EventTypeID_t
	unsigned int sourceID; /*!< @brief the source ID for this event
													* Source IDs are defined by the generator that made it.  This should
													* give authors flexibility to design their modules without having to
													* worry about ID space collision */
	unsigned int duration; /*!< @brief the time since this sequence started (like, how long the button has been pressed)
													*   ideally, this would be 0 for activate, (activate.timestamp-get_time()) for status and deactivate */
};

/*! @file
 * @brief Describes EventBase, the basic class for sending events around the system
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_5 $
 * $Revision: 1.14 $
 * $State: Rel $
 * $Date: 2003/10/07 01:00:28 $
 */

#endif
