#include "EventLogger.h"
#include "Events/EventRouter.h"
#include "Motion/MMAccessor.h"
#include "Motion/LedMC.h"
#include "ValueEditControl.h"
#include "StringInputControl.h"
#include "NullControl.h"
#include <sstream>

EventLogger::EventLogger() : ControlBase("Event Logger","Allows you to see/log all of the un-trapped events as they are generated"), logfilePath(), logfile(), verbosity(0) {
	for(unsigned int i=0; i<EventBase::numEGIDs; i++) {
		std::string tmp=EventBase::EventGeneratorNames[i];
		pushSlot(new NullControl(("[ ] "+tmp).c_str(),"Show/hide events from "+tmp));
	}
	pushSlot(NULL);
	pushSlot(new ValueEditControl<unsigned int>("Verbosity","Controls verbosity level: 0=name,type; 1=0+duration,timestamp; 2=1+magnitude","Please enter a new verbosity level: 0=name,type; 1=0+duration,timestamp; 2=1+magnitude",&verbosity));
	pushSlot(new ControlBase("[X] Console Output","If selected, outputs events to the console"));
	pushSlot(new StringInputControl("[ ] File Output","Please enter the filename to log to (in /ms/...)"));
}

ControlBase* EventLogger::doSelect() {
	ControlBase* ans=this;
	for(unsigned int i=0; i<hilights.size(); i++) {
		unsigned int cur=hilights[i];
		if(cur<EventBase::numEGIDs) {
			if(options[cur]->getName()[1]!=' ') {
				erouter->removeListener(this,(EventBase::EventGeneratorID_t)(cur));
				setStatus(cur,' ');
			} else {
				erouter->addListener(this,(EventBase::EventGeneratorID_t)(cur));
				setStatus(cur,'X');
			}
		} else if(cur==EventBase::numEGIDs+1) {
			ans=options[cur];
		} else if(cur==EventBase::numEGIDs+2) {
			if(options[cur]->getName()[1]!=' ') {
				setStatus(cur,' ');
			} else {
				setStatus(cur,'X');
			}
		} else if(cur==EventBase::numEGIDs+3) {
			if(options[cur]->getName()[1]!=' ') {
				logfile.close();
				options[cur]->setName("[ ] File Output");
			} else {
				ans=options[cur];
			}
		}
		sndman->PlayFile(config->controller.select_snd);
	}
	if(ans==this)
		refresh();
	return ans;
}

void EventLogger::refresh() {
	checkLogFile();
	ControlBase::refresh();
}

//!sends all events received to stdout and/or logfile
void EventLogger::processEvent(const EventBase& event) {
	std::ostringstream logdata;
	logdata << event.getName();
	switch(event.getTypeID()) {
	case EventBase::activateETID:
		logdata << "\tA"; break;
	case EventBase::statusETID:
		logdata << "\tS"; break;
	case EventBase::deactivateETID:
		logdata << "\tD"; break;
	case EventBase::numETIDs:
		logdata << "\tErr"; break;
	}
	if(verbosity>=1) {
		logdata << '\t' << event.getDuration() << '\t' << event.getTimeStamp();
		if(verbosity>=2) {
			logdata << '\t' << event.getMagnitude();
		}
	}
	
	if(options[EventBase::numEGIDs+2]->getName()[1]=='X')
		cout << "EVENT: " << logdata.str() << endl;
	checkLogFile();
	if(logfile)
		logfile << logdata.str() << endl;
}

void EventLogger::setStatus(unsigned int i, char c) {
	std::string tmp=options[i]->getName();
	tmp[1]=c;
	options[i]->setName(tmp);
}

void EventLogger::checkLogFile() {
	unsigned int cur=EventBase::numEGIDs+3;
	StringInputControl * strin=dynamic_cast<StringInputControl*>(options[cur]);
	ASSERTRET(strin!=NULL,"The StringInputControl is misplaced");
	if(strin->getLastInput()!=logfilePath) {
		logfile.close();
		logfilePath=strin->getLastInput();
		logfile.clear();
		if(logfilePath.size()!=0) {
			cout << "Opening `/ms/"<<logfilePath<<"'"<<endl;
			logfile.open(("/ms/"+logfilePath).c_str());
			if(!logfile.fail()) {
				setStatus(cur,'X');
				strin->setName(strin->getName()+": "+logfilePath);
			} else {
				cout << "opening failed" << endl;
			}
		}
	}
}

/*! @file
 * @brief Describes EventLogger, which allows logging of events to the console or a file
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_5 $
 * $Revision: 1.8 $
 * $State: Rel $
 * $Date: 2003/09/25 15:26:10 $
 */
