/*
 * almUtility.cc -- contains utility routines for the AIBO LocalMap system.
 *
 * Started on 12/16/2002 by tss
 */

#include "Configuration.h"

#include <cmath>

#ifdef UNIT_TEST_ALM_UT
#warning Compiling unit test code for almUtility.cc
#include <iostream>
#endif

/* This routine takes a grid cell index in the spherical depth map and returns
 * the corresponding azimuth and altitude angles. */
void dm_index2angles(int index, double &azimuth, double &altitude)
{
  static const int width  = ALM_DM_H_SIZE;
  static const int height = ALM_DM_V_SIZE;

  static const double xsize = ALM_DM_RIGHT - ALM_DM_LEFT;
  static const double ysize = ALM_DM_BOTTOM -ALM_DM_TOP;

  azimuth = ALM_DM_LEFT + xsize * (double)(index % width) / (double)width;
  altitude = ALM_DM_TOP + ysize * (double)(index / width) / (double)height;
}

/* This routine does the opposite of the routine above */
bool angles2dm_index(double azimuth, double altitude, int &index)
{
  static const int width  = ALM_DM_H_SIZE;
  static const int height = ALM_DM_V_SIZE;

  static const double xsize = ALM_DM_RIGHT - ALM_DM_LEFT;
  static const double ysize = ALM_DM_BOTTOM -ALM_DM_TOP;

  double dx = azimuth - ALM_DM_LEFT;
  double dy = altitude - ALM_DM_TOP;

  int xindex = (int) ((dx / xsize) * (double) width);
  int yindex = (int) ((dy / ysize) * (double) height);

  if((xindex < 0) || (yindex < 0)) return false;
  if((xindex >= width) || (yindex >= height)) return false;

  index = yindex*width + xindex;
  return true;
}

/* This routine converts an index into the horizontal height map and returns
 * the corresponding X and Y locations relative to the AIBO neck tilt
 * pivot */
void hm_index2xy(int index, double &x, double &y)
{
  static const int width  = 2*ALM_HM_SIZE;
  static const int height = 2*ALM_HM_SIZE;

  static const double xsize =  2*ALM_HM_RADIUS;
  static const double ysize = -2*ALM_HM_RADIUS;

  x = (double) -ALM_HM_RADIUS +
	xsize * (double)(index % width) / (double)width;
  y = (double)  ALM_HM_RADIUS +
	ysize * (double)(index / width) / (double)height;
}

/* This routine does the opposite of the routine above. It is the user's
 * responsibility to make certain the index is in bounds. */
bool xy2hm_index(double x, double y, int &index)
{
  static const int width  = 2*ALM_HM_SIZE;
  static const int height = 2*ALM_HM_SIZE;

  static const double xsize =  2*ALM_HM_RADIUS;
  static const double ysize = -2*ALM_HM_RADIUS;

  double dx = x - -ALM_HM_RADIUS;
  double dy = y -  ALM_HM_RADIUS;

  int xindex = (int) ((dx / xsize) * (double) width);
  int yindex = (int) ((dy / ysize) * (double) height);

  if((xindex < 0) || (yindex < 0)) return false;
  if((xindex >= width) || (yindex >= height)) return false;

  index = yindex*width + xindex;
  return true;
}

/* This routine converts an index into the global height map and returns
 * the corresponding allocentric X and Y locations. */
void gm_index2xy(int index, double &x, double &y)
{
  static const int width  = AGM_H_SIZE;
  static const int height = AGM_V_SIZE;

  static const double xsize = (double) AGM_RIGHT - (double) AGM_LEFT;
  static const double ysize = (double) AGM_BOTTOM - (double) AGM_TOP;

  x = (double) AGM_LEFT + xsize * (double)(index % width) / (double)width;
  y = (double) AGM_TOP  + ysize * (double)(index / width) / (double)height;
}

/* This routine does the opposite of the routine above. It is the user's
 * responsibility to make certain the index is in bounds. */
bool xy2gm_index(double x, double y, int &index)
{
  static const int width  = AGM_H_SIZE;
  static const int height = AGM_V_SIZE;

  static const double xsize = (double) AGM_RIGHT - (double) AGM_LEFT;
  static const double ysize = (double) AGM_BOTTOM - (double) AGM_TOP;

  double dx = x - AGM_LEFT;
  double dy = y - AGM_TOP;

  int xindex = (int) ((dx / xsize) * (double) width);
  int yindex = (int) ((dy / ysize) * (double) height);

  if((xindex < 0) || (yindex < 0)) return false;
  if((xindex >= width) || (yindex >= height)) return false;

  index = yindex*width + xindex;
  return true;
}

/* This routine takes a depth measurement and the pan/tilt angles of the
 * head and returns the 3D XYZ position of the measured point. The
 * XYZ coordinate system has its origin on the ground just beneath the
 * center of AIBO's tilt pivot point. The X axis points out in front of
 * the AIBO, the Y axis leftwards, and the Z axis straight up. */
void head_range2xyz(double depth, double pan, double tilt,
		    double &x, double &y, double &z)
{
  // To accomplish this task, we need a transformation matrix. The local
  // variables in this function correspond to the following transformation
  // matrix locations:
  //   [    t11      t12    sinntilt  t14 ]
  //   [ -sinnpan  cosnpan      0     t24 ]
  //   [    t31      t32    cosntilt  t34 ]
  //   [     0        0         0      1  ]
  // Since the vector for the distance of the obstacle from the head is
  // just of the form [ depth 0 0 1 ] in AIBO "xyz" head coordinates,
  // we don't need to compute the values for the inner two columns of the
  // matrix. So we ignore t12 and t32.

  double sinntilt = sin(-tilt);
  double cosntilt = cos(-tilt);
  double sinnpan  = sin(-pan);
  double cosnpan  = cos(-pan);

  double t11 = cosntilt*cosnpan;
  double t31 = -sinntilt*cosnpan;
  double t14 = AIBO_HEAD_LENGTH*cosntilt*cosnpan + AIBO_NECK_HEIGHT*sinntilt;
  double t24 = -AIBO_HEAD_LENGTH*sinnpan;
  double t34 = -AIBO_HEAD_LENGTH*sinntilt*cosnpan + AIBO_NECK_HEIGHT*cosntilt +
		AIBO_TILT_PIVOT_HEIGHT; // TODO: s/ATPH/motion model/

  x = t11*depth + t14;
  y = -sinnpan*depth + t24;
  z = t31*depth + t34;
}

/* This routine takes a location in the 3D XYZ coordinates mentioned above
 * and computes the range, azimuth, and altitude values for the location
 * from the center of the AIBO's tilt pivot point. */
void xyz2neck_range(double x, double y, double z,
		    double &depth, double &azimuth, double &altitude)
{
  double dz = z - AIBO_TILT_PIVOT_HEIGHT; // TODO -- replace w/ motion model

  depth = sqrt(x*x + y*y + dz*dz);
  azimuth = atan2(y, x);
  altitude = asin(dz/depth);
}

/* This routine does the reverse of xyz2neck_range. Not much more to say
 * about that */
void neck_range2xyz(double depth, double azimuth, double altitude,
		    double &x, double &y, double &z)
{
  // To accomplish this task, we need a transformation matrix. The local
  // variables in this function correspond to the following transformation
  // matrix locations:
  //   [    t11      t12   sinnalt            0            ]
  //   [ -sinnaz   cosnaz     0               0            ]
  //   [    t31      t32   cosnalt  AIBO_TILT_PIVOT_HEIGHT ]
  //   [     0        0       0               1            ]
  // Since the vector for the distance of the obstacle from the head is
  // just of the form [ depth 0 0 1 ] in AIBO "xyz" head coordinates,
  // we don't need to compute the values for the inner two columns of the
  // matrix. So we ignore t13 and t23.

  double sinnaz  = sin(-azimuth);
  double cosnaz  = cos(-azimuth);
  double sinnalt = sin(-altitude);
  double cosnalt = cos(-altitude);

  double t11 = cosnalt*cosnaz;
  double t31 = -sinnalt*cosnaz;

  x = t11*depth;
  y = -sinnaz*depth;
  z = t31*depth + AIBO_TILT_PIVOT_HEIGHT;
}

#ifdef UNIT_TEST_ALM_UT

int main(int argc, char **argv)
{
  int index;
  double az, alt, depth, x, y, z;

  using namespace std;

  cout << "   Mismatches of 1 or of " << ALM_DM_H_SIZE << "+-1 are probably OK"
       << endl;
  for(int i=0;
      i < ALM_DM_V_SIZE*ALM_DM_H_SIZE;
      i += ALM_DM_V_SIZE*ALM_DM_H_SIZE/13) {
    cout << "dm_index2angles(" << i <<", az, alt)";
    dm_index2angles(i, az, alt);
    cout << "\taz = " << az << "\talt = " << alt << endl;

    cout << "angles2dm_index(az, alt, index)";
    angles2dm_index(az, alt, index);
    cout << "\tindex = " << index;

    if(index != i) cout << "\t***MISMATCH***";

    cout << endl;
  }
  cout << endl << endl;

  for(int i=0;
      i < 4*ALM_HM_SIZE*ALM_HM_SIZE;
      i += 4*ALM_HM_SIZE*ALM_HM_SIZE/13) {
    cout << "hm_index2xy(" << i <<", x, y)";
    hm_index2xy(i, x, y);
    cout << "\tx = " << x << "\ty = " << y << endl;

    cout << "xy2hm_index(x, y, index)";
    xy2hm_index(x, y, index);
    cout << "\tindex = " << index;

    if(index != i) cout << "\t***MISMATCH***";

    cout << endl;
  }
  cout << endl << endl;

  for(int i=0;
      i < AGM_V_SIZE*AGM_H_SIZE;
      i += AGM_V_SIZE*AGM_H_SIZE/13) {
    cout << "gm_index2xy(" << i <<", x, y)";
    gm_index2xy(i, x, y);
    cout << "\tx = " << x << "\ty = " << y << endl;

    cout << "xy2gm_index(x, y, index)";
    xy2gm_index(x, y, index);
    cout << "\tindex = " << index;

    if(index != i) cout << "\t***MISMATCH***";

    cout << endl;
  }
  cout << endl << endl;

  cout << "head_range2xyz(200, 0, 0, x, y, z)";
  head_range2xyz(200, 0, 0, x, y, z);
  cout << "\tx = " << x << "  y = " << y << "  z = " << z << endl;
  cout << "head_range2xyz(200, R(45), 0, x, y, z)";
  head_range2xyz(200, R(45), 0, x, y, z);
  cout << "\tx = " << x << "  y = " << y << "  z = " << z << endl;
  cout << "head_range2xyz(200, 0, R(45), x, y, z)";
  head_range2xyz(200, 0, R(45), x, y, z);
  cout << "\tx = " << x << "  y = " << y << "  z = " << z << endl;
  cout << "head_range2xyz(200, R(45), R(45), x, y, z)";
  head_range2xyz(200, R(45), R(45), x, y, z);
  cout << "\tx = " << x << "  y = " << y << "  z = " << z << endl;
  cout << endl << endl;

  cout << "neck_range2xyz(200, 0, 0, x, y, z)";
  neck_range2xyz(200, 0, 0, x, y, z);
  cout << "\tx = " << x << "  y = " << y << "  z = " << z << endl;
  cout << "xyz2neck_range(x, y, z, depth, az, alt)";
  xyz2neck_range(x, y, z, depth, az, alt);
  cout << "\tdepth = " << depth << "  az = " << az << "  alt = " << alt << endl;
  cout << "neck_range2xyz(200, R(45), 0, x, y, z)";
  neck_range2xyz(200, R(45), 0, x, y, z);
  cout << "\tx = " << x << "  y = " << y << "  z = " << z << endl;
  cout << "xyz2neck_range(x, y, z, depth, az, alt)";
  xyz2neck_range(x, y, z, depth, az, alt);
  cout << "\tdepth = " << depth << "  az = " << az << "  alt = " << alt << endl;
  cout << "neck_range2xyz(200, 0, R(45), x, y, z)";
  neck_range2xyz(200, 0, R(45), x, y, z);
  cout << "\tx = " << x << "  y = " << y << "  z = " << z << endl;
  cout << "xyz2neck_range(x, y, z, depth, az, alt)";
  xyz2neck_range(x, y, z, depth, az, alt);
  cout << "\tdepth = " << depth << "  az = " << az << "  alt = " << alt << endl;

  return 0;
}

#endif
