/*
 * almStructures.h -- contains definitions for data structures used in the
 *   AIBO Local Map. Also, hm_cell is used in the global map as well.
 *
 * Started 12-12-2002, tss
 */

#ifndef _ALM_STRUCTURES_H_
#define _ALM_STRUCTURES_H_

#ifndef UNIT_TEST
#include "Vision/Vision.h"
#include "Vision/Visiondefines.h"
#endif

// Tekkotsu's segmented color type is an int. Our code was just using the dorky
// enum below, but now that we've integrated, it's an int.
#ifndef UNIT_TEST
typedef int colortype;
#else
typedef enum { RED, ORANGE, YELLOW, GREEN, BLUE, INDIGO, VIOLET } colortype;
#endif

//! A cell for the spherical depth map
typedef struct _dm_cell {
  float depth;		//!< depth in millimeters
  float confidence;	//!< 0 to 1 confidence value in cell data
  colortype color;	//!< The color of the object in this cell as seen from
			//!< the camera.
  int cluster;		//!< K-means uncertainty clustering. (Internal use only)
} dm_cell;

//! A cell for the horizontal height map
typedef struct _hm_cell {
  float height;		//!< height in millimeters; ground is 0
  float trav;		//!< 0 to 1 estimate of cell traverability (1=yes)
  float confidence;	//!< 0 to 1 confidence value in cell data
  colortype color;	//!< The color of the object in this cell as seen from
			//!< the camera.
  int cluster;		//!< K-means uncertainty clustering. (Internal use only)
} hm_cell;

// Pickers--abstract base class for classes that pick certain attributes
// out of dm_cells and hm_cells. Useful for dumping data with the several
// dump routines.
//   Yeah, templates probably could work better here.
//   PS: everything gets converted to float

//! @name Pickers
//! Pickers are handy routines that extract certain types of data from
//! hm_cell and dm_cell structs. They're used by the map accessor functions
//! in WorldModel2 to furnish the user with desired map data--the user simply
//! passes in a Picker, and they get the data back (as float only). Chances are
//! there's a better template-based solution to this problem.
//@{
//! Abstract base class for height map data pickers. Implementations available.
struct hmPicker {
	//!functor - TSS_TODO
  virtual float operator() (hm_cell &c) const = 0;
};
//! Abstract base class for depth map data pickers. Implementations available.
struct dmPicker {
	//!functor - TSS_TODO
  virtual float operator() (dm_cell &c) const = 0;
};

//! Picks height measurements out of the HHM or GHM.
struct hmPickHeight : public hmPicker {
	//!functor - TSS_TODO
  virtual float operator() (hm_cell &c) const { return c.height; }
};
//! Picks traversability values out of the HHM or GHM.
struct hmPickTrav : public hmPicker {
	//!functor - TSS_TODO
  virtual float operator() (hm_cell &c) const { return c.trav; }
};
//! Picks confidence values out of the HHM or GHM.
struct hmPickConfidence : public hmPicker {
	//!functor - TSS_TODO
  virtual float operator() (hm_cell &c) const { return c.confidence; }
};
//! Picks color measurements out of the HHM or GHM.
struct hmPickColor : public hmPicker {
	//!functor - TSS_TODO
  virtual float operator() (hm_cell &c) const { return c.color; }
};
//! Picks cluster membership out of the HHM or GHM. For completeness--not use!
struct hmPickCluster : public hmPicker {
	//!functor - TSS_TODO
  virtual float operator() (hm_cell &c) const { return c.cluster; }
};

//! Picks depth measurements out of the SDM.
struct dmPickDepth : public dmPicker {
	//!functor - TSS_TODO
  virtual float operator() (dm_cell &c) const { return c.depth; }
};
//! Picks confidence values out of the SDM.
struct dmPickConfidence : public dmPicker {
	//!functor - TSS_TODO
  virtual float operator() (dm_cell &c) const { return c.confidence; }
};
//! Picks color measurements out of the SDM.
struct dmPickColor : public dmPicker {
	//!functor - TSS_TODO
  virtual float operator() (dm_cell &c) const { return c.color; }
};
//! Picks cluster membership out of the SDM. For completeness--not use!
struct dmPickCluster : public dmPicker {
	//!functor - TSS_TODO
  virtual float operator() (dm_cell &c) const { return c.cluster; }
};
//@}
#endif
