/*
 * Configuration.h -- Contains configuration options for the AIBO Local Map
 *   and the AIBO Global Map
 *
 * Started on 12-13-2002
 */

#ifndef _ALM_CONFIGURATION_H_	// different symbol on purpose
#define _ALM_CONFIGURATION_H_

/* This macro converts degrees to radians */
#define R(item) ( ((double)(item))*M_PI/180 )


/* These constants affect the performance of the local map */
  // After each move, we tax the certainty of data that hasn't been
  // refreshed by a certain amount.
  //   Here's the tax on the spherical depth map (SDM) data
#define ALM_DM_TAX	0.8
  //   Here's the tax on the height map (HM) data
#define ALM_HM_TAX	0.9
  //   Here's the tax on the global map (GM) data
#define ALM_GM_TAX	0.9


/* Dimensions of the spherical depth map (SDM) */
  // The spherical depth map is bounded by tilt and pan angles, not
  // distances, for obvious reasons. When specifying these constants, it's
  // important to consider that they are likely restricted by the physical
  // bounds on head motion, and that the tilt dimensions don't really
  // correspond to head tilt angles but to vertical angles emanating from the
  // tilt pivot on the AIBO's chest. A permissible head tilt angle (e.g
  // facing straight downward) might not work when you just move it to
  // the chest (because the robot can't look directly at the spot beneath
  // it's chest!), so be conservative.
  //   Angle units are DEGREES and are oriented and aligned the same way
  // as the head joint angles in the Tekkotsu software, Sony documentation,
  // etc.

  // Top of the SDM. A shallow angle is recommended because at some pan
  // angles, the head can't move as far back as usual.
#define ALM_DM_TOP	R(25.0)
  // Bottom of the SDM
#define ALM_DM_BOTTOM	R(-75.0)
  // Left side of the SDM
#define ALM_DM_LEFT	R(85.0)
  // Right side of the SDM
#define ALM_DM_RIGHT	R(-85.0)

  // Vertical size of the SDM array
//#define ALM_DM_V_SIZE	301
#define ALM_DM_V_SIZE	51
  // Horizontal size of the SDM array
//#define ALM_DM_H_SIZE	539
#define ALM_DM_H_SIZE	89
  // These numbers are both odd so there is a row of the array directly along
  // the altitude and azimuth zero angles.


/* Dimensions of the height map (HM) */
  // The height map is a horizontal circular disk emanating from the ground
  // point directly beneath the AIBO's tilt joint. While the information in
  // the height map is stored in a regular square array, only the disk is
  // guaranteed to be updated as the AIBO moves.

  // Radius of the HM in millimeters
//#define ALM_HM_RADIUS	400
#define ALM_HM_RADIUS	600
  // Radius of the HM in array elements(when the radius is aligned with the
  // axis of the array). Or, half the size of one of the sides of the array.
//#define ALM_HM_SIZE	20
#define ALM_HM_SIZE	40


/* Dimensions of the global map (GM) */
  // The global map is a horizontal rectangle--it's an allocentric version
  // of the HM described above. Coordinates are the same as those used by
  // FastSLAM, so usage strategies should reflect the way FastSLAM is use
  // in your application.
  //   If you're using FastSLAM as glorified Monte Carlo Localization and
  // know the positions of all your landmarks, it's probably safe to simply
  // place the bounding box, in landmark relative coordinates, around the
  // area you expect to drive the AIBO. You should add some margins, but
  // not too many.
  //   If you're using FastSLAM to truly explore and discover landmarks,
  // you should make the GM big enough to enclose the entire area that
  // AIBO will explore, understanding that in the course of mapping it may
  // rotate and translate the locations of itself and landmarks away
  // from the places you originally expected it to find them. Give the
  // GM wide margins and some space for rotation.
  //   TODO in the future: automatic GM configuration

  // Top Y coordinate of the GM in millimeters
#define AGM_TOP		2000
  // Bottom Y coordinate of the GM in millimeters
#define AGM_BOTTOM	-2000
  // Left X coordinate of the GM in millimeters
#define AGM_LEFT	-2000
  // Right X coordinate of the GM in millimeters
#define AGM_RIGHT	2000

  // Vertical (Y) size of the GM array
#define AGM_V_SIZE	100
  // Horizontal (Z) size of the GM array
#define AGM_H_SIZE	100


/* Final array sizing. */
  // Some defines for shorthand that define the size of the HM, DM, and GM
  // arrays. You should not need to change these
#define HM_CELL_COUNT   4*ALM_HM_SIZE*ALM_HM_SIZE
#define DM_CELL_COUNT   ALM_DM_V_SIZE*ALM_DM_H_SIZE
#define GM_CELL_COUNT   AGM_V_SIZE*AGM_H_SIZE


/* Number of clusters for K-means uncertainty clustering in all the maps */
  // In the DM
#define ALM_DM_NUMCLUSTERS	4
  // In the HM
#define ALM_HM_NUMCLUSTERS	4
  // In the GM
#define AGM_NUMCLUSTERS		4
/* Number of iterations for K-means uncertainty clustering in all the maps */
#define AM_KMEANS_ITERATIONS	5


/* The standard deviation, in DM pixels (bad measure, I know. TODO)
 * of IR measurements per millimeter of distance */
#define ALM_IR_SPLAT_STDDEV	3/600


/* Ground plane assumption. How confident are we about ground plane assumption
 * measurements? */
#define ALM_GPA_CONFIDENCE	0.7


/* Physical dimensions of the AIBO */
  // The distance from the tilt pivot in the neck to the pan/roll pivot in
  // the head (mm):
#define AIBO_NECK_HEIGHT	48.0
  // The distance from the pan/roll pivot in the head to the IR depth ranger:
#define AIBO_HEAD_LENGTH	66.6
  // The average distance from the ground to the tilt pivot point. Hard to tell
  // for certain without a whole motion model.
#define AIBO_TILT_PIVOT_HEIGHT	185.0
  // The horizontal FOV of of the AIBO's camera
#define AIBO_CAM_H_FOV		R(57.6)
  // The vertical FOV of of the AIBO's camera
#define AIBO_CAM_V_FOV		R(47.8)
  // The horizontal distance from the neck joint to the point where AIBO
  // pivots when it makes a turning walk (mm)
#define AIBO_TURN_PIVOT_DIST	90.0
  // The absolute lowest an overhang can be without impeding the movement
  // of the AIBO.
#define AIBO_MIN_CLEARANCE	300
  // The absolute highest a bump on the ground can be without impeding
  // AIBO movement.
#define AIBO_MAX_BUMP		10
  // These constants are calibration constants for the IR sensor. They
  // should probably end up in a config file soon.
  // The AIBO's IR range measurements deviate linearly from the true
  // depths of objects they measure. Calibrate your AIBO with these
  // parameters, where
  //   true depth = AIBO_IR_CAL_MULTIPLIER * reported depth +
  //		    AIBO_IR_CAL_OFFSET
  // such that true depth reflects the true distance from the nose of
  // the AIBO to the object. You'll have to use a program or calculator
  // that can do linear regression on the data you collect--the TI-8x
  // series calculators and Matlab are all capable of this.
#define AIBO_IR_CAL_MULTIPLIER	0.78807
#define AIBO_IR_CAL_OFFSET	-15.19878

  // More soon

#endif
