/* The Aibo FastSLAM test suite. */

#ifdef __cplusplus
extern "C" {
#endif

#include "afsMain.h"
#include "afsParticle.h"
#include "afsMotionResample.h"
#include "afsFindBestPose.h"
#include "afsUtility.h"
#include "Configuration.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

extern afsParticle *Particles;

afsXY landmarks[AFS_NUM_LANDMARKS];

/* JUST ONE ENORMOUS MAIN FUNCTION. Hooray!!! */
int main(int argc, char **argv)
{
  int i, iteration;
  double theta, dtheta, dist_x, dist_y;
  afsParticle p; /* the robot location */
  afsParticle *q; /* where we think we are */
  FILE *pfh, *lfh, *mfh, *sfh, *efh;

  srand48(time(NULL));	/* initialize random number generators */
  srand(time(NULL));

  /* open matlab file for errors */
  efh = fopen("err.mat", "w");
  if(!efh) {
    fputs("couldn't open err.mat\n", stderr);
    return -1;
  }

  /* open matlab file for particles */
  pfh = fopen("particle.m", "w");
  if(!pfh) {
    fputs("couldn't open particle.m\n", stderr);
    return -1;
  }

  /* open matlab file for landmark estimates and ellipses */
  lfh = fopen("estimate.m", "w");
  if(!lfh) {
    fputs("couldn't open estimate.m\n", stderr);
    return -1;
  }

  /* open matlab file for measurements */
  mfh = fopen("measurement.m", "w");
  if(!mfh) {
    fputs("couldn't open measurement.m\n", stderr);
    return -1;
  }

  /* print headers for these three files */
  fputs("function particle(i)\nhold on\nswitch i\n", pfh);
  fputs("function estimate(i)\nhold on\nswitch i\n", lfh);
  fputs("function measurement(i)\nhold on\nswitch i\n", mfh);



  /* Randomly distribute landmarks over an 8-meter square */
  for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
    landmarks[i].x = (2*drand48() - 1) * 4000.0; /* we measure in mm */
    landmarks[i].y = (2*drand48() - 1) * 4000.0;
  }

  /* open matlab file for the basic scene */
  sfh = fopen("scene.m", "w");
  if(!sfh) {
    fputs("couldn't open scene.m\n", stderr);
    return -1;
  }

  /* Print landmarks out for Matlab */
  fputs("hold off\n", sfh);
  fprintf(sfh, "plot(%f,%f,'k*')\n", landmarks[0].x, landmarks[0].y);
  fputs("hold on\n", sfh);
  for(i=1; i<AFS_NUM_LANDMARKS; ++i)
    fprintf(sfh, "plot(%f,%f,'k*')\n", landmarks[i].x, landmarks[i].y);

  fputs("axis([-6000 6000 -6000 6000])\n", sfh);
  fclose(sfh);



  /* ignition! */
  afsInit();

  /* The real robot starts at 0,0,0. */
  afsParticleInit(&p);

  /* See if the user wanted to test landmark location initialization
   * with afsSetLandmark and distributed start with afsDistribute.
   * We'll call this "Monte Carlo Localization mode" or "mclmode" for
   * short */
  if((argc > 1) && !strcmp(argv[1], "mclmode")) {
    puts("Operating in Monte Carlo Localization mode");
    /* preset landmark guesses */
    for(i=0; i<AFS_NUM_LANDMARKS; ++i)
      afsSetLandmark(i, landmarks[i].x, landmarks[i].y, 500);
    /* distribute particle guesses all over */
    afsDistribute(-6000, 6000, 6000, -6000);
    /* oh, and make our location tough to guess, too--not 0,0 at first. */
    p.pose.x = (2*drand48() - 1) * 4000.0;
    p.pose.y = (2*drand48() - 1) * 4000.0;
  }

  /* now have the robot run around in circles forever, and print out the
   * global certainty value at each step */
  for(iteration=0; iteration<50; ++iteration) {
    afsParticle work_p; /* handy scratchpad particle */
    afsPose work_pose;  /* handy scratchpad pose */

    fprintf(pfh, "case %d\n", iteration);
    fprintf(lfh, "case %d\n", iteration);
    fprintf(mfh, "case %d\n", iteration);

    /* First we move ourselves. We try to hover around the point 1,1, but
     * we can only make very shallow turns. */
    {
      double goal_theta = atan2(1 - p.pose.y, 1 - p.pose.x);
      /* make robot relative */
      goal_theta -= p.pose.theta;
      /* constrain to [0, 2pi) */
      while(goal_theta < 0) goal_theta += 2*M_PI;
      while(goal_theta >= 2*M_PI) goal_theta -= 2*M_PI;

      if(goal_theta < M_PI) dtheta = 0.03;
      else dtheta = -0.03;
    }

    /* move the 'real' particle */
    afsMotionResample(&p, 70, 0, dtheta, 20000);
    /* Now we move the particles. */
    afsMotion(70, 0, dtheta, 20000);

    /* Now we take sensor measurements. Some of them we'll miss this time
     * around. */
    for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
      if(drand48() < 0.4) continue;

      /* compute bearing to landmark */
      dist_x = landmarks[i].x - p.pose.x;
      dist_y = landmarks[i].y - p.pose.y;
      theta = atan2(dist_y, dist_x);
      /* add measurement error */
      theta += normRand()*AFS_MEASURE_VARIANCE;
      /* show measurement in matlab */
      fprintf(mfh, "plot_line(%f,%f,%f,%f)\n", p.pose.x, p.pose.y,
			//sqrt(dist_x*dist_x + dist_y*dist_y), theta);
			1200.0, theta);
      /* make bearing relative to robot pose */
      theta = find_dtheta(p.pose.theta, theta);

      /* register landmark observation */
      afsMeasurement(i, theta);
      afsResample();
    }

    /* what have we found? */
    /* first, plot all particle xy locations in yellow. */
    for(i=0; i<AFS_NUM_PARTICLES; ++i) {
      /* fit it to our map of the world */
      work_pose = afsFindBestPose(&Particles[i], p.pose, landmarks);
      work_p = Particles[i];
      afsApplyBestPose(&Particles[i], &work_p, work_pose);

      fprintf(pfh, "%% %f %f %f\n", work_pose.x, work_pose.y, work_pose.theta);

      fprintf(pfh, "plot(%f, %f, 'y+')\n", work_p.pose.x,
					  work_p.pose.y);
    }
    /* now plot where we *should* be */
    fprintf(pfh, "plot(%f,%f,'r+')\n", p.pose.x, p.pose.y);

    /* now find and plot our best location guess */
    q = afsWhatsUp(NULL);

    work_pose = afsFindBestPose(q, p.pose, landmarks);
    afsApplyBestPose(q, &work_p, work_pose);
    //work_p = *q;
    fprintf(pfh, "plot(%f,%f,'g+')\n", work_p.pose.x, work_p.pose.y);

    /*
    for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
      if(q->landmarks[i].state == PRIMING) continue;

      fprintf(fh, "plot(%f,%f,'c*')\n", q->landmarks[i].mean.x,
					q->landmarks[i].mean.y);
    }
    */

    for(i=0; i<AFS_NUM_LANDMARKS; ++i) {
      if(work_p.landmarks[i].state == PRIMING) continue;

      fprintf(lfh, "plot(%f,%f,'b*')\n", work_p.landmarks[i].mean.x,
					work_p.landmarks[i].mean.y);
      fprintf(lfh, "plot_ellipse([%f %f; %f %f], %f, %f)\n",
			work_p.landmarks[i].variance.x,
			work_p.landmarks[i].variance.xy,
			work_p.landmarks[i].variance.xy,
			work_p.landmarks[i].variance.y,
			work_p.landmarks[i].mean.x,
			work_p.landmarks[i].mean.y);
    }

    {
      double error = afsCertainty();
      fprintf(stderr, "Iter. %d Certainty: %f\n", iteration, error);
      fprintf(efh, "%f\n", error);
    }
  }


  /* close up MATLAB files */
  fputs("end\naxis([-6000 6000 -6000 6000])\n", pfh);
  fputs("end\naxis([-6000 6000 -6000 6000])\n", lfh);
  fputs("end\naxis([-6000 6000 -6000 6000])\n", mfh);

  fclose(efh);
  fclose(pfh);
  fclose(lfh);
  fclose(mfh);

  return 0;
}

#ifdef __cplusplus
}
#endif
