#ifndef INCLUDED_CameraBehavior_h_
#define INCLUDED_CameraBehavior_h_

#include "Shared/WorldState.h"
#include "Events/EventRouter.h"
#include "Behaviors/BehaviorBase.h"
#include "Motion/MMAccessor.h"
#include "Motion/MotionManager.h"
#include "Motion/HeadPointerMC.h"
#include "Motion/TailWagMC.h"
#include "Motion/LedMC.h"
#include "Shared/SharedObject.h"

//! Will take images and write to log file
/*! Press the head button to take a picture, back button to write to memory
 *  stick.  This isn't necessarily up to date, but is included as sample code.
 *  We should have a way to save pictures to memstick instead of relying solely
 *  on having wireless to transmit them over. */
class CameraBehavior : public BehaviorBase {
 public:

	//! just sets up the variables
	CameraBehavior() :
    BehaviorBase(),
    camera_click(EventBase::buttonEGID,HeadFrButOffset,
        EventBase::deactivateETID,250),
	  sensor_update(EventBase::sensorEGID,SensorSourceID::UpdatedSID,
        EventBase::statusETID,0),
		headpointer_id(MotionManager::invalid_MC_ID),
    tailwag_id(MotionManager::invalid_MC_ID),
    led_id(MotionManager::invalid_MC_ID)
		{cout << "Camera created" << endl;}

	//! calls DoStop() if isActive()
	virtual ~CameraBehavior() { cout << "Camera destructed" << endl; if(isActive()) DoStop(); }

	//! Register for events and creates and adds two motion commands - a walker and a tail wag
	virtual void DoStart() {
		BehaviorBase::DoStart();
		//set up the shared motions
    SharedObject<HeadPointerMC> headpointer;
//    headpointer.MCBase()->setPriority(101);
		headpointer_id=motman->addMotion(headpointer);
		if(tailwag_id==MotionManager::invalid_MC_ID)
      tailwag_id=motman->addMotion(SharedObject<TailWagMC>());
    led_id=motman->addMotion(SharedObject<LedMC>());
		//register for events and timers
		erouter->addListener(this,camera_click);
		erouter->addListener(this,sensor_update);
    //disable tail servos for our own joystick
    motman->setPIDOff(TailOffset+TiltOffset);
    motman->setPIDOff(TailOffset+PanOffset);
		//prime the sensor update
		processEvent(sensor_update);
	}

	//! Removes its two motion commands
	virtual void DoStop() {
    //enable tail servos
    motman->setPIDDefault(TailOffset+TiltOffset);
    motman->setPIDDefault(TailOffset+PanOffset);
    //wag tail to indicate that we're doing something
    MMAccessor<TailWagMC>(tailwag_id).mc()->setActive(true);
    //initiate writing to log
		//CHANGE_ET commented out until ported
		/*    if (LogControlStream!=NULL) {
      LogControl control;
      control.command=LogControl::WriteLog;
      LogControlStream->writeBinary((const uchar *)&control,
          (int)sizeof(control));
					}*/
		//remove timers and listeners
		erouter->forgetListener(this);
		//remove motion commands, set them to invalid
		motman->removeMotion(headpointer_id);
		headpointer_id=MotionManager::invalid_MC_ID;
    motman->removeMotion(led_id);
    led_id=MotionManager::invalid_MC_ID;
//		motman->removeMotion(tailwag_id);
//		tailwag_id=MotionManager::invalid_MC_ID;
		BehaviorBase::DoStop();
	}

	//! Handles event processing
	/*! After every sensor update, set head in direction of tail */
	virtual void processEvent(const EventBase& e) {
    static int timer=0;
    if (timer==1) {
      MMAccessor<LedMC> led(led_id);
      led.mc()->clear();
    }
    if (e==sensor_update) {
      if (timer>0) timer--;
      double tilt=state->outputs[TailOffset+TiltOffset]/RAD(22);
      double pan=state->outputs[TailOffset+PanOffset]/RAD(22);
      cout << tilt << " " << pan << endl;
			//CHANGE_ET now using outputRanges from RobotInfo.h instead of head_tilt_min & etc.
      MMAccessor<HeadPointerMC>(headpointer_id).mc()->setJoints(
          .5*(outputRanges[HeadOffset+TiltOffset][0]+outputRanges[HeadOffset+TiltOffset][1]+(outputRanges[HeadOffset+TiltOffset][1]-outputRanges[HeadOffset+TiltOffset][0])*tilt),
          .5*(outputRanges[HeadOffset+PanOffset][0]+outputRanges[HeadOffset+PanOffset][1]+(outputRanges[HeadOffset+PanOffset][1]-outputRanges[HeadOffset+PanOffset][0])*pan), 0.0);
    } else if (e.equalOrLongerThan(camera_click)) {
      MMAccessor<LedMC> led(led_id);
      led.mc()->cycle(FaceLEDMask,250,1.0);
			//CHANGE_ET commented out until ported
			//      VisionInterface::SendRawImage(vision);
			//      VisionInterface::SendRLEImage(vision);
      timer=30;
    }
	}
	virtual std::string getName() const { return "CameraBehavior"; } //!< returns name of behavior
	virtual static std::string getClassDescription() { return "Point head with the tail and push a head button to save a picture"; }
 protected:
	inline double RAD(double x) { return x/180*M_PI; } //!< converts @a x degrees to radians

	const EventBase camera_click; //!< event mask for taking a picture (head button)
	const EventBase sensor_update; //!< event mask for sensor update
	MotionManager::MC_ID headpointer_id; //!< MC_ID for head pointer
	MotionManager::MC_ID tailwag_id;     //!< MC_ID for tail wag
  MotionManager::MC_ID led_id;         //!< MC_ID for leds
};

/*! @file
 * @brief Defines CameraBehavior, for taking pictures
 * @author alokl (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_4 $
 * $Revision: 1.4 $
 * $State: Exp $
 * $Date: 2003/06/12 23:41:39 $
 */

#endif
