/*
 * Tekkotsu framework class for the Second Generation World Model.
 * This class is the glue that connects all of the representations in
 * the 2GWM with the rest of Tekkotsu. It implements EventListener so it
 * can receive motion and video events; it maintains a queue of actions
 * that the AIBO can elect to perform to improve the certainty of the
 * world representations; and it also has access functions that allow
 * other code to get at the contents of the world model.
 *
 * This code checks for the definition of WANT_GLOBAL_MAP. If WANT_GLOBAL_MAP
 * is enabled, then global mapping functionality (FastSLAM, the global height
 * map) is compiled in. If not, these parts are omitted.
 */

#ifndef _WORLD_MODEL_2_H_
#define _WORLD_MODEL_2_H_

#include <vector>
#include <deque>
#include <math.h>

#include "Events/EventListener.h"
// for AFS_NUM_LANDMARKS
#include "WorldModel2/FastSLAM/Configuration.h"
// for afsParticle
#include "WorldModel2/FastSLAM/afsParticle.h"
// for Picker objects
#include "WorldModel2/Maps/almStructures.h"
// for serializer stuff
#include "Shared/Serializer.h"
// for wireless
#include "Wireless/Wireless.h"

//! Symbols for kludges that alter the behavior of WorldModel2 mapping

//! Various assumptions (kludges) you can enable or disable with enableKludge
//! and disableKludge to make the behavior of the maps appear better
namespace WM2Kludge {
  //! Don't incorporate measurements of items situated under the ground plane

  //! If your environment has no pits in it, there will be nothing that has
  //! a z value smaller than 0. Measurements which have such z values are surely
  //! noise and can be ignored.
  const unsigned int IgnoreZLessThanZero = 1;
  //! Don't incorporate measurements of green items in the map

  //! Our environment has green matting that uniquely indicates that we're
  //! looking at the ground. Since we can already guess that the ground is
  //! flat, we don't need to waste time on measurements of it.
  const unsigned int IgnoreGreenItems	 = 2;
  //! Delay passing of movement information to FastSLAM

  //! With lots of particles, FastSLAM takes forever to do motion resamples--too
  //! long for the robot to remain responsive if it's just doing a course correction
  //! while en route. With this kludge enabled, motion resampling will wait until the
  //! AIBO is completely stopped; then all the motions will be applied en masse.
  //! This will still have an impact on state machine timing, but at least it keeps
  //! AIBO from running into a wall. Be warned: you will not know where you are
  //! with LazyFastSLAM enabled until you stop moving and FastSLAM has had an
  //! opportunity to catch up with itself!
  const unsigned int LazyFastSLAM	 = 4;
}


//! Structure for containing motion requests.
//  I wish there were a more clean way to do this
typedef struct {
    //! Is this motion request from the DM, the HM, or the GM?
  enum { LOOK_AT, LOOK_DOWN_AT, GO_TO } type;
    // Hang on to your hats, it's a union!
    //! Depending on which it is, we use different data.
  union {
      //! Used for LOOK_AT, the requests made by the DM
    struct {
			double azimuth;  //!< angle from straight ahead (bearing)
			double altitude; //!< height relative to ? TSS_TODO
		} azalt;
      //! Used for LOOK_DOWN_AT and GO_TO, the requests made by
      //! the HM and the GM.
    struct {
			double x; //!< x
			double y; //!< y
		} xy;
  };
} MotionRequest;


//! Structure for keeping track of FastSLAM system updates for eventual
//! evaluation. See the documentation on the LazyFastSLAM kludge to see
//! why we need to keep this data around at times.
typedef struct _FastSLAM_update {
  enum { MOTION, LANDMARK } type; //!< the type of record
  double dx; //!< x distance
	double dy; //!< y distance
	double da; //!< angular distance
  long time; //!< timestamp
} FastSLAM_update;


//! A vector for motion requests
typedef std::vector<MotionRequest> MRvector;

//! A deque for FastSLAM updates
typedef std::deque<FastSLAM_update> FSUdeque;

// The WorldModel2 class itself
//! Tekkotsu's localization and mapping system <b>(IN ACTIVE DEVELOPMENT)</b>

//! WorldModel2 is a system of egocentric and allocentric maps designed to
//! represent the world around the AIBO. Presently, WorldModel2 is composed
//! of <b>beta</b> and <b>alpha</b> software, with most egocentric
//! functionality in beta and everything else in alpha.
//! <p>
//! The bulk of the alpha software in WorldModel2 is involved in global
//! mapping and localization. This software is disabled by default and
//! therefore escapes documentation by Doxygen. For more information on
//! these routines, read WorldModel2.h; to give them a try (hardly advisable)
//! set the environment variable GLOBAL_MAP to -DWANT_GLOBAL_MAP before
//! compiling.
class WorldModel2 : public EventListener, public Serializer {
  public:

      //! Constructor
    WorldModel2();
      //! Destructor
    virtual ~WorldModel2();

      //! Resets the world model to the start state

      //! Clears all maps ("empty" maps will contain an assumed ground plane
      //! at elevation 0) and, if allocentric functionality is enabled, resets
      //! the localization system to its initial zero-knowledge state.
    void resetWorldModel();
 
      //! @name Sensor toggles
      //! Enable or disable certain sensing mechanisms, which are off by default
      //! Disabling sensor mechanisms is useful when you have other CPU
      //! intensive things to do and don't want to spend time updating the
      //! maps. Note that there is no way to disable processing of WalkMC
      //! movement--this would lead to gross inconsistencies in the map.
      //! Note that the mechanisms all start off blind--you have to turn
      //! them on yourself before they'll work.
      //@{
      //! Enable infrared data registration

      //! The AIBO continually takes depth measurements with the IR range
      //! sensor in its nose. Calling this function will cause WorldModel2
      //! to integrate these range measurements into its maps, along with
      //! segmented color data from the vision system for the measured point.
      //! Note that IR measurements are only integrated when the AIBO is
      //! standing still in the erect posture from ms/data/motion/stand.pos.
    void enableIR();
    void disableIR();	//!< Disable infrared data registration

      //! Enable ground plane assumption mapping (ALPHA)

      //! Our AIBO environment has a green floor. We can assume that all green
      //! items in the color segmented image from the vision system are parts
      //! of the floor. Using triangulation and the assumption that the floor
      //! is a flat plane at z=0, we can fill in parts of the map
      //! instantaniously. Note that ground plane assumption mapping only
      //! occurs when the AIBO is standing still in the erect posture from
      //! ms/data/motion/stand.pos and staring straight ahead (i.e. all head
      //! positioning attributes are 0).
    void enableGPA();
    void disableGPA();	//!< Disable ground plane assumption mapping (ALPHA)
      //@}

      //! Enable a kludge (see the WM2Kludge namespace for details)
    void enableKludge(unsigned int kludge);
      //! Disable a kludge (see the WM2Kludge namespace)
    void disableKludge(unsigned int kludge);

      //! Process vision and motion events
    virtual void processEvent(const EventBase& event);

      //! Get suggestions about what to look at next (ALPHA)

      //! All maps within WorldModel2 give each of their grid cells
      //! 'confidence' scores: measures of how certain the mapping system
      //! is about the data they contain. At the moment, these scores are
      //! mostly ad-hoc, but later they may be based on more statistically
      //! rigorous techniques. In any case, getRequests performs a simple
      //! k-means cluster analysis on the certainty measurements of the
      //! WorldModel2 maps (where K and the number of iterations are
      //! specified in WorldModel2's Configuration.h) and places the centers
      //! of these clusters in appropriate MotionRequest structures, which
      //! are inserted into the requests vector. Your code may choose to
      //! honor these suggestions, or it may not.
    void getRequests(MRvector &requests);

      //! @name Map data access routines
      //! Access routines for the SDM and HHM data. You have your
      //! choice of "safer" and "faster". First are the "safer" methods,
      //! which copies the requested data into arrays furnished by the user.
      //! They're not *that* safe, though--you're passing a pointer to
      //! your array, and you'd better make certain that your array has
      //! enough space for the data.<p>
      //!   You'll probably want to include the header file
      //! WorldModel2/Maps/Configuration.h to get the defines DM_CELL_COUNT,
      //! HM_CELL_COUNT, and GM_CELL_COUNT, which describe the number of
      //! cells in the maps. The same file also contains plenty of other
      //! constants describing the maps, which may be of use to you.<p>
      //!   Safe access routines choose the data you want using "Pickers",
      //! functors described and documented in
      //! WorldModel2/Maps/almStructures.h that you specify as the first
      //! argument. Unfortunately, all the data is encoded as float, but
      //! this shouldn't create many problems.<p>
      //!   Fast access routines simply return pointers to the active map
      //! data itself. <b>Please do not change anything unless you know what
      //! you are doing!</b>
      //@{
      //! The safe access method for spherical depth map data.
    void pickDMData(dmPicker &p, float *dest);
      //! The fast access method for spherical depth map data. Use at own risk.
    dm_cell *invadeDMData(void);

      //! Safe access method for horizontal height map. Do see long description.

      //! You'll want to use the HM_CELL_COUNT define in
      //! WorldModel2/Maps/Configuration.h for sizing. Do remember that
      //! the HHM is *round* and is not guaranteed to maintain all the data
      //! in its array--only that within the circle of cells within
      //! ALM_HM_SIZE (Euclidean) of the central cell of the array.
    void pickHMData(hmPicker &p, float *dest);
      //! Risky fast access method for horiz. height map. See pickHMData note.
    hm_cell *invadeHMData(void);
      //@}

      // Here are public extensions to the class that are added when global
      // map features are desired.
#ifdef WANT_GLOBAL_MAP
      //! Specify the position of a landmark (for FastSLAM) (ALPHA)

      //! Recommended for use BEFORE any measurements have been made--either
      //! build a map from scratch with FastSLAM or specify a map and have
      //! FastSLAM navigate around in it.<p>
      //! If you use this function, you will probably want to employ the
      //! fsDistribute as well, effectively turning FastSLAM into Monte Carlo
      //! Localizaton
    void setLandmark(int landmark, double x, double y, double covariance);

      //! Random FastSLAM initialization within bounding box. (ALPHA)

      //! Distribute FastSLAM particles randomly (uniformly) throughout
      //! a bounding box. Recommended for use AFTER all landmarks have been
      //! specified with setLandmark (above) and BEFORE any measurements
      //! have been made. Use in this way effectively reduces FastSLAM to
      //! Monte Carlo Localization.<p>
      //! The tighter the bounding box, the better the starting performance.
      //! lx = left x, ty = top y, rx = right x, bottom y = by
    void fsDistribute(double lx, double ty, double rx, double by);

      // \addtogroup Sensor toggles
      //@{
      //! Enable the detection of markers (used by FastSLAM) (ALPHA)
    void enableMarkers();
      //! Disable marker detection (used by FastSLAM) (ALPHA)
    void disableMarkers();
      //@}

      //! Gets the current best FastSLAM map (ALPHA)

      //! Places the contents of the map in p. The map contains not only
      //! estimated landmark positions but also the current best guess of
      //! the robot's position and heading.
    void getFastSLAMMap(afsParticle &p);

      // \addtogroup Map data access routines
      //@{
      //! Safe access method for global map data.
    void pickGMData(hmPicker &p, float *dest);
      //! Risky fast access method for global map data.
    hm_cell *invadeGMData(void);
      //@}
#endif

  private:
      //! Constant indicates how frequently head position sampling should take
      //! place for the ground plane assumption. Units are milliseconds.
    static const int GPAdelay = 100;
      //! Constant indicates how frequently serialization of map data should
      //! be performed. Units are milliseconds.
    static const int SRLdelay = 1000;

      //! The AIBO's memory management system is borked. Consequently, all
      //! memory space is allocated statically and can only be used by one
      //! WorldModel2 at a time. This variable indicates whether there's
      //! already someone using the data.
    static bool locked;

      //! Which kludges are enabled? See WM2Kludge documentation for info.
    unsigned int kludges;

      //! This boolean is set to true if the system isn't certain that AIBO
      //! is stopped. In other words, in spite of its name, it may be true
      //! even when the AIBO isn't moving, e.g. during emergency stop.
      //! If it's false, though, the AIBO should surely be at rest.
    bool moving;

      // These booleans keep track of what sensing mechanisms are enabled.
    bool enabledIR;	//!< True if IR sensing is enabled (c.f. enableIR)
    bool enabledGPA;	//!< True if GPA mapping is enabled (c.f. enableGPA)

      //! The current velocities of the AIBO.
    double dx, dy, da;
      //! The time when we started going at current velocities. 0 is a special
      //! value for movingSince, indicating that no information on movement
      //! has been given to this WorldModel object yet. Routines that use
      //! this motion information should check for the 0 value and abend if
      //! they find it.
    long movingSince;

      //! Here are sockets for serialization of the egocentric map data
    Socket *sockDM, *sockHM;

      //! process sensors, incorporating IRs into local maps
    void processSensors();
      //! process locomotion events--movements
    void processLocomotion(const EventBase& event);
      //! Try to do ground plane asssumption
    void processGround();

      //! Perform serialization -- send data down the line, if needed.
    void serialize();

      // Here are private extensions to the class added when global map
      // features are desired
#ifdef WANT_GLOBAL_MAP
      //! Here are sockets for serialization of the global map data.
    Socket *sockGM, *sockFS;

      //! Constant for the max angle from center of AIBO's FOV in which
      //! we'll actually pay attention to viewed landmarks. This is a way
      //! of avoiding problemes with distortion at the edges of the
      //! image. Must be in radians.
    static const float maxMarkerAngle = 30*M_PI/180;
      //! How many markers do we need to see before we do a FastSLAM resampling?
    static const int minMarkersForFastSLAM = 2;

      //@name Sensor processing checklist flags.
      //@{
      //! After each traverse, there's a list of things we want to do. Once
      //! we've done them, we don't want to do them again until we've moved
    bool haveSeenMarker[AFS_NUM_LANDMARKS];

      //! Keeps track of whether marker sensing is enabled (ALPHA)
    bool enabledMarkers; // marker sensing for FastSLAM
      //@}

      //! Pending updates for FastSLAM. See note on LazyFastSLAM in Kludges
     FSUdeque fs_updates;

      //! process vision events, specifically markers (ALPHA)
    void processVision(const EventBase& event);
#endif

      //! dummy functions prevent warnings
    WorldModel2(const WorldModel2&);
      //! dummy functions prevent warnings
    WorldModel2& operator= (const WorldModel2&);
};

  //! Determine whether Aibo is in an erect stature, allowing us to do
  //! measurements. TODO: replace with a real motion model
bool aiboIsErect();

  //! Determine whether Aibo is looking dead ahead. Needed for ground plane
  //! assumption, for the moment.
bool aiboStaresDeadAhead();

  //! Determine whether Aibo is keeping its head level (i.e. no tilt or roll).
  //! Needed for FastSLAM at the moment.
bool aiboIsLevelHeaded();

#endif
