/*
 * Main interface to the AIBO FastSLAM.
 */

#ifdef __cplusplus
extern "C" {
#endif

#include "afsParticle.h"
#include "afsMotionResample.h"
#include "afsMeasurementUpdate.h"
/* #include "afsFindBestPose.h" */

#include "Configuration.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* Globals mean never having to say malloc. ;-) */
afsParticle ParticleSets[2][AFS_NUM_PARTICLES];
afsParticle *Particles, *newParticles;
double Weights[AFS_NUM_PARTICLES];

/* Initialize everything */
void afsInit()
{
  int i;

  Particles = ParticleSets[0];
  newParticles = ParticleSets[1];
  for(i=0; i<AFS_NUM_PARTICLES; ++i) afsParticleInit(&Particles[i]);
}

/* Report a movement */
void afsMotion(double dx, double dy, double da, unsigned int time)
{
  /* If there was no motion at all, just don't do anything. */
  if((dx == 0) && (dy == 0) && (da == 0)) return;

/* Rescale motion parameters to reflect actual AIBO motion */
#include "motionReshapeKludge.h"

  /* Likewise don't do anything if there was no time spent moving */
  if(time == 0) return;

  int i;
  for(i=0; i<AFS_NUM_PARTICLES; ++i)
    afsMotionResample(&Particles[i], dx, dy, da, time);
}

/* Manually set a landmark location estimate. landmark is the landmark
 * to set, x and y are the location, and covariance is the spherical
 * covariance to set about the landmark. You should probably set it to
 * something small, like 500. */
void afsSetLandmark(int landmark, double x, double y, double covariance)
{
  int i;
  for(i=0; i<AFS_NUM_PARTICLES; ++i) {
    afsLandmarkLoc *l = &Particles[i].landmarks[landmark];

    l->mean.x = x;
    l->mean.y = y;
    l->variance.x = l->variance.y = covariance;
    l->variance.xy = 0.0;
    l->state = PRIMED;
  }
}

/* Distribute the particle robot location estimates uniformly throughout
 * the specified bounding box. Recommended for use immediately after
 * afsSetLandmark is used to set all landmark locations and before any
 * measurement updates has taken place. The tighter the bounding box, 
 * the better the performance. This function with the prior function 
 * essentially reduces FastSLAM to Monte Carlo Localization. */
/* lx = left x, ty = top y, rx = right x, by = bottom y */
void afsDistribute(double lx, double ty, double rx, double by)
{
  double dx = rx - lx;
  double dy = ty - by;
  int i;

  for(i=0; i<AFS_NUM_PARTICLES; ++i) {
    Particles[i].pose.x = lx + (double) rand() * dx / (double) RAND_MAX;
    Particles[i].pose.y = by + (double) rand() * dy / (double) RAND_MAX;
    Particles[i].pose.theta = (double) rand() * 2.0*M_PI / (double) RAND_MAX;
  }
}

/* Report a sensor measurement for each particle */
void afsMeasurement(int landmark, double theta)
{
  int i;

  for(i=0; i<AFS_NUM_PARTICLES; ++i) 
    afsMeasurementUpdate(&Particles[i], landmark, theta);
}

/* Resample all the particles based on the last sensor measurement. */
void afsResample()
{
  int i, num_weighted=0;
  double weight_sum=0.0, prior_weight=0.0, surrogate_weight;

  double current_wt_point, wt_increment;
  int current_wt_index = 0;

  /* Get the weights for all the particles from the last measurement update */
  for(i=0; i<AFS_NUM_PARTICLES; ++i)
    if(Particles[i].gotweight) {
      ++num_weighted;
      weight_sum += Particles[i].weight;
      Weights[i] = Particles[i].weight;
    }
    /* In this case, the landmark location has not been triangulated yet,
     * so we don't get a weight back. */
    else Weights[i] = -1; /* negative means no weight yet. */

  /* Now we come up with a weight for particles that weren't weighted. It's
   * just the mean of the other particle weights, or, if no particle is
   * weighted, it's 1/AFS_NUM_PARTICLES */
  surrogate_weight = num_weighted ? weight_sum / (double) num_weighted
				  : 1 / (double) AFS_NUM_PARTICLES;

  /* Now how much total weight do we have? */
  weight_sum += surrogate_weight * (double) (AFS_NUM_PARTICLES - num_weighted);

  /* Now, in one fell blow, we enter in all the substitute weights
   * and make them a cumulative tally (so we can sample proportionally). */
  for(i=0; i<AFS_NUM_PARTICLES; ++i)
    if(Weights[i] >= 0.0)
      prior_weight = Weights[i] = Weights[i] + prior_weight;
    else
      prior_weight = Weights[i] = surrogate_weight + prior_weight;

  /* We resample all the particles according to the low variance resampling
   * algorithm that Mike told me about. You randomly select a starting point
   * somewhere in [0 weight_sum) and do this:
   * 1.  Choose particle corresponding to current point
   * 2.  Increment current point by weight_sum/AFS_NUM_PARTICLES. If it gets
   *     bigger than weight_sum, subtract weight_sum. GOTO 1 unless we've
   *     sampled enough already!
   * The exciting part is that this resamples in O(N) and is somehow, according
   * to Mike, statistically purer. Don't ask me! */
  /* Choose initial current point */
  current_wt_point = (double) rand() * weight_sum / (double) RAND_MAX;
  /* Choose weight increment */
  wt_increment = weight_sum / (double) AFS_NUM_PARTICLES;

  /* Now we resample all the particles. */
  for(i=0; i<AFS_NUM_PARTICLES; ++i) {
    /* Find out which particle corresponds to the current wt point */
    /* current_wt_index initialized at declaration */
    while(current_wt_point > Weights[current_wt_index])
      /* this code handles wraparound */
      if(++current_wt_index >= AFS_NUM_PARTICLES) {
	current_wt_index = 0;
        current_wt_point = fmod(current_wt_point, weight_sum);
      }
    /* copy the particle into the new particle set */
    afsParticleCopy(&Particles[current_wt_index], &newParticles[i]);
    /* increment current weight point */
    current_wt_point += wt_increment;
    /* Set weight of new particles to 1.0, to allow for multiplicative
     * weighting later. See afsMeasurementUpdate for more details. */
    newParticles[i].weight = 1.0;
  }

  /* Make the new particles the current particles and recycle the space for
   * the old particles. */
  {
    afsParticle* temp = Particles;
    Particles = newParticles;
    newParticles = temp;
  }

  /* that's it! */
}

/* Scratch space for afsWhatsUp */
//static afsXY rectified_locations[AFS_NUM_PARTICLES];

/* Get the particle that is most representative of where we are and what the
 * world looks like. This particle may be synthesized or it may be taken
 * directly out of the particle list. Don't modify it, please. The error for
 * this particle will be put into the variable pointed to by error, so long
 * as the address isn't NULL. */
afsParticle *afsWhatsUp(double *error)
{
  /* Simply averages the landmark locations and the robot headings of all
   * the particles in the filter. Apparently this is the right thing to
   * do according to a conversation with Mike Montemerlo--although the old
   * method commented out below, along with the map. It's certainly much
   * simpler and faster. */
  int i, j;
  static afsParticle avg;
  double theta_x, theta_y;

  /* Initialize the particle to be all zeroes. We have to set the landmark
   * locations explicitly to zero, since afsParticleInit uses warning
   * values instead. */
  afsParticleInit(&avg);
  for(j=0; j<AFS_NUM_LANDMARKS; ++j)
    avg.landmarks[j].mean.x = avg.landmarks[j].mean.x = 0.0;
  /* These variables are used for averaging the robot's bearing. We have
   * to average the x and y components of the bearing, since averaging
   * the bearing itself, a circular quantity between 0 and 2pi, would
   * produce nonsense */
  theta_x = theta_y = 0.0;

  /* Sum up all the robot pose information */
  for(i=0; i<AFS_NUM_PARTICLES; ++i) {
    avg.pose.x += Particles[i].pose.x;
    avg.pose.y += Particles[i].pose.y;

    theta_x += cos(Particles[i].pose.theta);
    theta_y += sin(Particles[i].pose.theta);
  }

  /* Division step of the pose averaging */
  avg.pose.x /= (double) AFS_NUM_PARTICLES;
  avg.pose.y /= (double) AFS_NUM_PARTICLES;
  theta_x /= AFS_NUM_PARTICLES;
  theta_y /= AFS_NUM_PARTICLES;
  /* Here we turn the angle components into a real angle */
  avg.pose.theta = atan2(theta_y, theta_x);
  if(avg.pose.theta < 0) avg.pose.theta += 2*M_PI;

  /* Now we work on averaging landmark positions. This is a bit trickier
   * since some particles will not have certain landmarks initialized, while
   * others will. */

  for(j=0; j<AFS_NUM_LANDMARKS; ++j) {
    int primed_count = 0;
    double mx=0.0, my=0.0;		/* mean of landmark gaussian means */
    double covx=0.0, covxy=0.0, covy=0.0;	/* similar stuff for cov. MX */
    double dcvx=0.0, dcvxy=0.0, dcvy=0.0;	/* similar stuff for cov. MX */

    /* Finding the mean of the gaussian means is simple--just averaging.
     * Finding the covariance matrix of the combined gaussian is harder. Here,
     * we add the summed covariances of all estimates (cov* variables) to
     * the summed outer products of differences between the mean of means
     * and the individual means themselves. */

    /* summation stage for mean averaging and covariance computation */
    for(i=0; i<AFS_NUM_PARTICLES; ++i) {
      afsLandmarkLoc *l = &(Particles[i].landmarks[j]);

      if(l->state == PRIMED) {
	mx += l->mean.x;
	my += l->mean.y;

	covx  += l->variance.x;
	covxy += l->variance.xy;
	covy  += l->variance.y;

	++primed_count;
      }
    }

    /* break if there are no particles that have this landmark PRIMED. No
     * need to do anything special, since landmark in avg is initialized
     * PRIMING */
    if(primed_count == 0) break;

    /* division stage for mean averaging */
    mx /= (double) primed_count;
    my /= (double) primed_count;

    /* summation stage for outer products of distances from mean of means */
    for(i=0; i<AFS_NUM_PARTICLES; ++i) {
      afsLandmarkLoc *l = &(Particles[i].landmarks[j]);

      if(l->state == PRIMED) {
	double dx = l->mean.x - mx;
	double dy = l->mean.y - my;

	dcvx  += dx * dx;
	dcvxy += dx * dy;
	dcvy  += dy * dy;
      }
    }

    /* Final consolidation of covariance bits */
    covx  = (covx  + dcvx ) / (double) primed_count;
    covxy = (covxy + dcvxy) / (double) primed_count;
    covy  = (covy  + dcvy ) / (double) primed_count;

    /* finally, set landmark location estimate in avg */
    avg.landmarks[j].state = PRIMED;
    avg.landmarks[j].mean.x = mx;
    avg.landmarks[j].mean.y = my;
    avg.landmarks[j].variance.x  = covx;
    avg.landmarks[j].variance.xy = covxy;
    avg.landmarks[j].variance.y  = covy;
  }

  /* This calculation is pretty bogus, but it might be useful. Just the
   * average distance from the X,Y pose of all particles to the X,Y pose
   * of the average particle */
  if(error) {
    double dist = 0.0;
    for(i=0; i<AFS_NUM_PARTICLES; ++i) {
      double dx = avg.pose.x - Particles[i].pose.x;
      double dy = avg.pose.y - Particles[i].pose.y;
      dist += sqrt(dx*dx + dy*dy);
    }
    *error = dist / (double) AFS_NUM_PARTICLES;
  }

  return &avg;

/* This old code finds out the state of the particle filter the old-fashioned
 * and complicated way, which I have since learned is incorrect. It finds
 * a sort of representative "average" particle and then determines which
 * particle in the filter is closest to this. */
#if 0
  int i;
  afsParticle *closest = NULL;
  double best_dist = HUGE_VAL;
  afsXY avg_location;

  /* REMOVE */
  double avg_dist = 0;

  afsXY guessmap[AFS_NUM_LANDMARKS];
  afsPose guesspose, best_pose;

  avg_location.x = avg_location.y = 0;

  /* We start by generating the most likely map of the environment and the
   * most likely pose from all of the particles */
  afsGuessState(Particles, &guesspose, guessmap);

  /* Now we find out which particle has the map most like this one */
  for(i=0; i<AFS_NUM_PARTICLES; ++i) {
    /* Transform particle i to best match the guessed map */
    best_pose = afsFindBestPose(&Particles[i], guesspose, guessmap);

    /* Cache this location in rectified_locations */
    /* Summing step of average location */
    avg_location.x += (rectified_locations[i].x = best_pose.x);
    avg_location.y += (rectified_locations[i].y = best_pose.y);
  }

  /* Division step of average location */
  avg_location.x /= (double) AFS_NUM_PARTICLES;
  avg_location.y /= (double) AFS_NUM_PARTICLES;

  /* Figure out which rectified particle has the robot closest to the mean
   * location */
  for(i=0; i<AFS_NUM_PARTICLES; ++i) {
    double xdist, ydist, dist;

    xdist = rectified_locations[i].x - avg_location.x;
    ydist = rectified_locations[i].y - avg_location.y;
    dist = sqrt(xdist*xdist + ydist*ydist);

    /* REMOVE */
    avg_dist += dist;

    if(dist < best_dist) {
      best_dist = dist;
      closest = &Particles[i];
    }
  }

  /* REMOVE */
  fprintf(stderr, "AVERAGE DISTANCE: %f\n", avg_dist/(double)AFS_NUM_PARTICLES);

  /* Compute error for this particle if the user wants it. */
  if(error) {
    afsParticle work_p;

    best_pose = afsFindBestPose(closest, guesspose, guessmap);
    afsApplyBestPose(closest, &work_p, best_pose);
    *error = afsParticleError(&work_p, guesspose, guessmap);
  }

  return closest;
#endif
}

/* Presently just returns the error estimate from afsWhatsUp. Old
 * functionality described below. */
/* Get an indication of how certain the FastSLAM algorithm is about its
 * location. The lower the number, the better. Particles with unprimed
 * landmarks will tend to lower the error, but with more priming, things
 * become more reasonable */
double afsCertainty()
{
  double error;
  afsWhatsUp(&error);
  return error;
#if 0
  int i;
  double total_error = 0;

  afsXY guessmap[AFS_NUM_LANDMARKS];
  afsPose guesspose;

  /* We start by generating the most likely map of the environment and the
   * most likely pose from all of the particles */
  afsGuessState(Particles, &guesspose, guessmap);

  /* Now we find out how close all the particles are to this map */
  for(i=0; i<AFS_NUM_PARTICLES; ++i) {
    afsPose best_pose;
    afsParticle work_p;

    /* Transform particle i to best match the guessed map */
    best_pose = afsFindBestPose(&Particles[i], guesspose, guessmap);
    afsApplyBestPose(&Particles[i], &work_p, best_pose);

    /* Compute error for this particle */
    total_error += afsParticleError(&work_p, guesspose, guessmap);
  }

  return log(total_error);
#endif
}

/* Get direct access to the current particle set. Not approved for general
 * use--this is only used by the WorldModel2 serializer to send particle
 * filter information data for display. Software that makes (illicit) use
 * of this routine should not count on the pointer it returns remaining
 * accurate after a resampling. */
afsParticle *afsInvadeFSData()
{
  return Particles;
}

#ifdef __cplusplus
}
#endif
