#ifndef __NETSTREAM_H__
#define __NETSTREAM_H__

#include <cstdio>
#include <cstring>

#include <iostream>
#include <iosfwd>
#include <streambuf.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include <string.h>
#include <string>

#define INVALID_SOCKET (~0)

using namespace std;

template <class charT, class traits>
class basic_netbuf : public streambuf {
  public:
  //  Types:
	typedef charT                     char_type;
	typedef typename traits::int_type int_type;
	typedef typename traits::pos_type pos_type;
	typedef typename traits::off_type off_type;
	typedef traits                    traits_type;
  //Constructors/Destructors:
	basic_netbuf();
	basic_netbuf(const IPaddr& addr);
	basic_netbuf(const IPaddr::ipnum_t& host, const IPaddr::ipport_t port);
	basic_netbuf(const IPaddr::const_ipname_t& host, const IPaddr::ipport_t port);
	basic_netbuf(size_t buf_in_size, size_t buf_out_size);
	virtual ~basic_netbuf();
	
	
	
  //basic_netbuf Functions:
  public:
	virtual bool			open(const IPaddr& addr);
	virtual bool			open(const IPaddr::const_ipname_t& str);
	virtual bool			open(const IPaddr::ipnum_t& addr, const IPaddr::ipport_t& aPort);
	virtual bool			open(const IPaddr::const_ipname_t& ahost, unsigned int aPort) { return open(IPaddr(ahost,aPort)); }
	virtual bool			is_open()	const { return (sock!=INVALID_SOCKET); }
	virtual void			close();

	virtual void			setEcho(bool val = true) { is_echoing = val; }
	virtual bool			getEcho() { return is_echoing; }
  protected:
    static void			printBuffer(const char* buf, int buflen, const char* header);
    void					Init() { Init(def_buf_in_size, def_buf_out_size); }
    void					Init(size_t insize, size_t outsize);

	
  //Inherited Functions:
  public:
	virtual void			in_sync(); //users shouldn't need to call this directly... but can if want to
	virtual void			out_flush();

  protected:
	//  lib.streambuf.virt.get Get area:
	virtual streamsize	showmanyc();
	//	virtual streamsize xsgetn(char_type* s, streamsize n);
	virtual int_type		underflow();
	virtual int_type		uflow();
	
	//  lib.streambuf.virt.pback Putback:
	//	virtual int_type pbackfail(int_type c = traits::eof() );
	//  lib.streambuf.virt.put Put area:
	//	virtual streamsize xsputn(const char_type* s, streamsize n);
	virtual int_type		overflow(int_type c  = traits::eof());
	
	//  lib.streambuf.virt.buffer Buffer management and positioning:
	//	virtual _Myt basic_netbuf<char_type, traits_type>* setbuf(char_type* s, streamsize n);
	//	virtual pos_type seekoff(off_type off, ios_base::seekdir way, ios_base::openmode which = ios_base::in | ios_base::out);
	//	virtual pos_type seekpos(pos_type sp, ios_base::openmode which = ios_base::in | ios_base::out);
	virtual int			sync();
	//  lib.streambuf.virt.locales Locales:
	//	virtual void imbue(const locale &loc);
  //Data:
  protected:
	charT *buf_in, *buf_out;
	bool using_buf_in, using_buf_out;
	static const size_t def_buf_in_size;
	static const size_t def_buf_out_size;
	int sock;
	bool is_echoing;
};
template <class charT, class traits>
const size_t basic_netbuf<charT,traits>::def_buf_in_size=1<<8;
template <class charT, class traits>
const size_t basic_netbuf<charT,traits>::def_buf_out_size=1<<12;


template <class charT, class traits>
class basic_iNetStream
	: public istream
{
public:
	typedef charT                     char_type;
	typedef typename traits::int_type int_type;
	typedef typename traits::pos_type pos_type;
	typedef typename traits::off_type off_type;
	typedef traits                    traits_type;
	//  lib.ifstream.cons Constructors:
	basic_iNetStream() : istream(&nb), nb() {}
	basic_iNetStream(const IPaddr::ipnum_t& host, const IPaddr::ipport_t port) : basic_iNetStream(&nb), nb(host,port) {}
	basic_iNetStream(const IPaddr::const_ipname_t& host, const IPaddr::ipport_t port) : basic_iNetStream(&nb), nb(host,port) {}
	basic_iNetStream(const IPaddr& addr) : istream(&nb), nb(addr) {}
	basic_iNetStream(size_t buf_in_size, size_t buf_out_size) : istream(&nb), nb(buf_in_size,buf_out_size) {}
	//  lib.ifstream.members Members:
	inline basic_netbuf<charT, traits>* rdbuf() const { return const_cast<basic_netbuf<charT, traits>*>(&nb); }
	
	inline bool			open(const IPaddr& addr) { return nb.open(addr); }
	inline bool			open(const IPaddr::const_ipname_t& str) { return nb.open(str); }
	inline bool			open(const IPaddr::ipnum_t& addr, const IPaddr::ipport_t& aPort) { return nb.open(addr,aPort); }
	inline bool			open(const IPaddr::const_ipname_t& ahost, unsigned int aPort) { return nb.open(ahost,aPort); }
	inline bool			is_open()	const { return nb.is_open(); }
/*	inline IPaddr::ipnum_t		remotehost() { return nb.remotehost(); }
	inline IPaddr::ipport_t		remoteport() { return nb.remoteport(); }
	inline IPaddr::ipnum_t		localhost() { return nb.localhost();}
	inline IPaddr::ipport_t		localport() { return nb.localport();}
*/
	inline void				close() { return nb.close(); }
	inline void				setEcho(bool val=true) { return nb.setEcho(val); }
	inline bool				getEcho() { return nb.getEcho(); }
private:
	basic_netbuf<charT, traits> nb;
};


template <class charT, class traits>
class basic_oNetStream
	: public ostream
{
public:
	typedef charT                     char_type;
	typedef typename traits::int_type int_type;
	typedef typename traits::pos_type pos_type;
	typedef typename traits::off_type off_type;
	typedef traits                    traits_type;
	//  lib.ifstream.cons Constructors:
	basic_oNetStream() : ostream(&nb), nb() {}
	basic_oNetStream(const IPaddr::ipnum_t& host, const IPaddr::ipport_t port) : basic_oNetStream(&nb), nb(host,port) {}
	basic_oNetStream(const IPaddr::const_ipname_t& host, const IPaddr::ipport_t port) : basic_oNetStream(&nb), nb(host,port) {}
	basic_oNetStream(const IPaddr& addr) : basic_oNetStream(&nb), nb(addr) {}
	basic_oNetStream(size_t buf_in_size, size_t buf_out_size) : ostream(&nb), nb(buf_in_size,buf_out_size) {}
	//  lib.ifstream.members Members:
	inline basic_netbuf<charT, traits>* rdbuf() const { return const_cast<basic_netbuf<charT, traits>*>(&nb); }
	
	inline bool			open(const IPaddr& addr) { return nb.open(addr); }
	inline bool			open(const IPaddr::const_ipname_t& str) { return nb.open(str); }
	inline bool			open(const IPaddr::ipnum_t& addr, const IPaddr::ipport_t& aPort) { return nb.open(addr,aPort); }
	inline bool			open(const IPaddr::const_ipname_t& ahost, unsigned int aPort) { return nb.open(ahost,aPort); }
	inline bool			is_open()	const { return nb.is_open(); }
/*	inline IPaddr::ipnum_t		remotehost() { return nb.remotehost(); }
	inline IPaddr::ipport_t		remoteport() { return nb.remoteport(); }
	inline IPaddr::ipnum_t		localhost() { return nb.localhost();}
	inline IPaddr::ipport_t		localport() { return nb.localport();}
*/
	inline void				close() { return nb.close(); }
	inline void				setEcho(bool val=true) { return nb.setEcho(val); }
	inline bool				getEcho() { return nb.getEcho(); }
private:
	basic_netbuf<charT, traits> nb;
};


template <class charT, class traits>
class basic_ioNetStream
	: public iostream
{
public:
	typedef charT                     char_type;
	typedef typename traits::int_type int_type;
	typedef typename traits::pos_type pos_type;
	typedef typename traits::off_type off_type;
	typedef traits                    traits_type;
	//  lib.ifstream.cons Constructors:
	basic_ioNetStream() : iostream(&nb), nb() {}
	basic_ioNetStream(const IPaddr& addr) : basic_ioNetStream(&nb), nb(addr) {}
	basic_ioNetStream(const IPaddr::ipnum_t& host, const IPaddr::ipport_t port) : basic_ioNetStream(&nb), nb(host,port) {}
	basic_ioNetStream(const IPaddr::const_ipname_t& host, const IPaddr::ipport_t port) : basic_ioNetStream(&nb), nb(host,port) {}
	basic_ioNetStream(size_t buf_in_size, size_t buf_out_size) : iostream(&nb), nb(buf_in_size,buf_out_size) {}
	//  lib.ifstream.members Members:
	inline basic_netbuf<charT, traits>* rdbuf() const { return const_cast<basic_netbuf<charT, traits>*>(&nb); }
	
	inline bool			open(const IPaddr& addr) { return nb.open(addr); }
	inline bool			open(const IPaddr::const_ipname_t& str) { return nb.open(str); }
	inline bool			open(const IPaddr::ipnum_t& addr, const IPaddr::ipport_t& aPort) { return nb.open(addr,aPort); }
	inline bool			open(const IPaddr::const_ipname_t& ahost, unsigned int aPort) { return nb.open(ahost,aPort); }
	inline bool			is_open()	{ return nb.is_open(); }
/*	inline IPaddr::ipnum_t		remotehost() { return nb.remotehost(); }
	inline IPaddr::ipport_t		remoteport() { return nb.remoteport(); }
	inline IPaddr::ipnum_t		localhost() { return nb.localhost();}
	inline IPaddr::ipport_t		localport() { return nb.localport();}
*/
	inline void				close() { nb.close(); }
	inline void				setEcho(bool val=true) { return nb.setEcho(val); }
	inline bool				getEcho() { return nb.getEcho(); }
private:
	basic_netbuf<charT, traits> nb;
};

template<class T>
class char_traits {
public:
  typedef T char_type;
  typedef int int_type;
  typedef T* pos_type;
  typedef unsigned int off_type;
  static void copy(pos_type dst, pos_type src, off_type size) {
    memcpy(dst,src,size);
  }
  static void move(pos_type dst, pos_type src, off_type size) {
    memmove(dst,src,size);
  }
  static int to_int_type(T c) { return c; }
  static int eof() { return EOF; }
};

typedef basic_netbuf<char, char_traits<char> > netbuf;
typedef basic_iNetStream<char, char_traits<char> > iNetStream;
typedef basic_oNetStream<char, char_traits<char> > oNetStream;
typedef basic_ioNetStream<char, char_traits<char> > ioNetStream;




template <class charT, class traits>
basic_netbuf<charT,traits>::basic_netbuf() 
  : buf_in(NULL), buf_out(NULL), using_buf_in(false), using_buf_out(false),
	sock(INVALID_SOCKET), is_echoing(false) {
  Init();
}
template <class charT, class traits>
basic_netbuf<charT,traits>::basic_netbuf(const IPaddr& addr)
  : buf_in(NULL), buf_out(NULL), using_buf_in(false), using_buf_out(false),
	sock(INVALID_SOCKET), is_echoing(false) {
  Init();
  open(addr);
}

template <class charT, class traits>
basic_netbuf<charT,traits>::basic_netbuf(const IPaddr::ipnum_t& host, const IPaddr::ipport_t port)
  : buf_in(NULL), buf_out(NULL), using_buf_in(false), using_buf_out(false),
	sock(INVALID_SOCKET), is_echoing(false) {
  Init();
  open(host,port);
}

template <class charT, class traits>
basic_netbuf<charT,traits>::basic_netbuf(const IPaddr::const_ipname_t& host, const IPaddr::ipport_t port)
  : buf_in(NULL), buf_out(NULL), using_buf_in(false), using_buf_out(false),
	sock(INVALID_SOCKET), is_echoing(false) {
  Init();
  open(host,port);
}

template <class charT, class traits>
basic_netbuf<charT,traits>::basic_netbuf(size_t buf_in_size, size_t buf_out_size) 
  : buf_in(NULL), buf_out(NULL), using_buf_in(false), using_buf_out(false),
	sock(INVALID_SOCKET), is_echoing(false) {
  Init();
}

template <class charT, class traits>
basic_netbuf<charT,traits>::~basic_netbuf() {
	if(using_buf_in)
		delete [] buf_in;
	if(using_buf_out)
		delete [] buf_out;
}

template<class charT, class traits>
void
basic_netbuf<charT,traits>::Init(size_t insize, size_t outsize) {
  buf_in = new charT[insize];
  buf_out = new charT[outsize];
  using_buf_in = using_buf_out = true;
  setg(buf_in,buf_in+insize,buf_in+insize);
  setp(buf_out,buf_out+outsize);
//  cout << "Buffer is:" << endl;
//  cout << "Input buffer: " << egptr() << " to " << egptr() << " at " << gptr() << endl;
//  cout << "Output buffer: " << pbase() << " to " << epptr() << " at " << pptr() << endl;
}

template <class charT, class traits>
bool
basic_netbuf<charT,traits>::open(const IPaddr& addr) {
	if(addr.get_num())
		return open(addr.get_num(),addr.get_port());
	return false;
}

template <class charT, class traits>
bool
basic_netbuf<charT,traits>::open(const IPaddr::const_ipname_t& str) {
	const char*const colon = strrchr(str,':');
	if(colon==NULL)
		return false;
	char* tmp = new char[strlen(str)];
	char * d = tmp;
	for(const char * s=str; s!=colon; s++,d++)
		*d=*s;
	*d='\0';
	IPaddr::ipport_t port = atoi(colon+1);
	bool res = open(tmp,port);
	delete [] tmp;
	return res;
}

template <class charT, class traits>
bool
basic_netbuf<charT,traits>::open(const IPaddr::ipnum_t& addr, const IPaddr::ipport_t& aPort) { //depends on timeout
	struct sockaddr_in server;
	
	memset((char *) &server, sizeof(server), 0);
	server.sin_family      = AF_INET;
	server.sin_addr.s_addr = addr;
	server.sin_port        = htons(aPort);

	// create socket
	sock = socket(AF_INET, SOCK_STREAM, 0);
	if(sock < 1) {
		cout << "NetClient error: socket failed to create stream socket" << endl;
		sock = INVALID_SOCKET;
		return false;
	}

	// connect to server.
	int err = connect(sock, (struct sockaddr *) &server, sizeof(server));
	if(err < 0) {
		cout << "NetClient error: connect failed to connect to requested address" << endl;
		sock = INVALID_SOCKET;
		return false;
	}

	return true;
}

template <class charT, class traits>
void
basic_netbuf<charT,traits>::close() {
	cout << "close called" << endl;
	if(!is_open())
		return;
	if(sock != INVALID_SOCKET) {
		cout << "closing" << endl;
		::close(sock);
		cout << "closed" << endl;
		sock = INVALID_SOCKET;
	}
}

template <class charT, class traits>
void
basic_netbuf<charT,traits>::printBuffer(const char *buf, int buflen, const char *header) {
	int line_begin = 1;
	for(int i = 0; i < buflen; i++) {
		if(buf[i] == '\n') {
			line_begin = 1;
			fputc('\n', stderr);
		} else {
			if(line_begin) fputs(header, stderr);
			line_begin = 0;
			fputc(buf[i], stderr);
		}
	}
}

template <class charT, class traits>
void
basic_netbuf<charT,traits>::in_sync() {
	if(!is_open()) {
		cout << "NetClient error: must open connection before reading from it" << endl;
		return;
	}
	if(gptr()>egptr())
		gbump( egptr()-gptr() );
	unsigned long n = gptr()-eback()-1;
	charT * temp_buf = new char[n*sizeof(charT)];
	unsigned int used = read(sock, temp_buf, n*sizeof(charT));
	if(used<=0) {
		cout << "NetClient error: connection dropped" << endl;
		delete [] temp_buf;
		close();
		return;
	}
	if(is_echoing)
		printBuffer(temp_buf, used, "NetClient: recv: ");
	//TODO - what if sizeof(charT)>1?  We might need to worry about getting/storing partial char
	used/=sizeof(charT);
	size_t remain = egptr()-eback()-used;
	traits::move(eback(),egptr()-remain,remain);
	traits::copy(egptr()-used,temp_buf,used);
	delete [] temp_buf;
	gbump( -used );
}

template <class charT, class traits>
void
basic_netbuf<charT,traits>::out_flush() {
	if(!is_open()) {
		cout << "NetClient error: must open connection before writing to it" << endl;
		return;
	}
	size_t n = (pptr()-pbase())*sizeof(charT);
	if(n==0)
		return;
	size_t total = 0;
	while(total<n) {
		int sent = write(sock, pbase()+total, n-total);
		if(sent < 0) {
			cout << "NetClient error: send error" << endl;
			perror("send");
			close();
			return;
		}
		if(is_echoing)
			printBuffer(pbase()+total, sent, "NetClient: sent: ");
		total += sent;
	}
	total/=sizeof(charT);
	n/=sizeof(charT);
	if(total!=n)
		traits::move(pbase(),pbase()+total,n-total);
	pbump( -total );
}


template <class charT, class traits>
inline streamsize
basic_netbuf<charT, traits>::showmanyc() {
	return (gptr()<egptr())?(egptr()-gptr()):0;
}

template <class charT, class traits>
inline typename basic_netbuf<charT, traits>::int_type
basic_netbuf<charT, traits>::underflow() {
	in_sync();
	if(gptr()<egptr())
		return traits::to_int_type(*gptr());
//	if( cur_input()!=kOTInvalidEndpointRef )
//		return uflow();
//	cout << "UNDERFLOW" << endl;
	return traits::eof();
}

template <class charT, class traits>
inline typename basic_netbuf<charT, traits>::int_type
basic_netbuf<charT, traits>::uflow() {
	in_sync();
	if(gptr()<egptr()) {
		int_type ans = traits::to_int_type(*gptr());
		gbump(1);
		return ans;
	}
//	cout << "UNDERFLOW" << endl;
	return traits::eof();
}

template <class charT, class traits>
inline typename basic_netbuf<charT, traits>::int_type
basic_netbuf<charT, traits>::overflow(int_type c) { 
	out_flush();
	*pptr() = c;
	pbump(1);
	return is_open()?c:traits::eof();
}

template <class charT, class traits>
inline int
basic_netbuf<charT, traits>::sync() {
	out_flush();
	in_sync();
	return is_open()?0:-1;
}


#endif
