//-*-c++-*-
/*=========================================================================
    CMPack'02 Source Code Release for OPEN-R SDK v1.0
    Copyright (C) 2002 Multirobot Lab [Project Head: Manuela Veloso]
    School of Computer Science, Carnegie Mellon University
  -------------------------------------------------------------------------
    This software is distributed under the GNU General Public License,
    version 2.  If you do not have a copy of this licence, visit
    www.gnu.org, or write: Free Software Foundation, 59 Temple Place,
    Suite 330 Boston, MA 02111-1307 USA.  This program is distributed
    in the hope that it will be useful, but WITHOUT ANY WARRANTY,
    including MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  -------------------------------------------------------------------------
    Additionally licensed to Sony Corporation under the following terms:

    This software is provided by the copyright holders AS IS and any
    express or implied warranties, including, but not limited to, the
    implied warranties of merchantability and fitness for a particular
    purpose are disclaimed.  In no event shall authors be liable for
    any direct, indirect, incidental, special, exemplary, or consequential
    damages (including, but not limited to, procurement of substitute
    goods or services; loss of use, data, or profits; or business
    interruption) however caused and on any theory of liability, whether
    in contract, strict liability, or tort (including negligence or
    otherwise) arising in any way out of the use of this software, even if
    advised of the possibility of such damage.
  =========================================================================
*/
#ifndef INCLUDED_Vision_h
#define INCLUDED_Vision_h

#include <math.h>

#include "cmvision.h"
#include "VisionInterface.h"

using namespace VisionInterface;

#include "Shared/Config.h"
#include "Visiondefines.h"
#include "Wireless/Wireless.h"
#include "Events/VisionEvent.h"

#define MAX_TMAPS 16


typedef uchar pixel;
typedef uchar cmap_t;
typedef CMVision::image<uchar> image;
typedef CMVision::color_class_state color_class_state;
typedef CMVision::run<cmap_t> run;
typedef CMVision::region region;

const int bits_y = 4;
const int bits_u = 6;
const int bits_v = 6;

/* focal dist of camera:
   fh = (w/2) / tan(h/2) = (176/2) / (100 / (2*109)) = 88 / (100/109)
      = 95.92
   fv = (h/2) / tan(v/2) = (144/2) / (100 / (2*150)) = 60 / (100/150)
      = 90
 */
static const double FocalDist =(176.0/2)/tan(HorzFOV/2); // 158.76
static const double FocalDistV=(144.0/2)/tan(VertFOV/2); // 161.71
// size of vertical pixel in terms of size of horizontal pixel
static const double YPixelSize=(FocalDistV/FocalDist);

static const double RobotArea = 13711; // mm^2 (empirically guessed, side view)

#define MIN_EXP_REGION_SIZE 16
#define MIN_EXP_RUN_LENGTH   8

// additional information needed by vision about objects not exported to other objects
struct VisionObjectInfo {
  region *reg,*reg2;
};

struct Marker {
  region* regs[3];
  int cen_x;
  int cen_y;
  int marker;
};

struct VisionEventSpec {
  int listeners; // number of listeners
  int filter;
  int count;
  double confidence;
  float cx, cy;
  bool present;
};

class VisionSerializer;

class Vision{
  friend class VisionSerializer;
public:
  Vision();
  ~Vision() { }

  unsigned long frameTimestamp;
  int frame_count;

  int num_tmaps;
  int cur_tmap;
  cmap_t *cmap,*tmap[MAX_TMAPS];
  run *rmap,*rmap2;
  region *reg;
  int yindex[144];

  VisionObjectInfo vobj_info[NUM_VISION_OBJECTS];
  VisionEventSpec vevent_spec[NUM_VEVENTS];

  color_class_state color[MAX_COLORS];

  int width,height;
  int max_width,max_height;
  int max_runs,max_regions;
  int num_colors,num_runs,num_regions;

  double body_angle, body_height;
  double head_angles[3];

  vector3d camera_loc,camera_dir,camera_up,camera_right;

  // distortion calibration stuff
  double sq_distort_coeff,lin_distort_coeff;

  int setThreshold(int threshold_id);

  const vector3d &get_camera_loc() {return camera_loc;}
  const vector3d &get_camera_dir() {return camera_dir;}
  int getColor(int x,int y)
    {return((x>=0 && y>=0 && x<width && y<height)?
            cmap[y*width+x] : COLOR_BACKGROUND);}
  int getWidth() {return width;}
  int getHeight() {return height;}
  void initialize();
  void setCameraParam();
  void initializeEventSpecs();

  void enableEvents(int vevent);
  void enableEvents(int vevent, int noise);
  void disableEvents(int vevent);
  
  void setNoiseThreshold(int vevent, int noise);
  
  bool close();

  bool processFrame(const uchar *data_y, const uchar *data_u, const uchar *data_v, int width, int height);
  bool saveThresholdImage(char *filename);

  void sendRawImage ();
  void sendRLEImage ();
  void sendColorArea();

  bool calcTotalArea;

  Marker markers[3];
  int vis_markers;
  ObjectInfo *obj_info;

private:

  int outCountAvgColor;  // number of frames since avg img color was output
  int outCountColorArea; // number of frames since color area was output
  int outCountRaw;       // number of frames since raw image was output
  int outCountRLE;       // number of frames since RLE image was output

  VisionSerializer* vser;

  // other stuff
  CMVision::image_yuv<const uchar> img;

  bool thresholdImage(CMVision::image_idx<rgb> &img);
  bool thresholdImage(CMVision::image_yuv<const uchar> &img);

  template<class image>
  bool runLowLevelVision(image &img);

  int getColorUnsafe(int x,int y)
    {return(cmap[y*width+x]);}
  int getNearColor(int x,int y)
    {return(cmap[bound(y,0,height-1)*width+bound(x,0,width-1)]);}
  // adds the colors found to the color_cnt histogram
  int addToHistHorizStrip(int y, int x1, int x2, int *color_cnt);
  int addToHistVertStrip (int x, int y1, int y2, int *color_cnt);
  void createEvent(unsigned int tid, unsigned int sid, float cenX, float cenY);

  // finds the direction of the ray through this pixel in robot coordinates
  vector3d getPixelDirection(double x, double y);
  // finds the leftest and rightmost ego angles on ground for a bounding box
  void findSpan(double &left, double &right, double x1, double x2, double y1, double y2);
  // calculates the edge mask for a bounding box
  int calcEdgeMask(double x1, double x2, double y1, double y2);
  int calcEdgeMask(int x1,int x2,int y1,int y2);
  int calcEdgeMask(region *tmpreg)
    {return(calcEdgeMask(tmpreg->x1,tmpreg->x2,tmpreg->y1,tmpreg->y2));}

  int isIn(region *r1, region *r2);
  int isAdjacent(region *r1, region *r2);

  bool findHand(VObject *hand,VisionObjectInfo *hand_info);
  bool findBall(int ball_color, VObject *ball,VisionObjectInfo *ball_info);
  bool findThing(VObject *thing,VisionObjectInfo *thing_info);
  bool findMarkers();
  bool findGesture(VisionObjectInfo *hand_info);
  int identifyMarker(int color1, int color2, int color3);
  bool generateEvent(int vevent, double conf, int cenX, int cenY);

  bool runHighLevelVision(ObjectInfo *obj_info);

	Vision(const Vision&); //!< don't copy
	Vision operator=(const Vision&); //!< don't assign
};

extern Vision* vision;

/*! @file
 * @brief Does majority of vision processing
 * @author CMU RoboSoccer 2001-2002 (Creator)
 * @author alokl (ported)
 * 
 * @verbinclude CMPack_license.txt
 *
 * $Author: alokl $
 * $Name: tekkotsu-1_3 $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2003/03/27 04:25:04 $
 */
#endif
