//-*-c++-*-
#ifndef INCLUDED_HeadPointerMC_h
#define INCLUDED_HeadPointerMC_h

#include "MotionCommand.h"
#include "OutputCmd.h"
#include "Shared/RobotInfo.h"

//! This class gives some quick and easy functions to point the head at things
/*! @note This class does not autoprune by default, which is contrary to the default of most other MotionCommands. */
class HeadPointerMC : public MotionCommand {
 public:
	//! constructor, defaults to active, BodyRelative, all joints at 0
	HeadPointerMC();
	//! destructor
	virtual ~HeadPointerMC() {}

	//! Various modes the head can be in.  In the future may want to add ability to explicitly track an object or point in the world model
	enum CoordFrame_t {
		BodyRelative,    //!<holds neck at a specified position, like a PostureEngine, but neck specific
		GravityRelative  //!<uses accelerometers to keep a level head, doesn't apply for pan joint, but in future could use localization for pan
	};

	       void   setWeight(double w);                               //!< sets the weight values for all the neck joints
	inline void   setWeight(RobotInfo::TPROffset_t i, double weight) { dirty=true; headJoints[i].weight=weight; } //!< set a specific head joint weight, pass one of RobotInfo::TPROffset_t, not a full offset!
	inline void   setActive(bool a)                                  { active=a; } //!< sets #active flag, see isDirty()
	inline bool   getActive() const                                  { return active; } //!< returns #active flag, see isDirty()

	//! converts a value @a v in @a srcmode to a value in @a tgtmode that would leave the joint angle for joint @a i constant (you probably won't need to call this directly)
	inline double convert(RobotInfo::TPROffset_t i, double v, CoordFrame_t srcmode, CoordFrame_t tgtmode) const {
		return (srcmode==tgtmode)?v:convFromBodyRelative(i,convToBodyRelative(i, v, srcmode),tgtmode);
	}

	//!Note that none of these are @c virtual, so you don't have to checkout to use them, you @e should be able to use MotionManager::peekMotion()
	//!@name Joint Accessors
	       void   setJoints(double tilt, double pan, double roll);	                      //!< Directly sets the neck values, uses current mode
	       void   setMode(CoordFrame_t m, bool convert=true);                                   //!< sets all the joints to the given mode, will convert the values to the new mode if @a convert is true
	       void   setJointMode(RobotInfo::TPROffset_t i, CoordFrame_t m, bool convert=true);    //!< set a specific head joint's mode, will convert from previous mode's value to next mode's value if convert is true.  Pass one of RobotInfo::TPROffset_t, not a full offset!
	inline void   setJointValue(RobotInfo::TPROffset_t i, double value)                   { dirty=true; headValues[i]=value;} //!< set a specific head joint's value (for whatever mode it's in), pass one of RobotInfo::TPROffset_t, not a full offset!
	inline void   setJointValueAndMode(RobotInfo::TPROffset_t i, double value, CoordFrame_t m)  { dirty=true; headValues[i]=value; headModes[i]=m; } //!< set a specific head joint, pass one of RobotInfo::TPROffset_t, not a full offset!
	inline void   setJointValueFromMode(RobotInfo::TPROffset_t i, double value, CoordFrame_t m) { dirty=true; headValues[i]=convert(i,value,m,headModes[i]); } //!< set a specific head joint auto converting @a value from mode @a m to the current mode, pass one of RobotInfo::TPROffset_t, not a full offset!
	inline CoordFrame_t getJointMode(RobotInfo::TPROffset_t i) const                            { return headModes[i]; } //!< returns the current mode for joint @a i (use RobotInfo::TPROffset_t, not global offset)
	inline double getJointValue(RobotInfo::TPROffset_t i) const                           { return headValues[i]; } //!< returns the current value (relative to the current mode) of joint @a i, use getOutputCmd() if you want to know the actual @b target joint value (to get the actual current joint position, look in WorldState
	//@}

	//!@name Inherited:
	virtual int              updateOutputs(  );             //!< Updates where the head is looking
	virtual const OutputCmd& getOutputCmd(unsigned int i);  //!< returns one of the #headJoints entries or ::unusedJoint if not a head joint
	virtual int              isDirty()                      { return (dirty && active)?1:0; } //!< true if a change has been made since the last updateJointCmds() and we're active
	virtual int              isAlive()                      { return true; }
	//@}

 protected:
	double convToBodyRelative(RobotInfo::TPROffset_t i, double v, CoordFrame_t mode) const;   //!< converts to a body relative measurement for joint @a i
	double convFromBodyRelative(RobotInfo::TPROffset_t i, double v, CoordFrame_t mode) const; //!< converts from a body relative measurement for joint @a i

	bool dirty;                         //!< true if a change has been made since last call to updateJointCmds()
	bool active;                        //!< set by accessor functions, defaults to true
	OutputCmd headJoints[NumHeadJoints]; //!< stores the current target head angles
	double headValues[NumHeadJoints];   //!< stores the target value of each joint, relative to #headModes
	CoordFrame_t headModes[NumHeadJoints];    //!< stores the current mode of each joint, for instance so tilt can be GravityRelative while pan is static
	static const OutputCmd unused;      //!< handy when we need a reference to an unused joint
};

/*! @file
 * @brief Describes HeadPointerMC, a class for various ways to control where the head is looking
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_3 $
 * $Revision: 1.4 $
 * $State: Exp $
 * $Date: 2003/03/04 05:46:05 $
 */

#endif

