#include "MMCombo.h"
#include "Shared/WorldState.h"
#include "Shared/Profiler.h"
#include "Shared/debuget.h"
#include "Shared/Config.h"
#include "Shared/SharedObject.h"
#include "Shared/ProcessID.h"
#include "Events/EventRouter.h"
#include "Behaviors/BehaviorBase.h"
#include "Motion/MotionManager.h"
#include "SoundPlay/SoundManager.h"
#include "Vision/Vision.h"
#include "Events/TextMsgEvent.h"

#include "Shared/ERS210Info.h"
#include "Shared/ERS220Info.h"

#include <OPENR/OSyslog.h>
#include <OPENR/core_macro.h>
#include <OPENR/OFbkImage.h>

using namespace std;

extern BehaviorBase& startupBehavior;

MMCombo::MMCombo()
	: OObject(), motmanMemRgn(NULL), worldStateMemRgn(NULL),
		soundManagerMemRgn(NULL), eventTranslatorQueueMemRgn(NULL),
		runLevel(0), num_open(0), etrans(), RPOPENR_isready(true), isStopped(true)
{
	for(unsigned int i=0; i<NumOutputs; i++) {
		primIDs[i]=oprimitiveID_UNDEF;
		open[i]=false;
	}
	Profiler::initBuckets();
}


OStatus
MMCombo::DoInit(const OSystemEvent&)
{
	isStopped=false;

	NEW_ALL_SUBJECT_AND_OBSERVER;
	REGISTER_ALL_ENTRY;
	SET_ALL_READY_AND_NOTIFY_ENTRY;
		
	// make sure the library doesn't drop data "for" us on this reliable communication channel
	observer[obsReceiveWorldState]->SetBufCtrlParam(0,1,1);
	observer[obsReceiveMotionManager]->SetBufCtrlParam(0,1,1);
	observer[obsMotionManagerComm]->SetBufCtrlParam(0,1,MotionManager::MAX_MOTIONS+1);
	//+1 to MAX_MOTIONS so we can get a delete message after we've filled up
		
	//Read config file
	config=new Config("/ms/config/tekkotsu.cfg");

	erouter = new EventRouter;

	if(strcmp(objectName,"MainObj")==0) {
		ProcessID::setID(ProcessID::MainProcess);

		bool isSlowOutput[NumOutputs];
		for(unsigned int i=0; i<NumOutputs; i++)
			isSlowOutput[i]=!IsFastOutput[i];

		SetupOutputs(isSlowOutput);

		//Request power status updates
		OPowerStatus observationStatus;
		observationStatus.Set(orsbALL,obsbALL,opsoREMAINING_CAPACITY_NOTIFY_EVERY_CHANGE,opsoTEMPERATURE_NOTIFY_EVERY_CHANGE,opsoTIME_DIF_NOTIFY_EVERY_CHANGE,opsoVOLUME_NOTIFY_EVERY_CHANGE);
		OServiceEntry entry(myOID_, Extra_Entry[entryGotPowerEvent]);
		OStatus result = OPENR::ObservePowerStatus(observationStatus, entry);
		if(result != oSUCCESS) {
			OSYSLOG1((osyslogERROR, "%s : %s %d","MMCombo::DoStart()","OPENR::ObservePowerStatus() FAILED", result));
			return oFAIL;
		}

    //Setup wireless
    wireless = new Wireless();
    sout=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*6);
    serr=wireless->socket(SocketNS::SOCK_STREAM,Wireless::WIRELESS_DEF_RECV_SIZE,Wireless::WIRELESS_DEF_SEND_SIZE*2);
		serr->setFlushType(SocketNS::FLUSH_BLOCKING);
    sout->setTextForward();
    serr->setForward(sout);

		//worldStateMemRgn -> state setup
		worldStateMemRgn = InitRegion(sizeof(WorldState));
		state=new ((WorldState*)worldStateMemRgn->Base()) WorldState;

		//eventTranslatorQueueMemRgn -> etrans setup
		eventTranslatorQueueMemRgn = InitRegion(sizeof(EventTranslator::Queue_t));
		EventTranslator::Queue_t * etransq=new ((EventTranslator::Queue_t*)eventTranslatorQueueMemRgn->Base()) EventTranslator::Queue_t;
		etrans.setQueue(etransq);
		MotionCommand::setQueue(etransq);

    //Setup vision
    vision=new Vision();

	}
	if(strcmp(objectName,"MotoObj")==0) {
		ProcessID::setID(ProcessID::MotionProcess);

		SetupOutputs(IsFastOutput);
    OPENR::SetMotorPower(opowerON);
		OPENR::EnableJointGain(oprimitiveID_UNDEF); //oprimitiveID_UNDEF means enable all

		//motmanMemRgn -> motman setup
		motmanMemRgn = InitRegion(sizeof(MotionManager));
		motman = new (motmanMemRgn->Base()) MotionManager();
		motman->InitAccess(subject[sbjMotionManagerComm]);
	}
	startupBehavior.AddReference();
	return oSUCCESS;
}

OStatus
MMCombo::DoStart(const OSystemEvent&)
{
	//	cout << objectName << "::DoStart() " << endl;

	// initialize the current power status, doesn't always give us
	// a power update right away otherwise
	if(strcmp(objectName,"MainObj")==0) {
		wireless->listen(sout, config->main.console_port);
		wireless->listen(serr, config->main.stderr_port);
		OPowerStatus power;
		OPENR::GetPowerStatus(&power);
		state->read(power,erouter);
	}

	isStopped=false;

	ENABLE_ALL_SUBJECT;
	ASSERT_READY_TO_ALL_OBSERVER;

	addRunLevel();
	
	return oSUCCESS;
}

OStatus
MMCombo::DoStop(const OSystemEvent&)
{
	cout << objectName << "::DoStop()..." << endl;
	if(strcmp(objectName,"MainObj")==0) {
		startupBehavior.DoStop();
		wireless->close(sout);
		wireless->close(serr);
	}
	DISABLE_ALL_SUBJECT;
	DEASSERT_READY_TO_ALL_OBSERVER;
	isStopped=true;
	cout << objectName << "::DoStop()-DONE" << endl;
	return oSUCCESS;
}

OStatus
MMCombo::DoDestroy(const OSystemEvent&)
{
	cout << objectName << "::DoDestroy()..." << endl;
	startupBehavior.RemoveReference();
	if(strcmp(objectName,"MainObj")==0) {
		delete erouter;
		motmanMemRgn->RemoveReference();
	}
	if(strcmp(objectName,"MotoObj")==0) {
		worldStateMemRgn->RemoveReference();
		eventTranslatorQueueMemRgn->RemoveReference();
	}
	soundManagerMemRgn->RemoveReference();
	DELETE_ALL_SUBJECT_AND_OBSERVER;
	cout << objectName << "::DoDestroy()-DONE" << endl;
	return oSUCCESS;
}

/*! Called when MotoObj is initially ready as well as when it has finished
 *  processing the previous message - we only want to do this the first time
 *  otherwise we infinite loop. */
void
MMCombo::ReadyRegisterWorldState(const OReadyEvent&){
	static bool is_init=true;
	if(is_init) {
		is_init=false;
		cout << objectName << " Registering WorldState" << endl;
		if(strcmp(objectName,"MainObj")==0) {
			subject[sbjRegisterWorldState]->SetData(worldStateMemRgn);
			subject[sbjRegisterWorldState]->NotifyObservers();
		}
	}
}

void
MMCombo::GotWorldState(const ONotifyEvent& event){
	cout << objectName << "-GOTWORLDSTATE..." << flush;
	//	PROFSECTION("GotMemRegion()",state->mainProfile);
	if(strcmp(objectName,"MotoObj")==0) {
		ASSERT(event.NumOfData()==1,"Too many WorldStates");
		worldStateMemRgn = event.RCData(0);
		worldStateMemRgn->AddReference();
		state = reinterpret_cast<WorldState*>(worldStateMemRgn->Base());
	}
  observer[obsReceiveWorldState]->AssertReady();
	cout << "done" << endl;
}

		
/*! Called when MainObj is initially ready as well as when it has finished
 *  processing the previous message - we only want to do this the first time
 *  otherwise we infinite loop. */
void
MMCombo::ReadyRegisterMotionManager(const OReadyEvent&){
	static bool is_init=true;
	if(is_init) {
		is_init=false;
		cout << objectName << " Registering MotionManager" << endl;
		if(strcmp(objectName,"MotoObj")==0) {
			subject[sbjRegisterMotionManager]->SetData(motmanMemRgn);
			subject[sbjRegisterMotionManager]->NotifyObservers();
		}
	}
}

void
MMCombo::GotMotionManager(const ONotifyEvent& event){
	cout << objectName << "-GOTWORLDSTATE..." << flush;
	//	PROFSECTION("GotMemRegion()",state->motoProfile);
	if(strcmp(objectName,"MainObj")==0) {
		ASSERT(event.NumOfData()==1,"Too many MotionManagers");
		motmanMemRgn = event.RCData(0);
		motmanMemRgn->AddReference();
		motman = reinterpret_cast<MotionManager*>(motmanMemRgn->Base());
		cout << "MAIN INIT MOTMAN..." << flush;
		//			hexout(event.RCData(event_data_id)->Base(),128);
		motman->InitAccess(subject[sbjMotionManagerComm]);
		addRunLevel();
	}
  observer[obsReceiveMotionManager]->AssertReady();
	cout << "done" << endl;
}

/*! Called when MotoObj is initially ready as well as when it has finished
 *  processing the previous message - we only want to do this the first time
 *  otherwise we infinite loop. */
void
MMCombo::ReadyRegisterEventTranslatorQueue(const OReadyEvent&){
	static bool is_init=true;
	if(is_init) {
		is_init=false;
		cout << objectName << " Registering EventTranslatorQueue" << endl;
		if(strcmp(objectName,"MainObj")==0) {
			subject[sbjRegisterEventTranslatorQueue]->SetData(eventTranslatorQueueMemRgn);
			subject[sbjRegisterEventTranslatorQueue]->NotifyObservers();
		}
	}
}

void
MMCombo::GotEventTranslatorQueue(const ONotifyEvent& event){
	cout << objectName << "-GOTEventTranslatorQueue..." << flush;
	//	PROFSECTION("GotMemRegion()",state->mainProfile);
	if(strcmp(objectName,"MotoObj")==0) {
		ASSERT(event.NumOfData()==1,"Too many EventTranslatorQueue");
		eventTranslatorQueueMemRgn = event.RCData(0);
		eventTranslatorQueueMemRgn->AddReference();
		EventTranslator::Queue_t * etransq=reinterpret_cast<EventTranslator::Queue_t*>(eventTranslatorQueueMemRgn->Base());
		etrans.setQueue(etransq);
		MotionCommand::setQueue(etransq);
		erouter->addTrapper(&etrans);
	}
  observer[obsReceiveEventTranslatorQueue]->AssertReady();
	cout << "done" << endl;
}

		
void
MMCombo::ReadySendJoints(const OReadyEvent& sysevent) {

	if(isStopped) {
		//cout << "BAH!ReadySendJoints" << endl;
		return;
	}

	static unsigned int id=-1U;
	Profiler::Timer timer;
	if(ProcessID::getID()==ProcessID::MotionProcess) {
		if(state) {
			if(id==-1U)
				id=state->motionProfile.getNewID("ReadySendJoints()");
			timer.setID(id,&state->motionProfile);
		}
	}	else if(ProcessID::getID()==ProcessID::MainProcess) {
		if(id==-1U)
			id=state->mainProfile.getNewID("ReadySendJoints()");
		timer.setID(id,&state->mainProfile);
	}

	if(num_open==0) //If we don't have any joints to open, leave now. (i.e. MainObj on a 220, has no ears)
		return;

	// Find an unused command vector
	RCRegion* rgn=NULL;
	for (unsigned int i = 0; i < NUM_COMMAND_VECTOR; i++) {
		if (region[i]->NumberOfReference() == 1) {
			rgn=region[i];
			/*			if(strcmp(objectName,"MainObj")==0) {
							static unsigned int lasttime=get_time();
							unsigned int thistime=get_time();
							cout << '*' << i << ' ' << thistime << '\t' << (thistime-lasttime) << endl;
							lasttime=thistime;
							}*/
			break;
		}
	}
	ASSERTRET(rgn!=NULL,"Could not find unused command vector");
	ASSERTRET(rgn->Base()!=NULL,"Bad Command Vector");
	OCommandVectorData* cmdVecData = reinterpret_cast<OCommandVectorData*>(rgn->Base());
	
	// Update the outputs (note that Main is doing the ears)
	//I'm using an id compare instead of the slightly more readable strcmp for a tiny bit of speed
	if(ProcessID::getID()==ProcessID::MotionProcess) {
		float outputs[NumFrames][NumOutputs];
		if(state!=NULL) {
			motman->getOutputs(outputs);
			motman->updatePIDs(primIDs);
			motman->updateWorldState();
		} else {
			for(unsigned int f=0; f<NumFrames; f++)
				for(unsigned int i=0; i<NumOutputs; i++)
					outputs[f][i]=0;
		}

		// Should be a relatively simple matter to copy angles into commands...
		unsigned int used=0; //but only copy open joints (so main does ears, motion does everything else)
		for(unsigned int i=PIDJointOffset; i<PIDJointOffset+NumPIDJoints; i++)
			if(open[i]) {
				OJointCommandValue2* jval = reinterpret_cast<OJointCommandValue2*>(cmdVecData->GetData(used)->value);
				for(unsigned int frame=0; frame<NumFrames; frame++)
					jval[frame].value = (slongword)(outputs[frame][i]*1e6);
				used++;
			}
		for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
			if(open[i]) {
				OLEDCommandValue2* jval = reinterpret_cast<OLEDCommandValue2*>(cmdVecData->GetData(used)->value);
				for(unsigned int frame=0; frame<NumFrames; frame++)
					jval[frame].led = calcLEDValue(i-LEDOffset,outputs[frame][i]);
				used++;
			}
		for(unsigned int i=BinJointOffset; i<BinJointOffset+NumBinJoints; i++)
			if(open[i]) {
				OJointCommandValue3* jval = reinterpret_cast<OJointCommandValue3*>(cmdVecData->GetData(used)->value);
				for(unsigned int frame=0; frame<NumSlowFrames; frame++)
					jval[frame].value = (outputs[frame][i]<.5?ojoint3_STATE1:ojoint3_STATE0);
				used++;
			}
	}	else if(ProcessID::getID()==ProcessID::MainProcess) {
		// Just copy over the current ear state from WorldState in case this is in Main
		unsigned int used=0; //but only copy open joints (so main does ears, motion does everything else)
		for(unsigned int i=BinJointOffset; i<BinJointOffset+NumBinJoints; i++)
			if(open[i]) {
				OJointCommandValue3* jval = reinterpret_cast<OJointCommandValue3*>(cmdVecData->GetData(used)->value);
				for(unsigned int frame=0; frame<NumSlowFrames; frame++)
					jval[frame].value = (state->outputs[i]<.5?ojoint3_STATE1:ojoint3_STATE0);
				used++;
			}
	}

	// Send outputs to system
	subject[sbjMoveJoint]->SetData(rgn);

	// The first time this is called, we actually need to send *two* buffers
	// in order to get the double buffering going... (well, actually generalized
	// for NUM_COMMAND_VECTOR level buffering)
	static unsigned int initCount=1;
	if(initCount<NUM_COMMAND_VECTOR) {
		initCount++;
		ReadySendJoints(sysevent);
	} else //recursive base case
		subject[sbjMoveJoint]->NotifyObservers();
}

void
MMCombo::GotSensorFrame(const ONotifyEvent& event){
	//	if(state && state->buttons[RFrPawOffset])
	//	cout << "SENSOR..."<<flush;
	if(isStopped) {
		//cout << "BAH!GotSensorFrame" << endl;
		return;
	}

	PROFSECTION("GotSensorFrame()",state->mainProfile);
	etrans.translateEvents();

  OSensorFrameVectorData* rawsensor = reinterpret_cast<OSensorFrameVectorData*>(event.RCData(0)->Base());
	state->read(rawsensor[0],erouter);
	erouter->processTimers();
	static unsigned int throwaway=1; //i thought the first few sensor updates might be flakey, but now i think not.  But a good way to delay startup.
	if(throwaway!=0) {
		throwaway--;
		if(throwaway==0)
			addRunLevel();
	}
  observer[obsSensorFrame]->AssertReady();
  if (state!=NULL && state->wsser!=NULL)
    state->wsser->serialize();
	//	if(state && state->buttons[RFrPawOffset])
	//	cout << "done" << endl;
}

void
MMCombo::GotImage(const ONotifyEvent& event){
	if(isStopped) {
		//cout << "BAH!GotImage" << endl;
		return;
	}

	PROFSECTION("GotImage()",state->mainProfile);
	etrans.translateEvents();

  const OFbkImageVectorData *fbkImageVectorData=reinterpret_cast<const OFbkImageVectorData*>(event.Data(0));
  OFbkImageInfo* info = const_cast<OFbkImageVectorData*>(fbkImageVectorData)->GetInfo(0);
  const byte* dat = const_cast<OFbkImageVectorData*>(fbkImageVectorData)->GetData(0);
  const OFbkImage yimage(info, const_cast<byte*>(dat), ofbkimageBAND_Y);
  const OFbkImage uimage(info, const_cast<byte*>(dat), ofbkimageBAND_Cr);
  const OFbkImage vimage(info, const_cast<byte*>(dat), ofbkimageBAND_Cb);
            
  const byte* bytes_y = yimage.Pointer();
  const byte* bytes_u = uimage.Pointer();
  const byte* bytes_v = vimage.Pointer();

  int width  = yimage.Width();
  int height = yimage.Height();

  vision->processFrame(bytes_y, bytes_u, bytes_v, width, height);
 
  erouter->processTimers();
  
  observer[obsImage]->AssertReady();
}

void
MMCombo::GotPowerEvent(void * msg){
	if(isStopped) {
		//cout << "BAH!GotPowerEvent" << endl;
		return;
	}

	//	cout << "POWER..."<<flush;
	PROFSECTION("PowerEvent()",state->mainProfile);
	etrans.translateEvents();

	static bool first=true;
	if(first) {
		addRunLevel();
		first=false;
	}
	const OPowerStatus* result = &static_cast<OPowerStatusMessage*>(msg)->powerStatus;
	state->read(*result,erouter);
	erouter->processTimers();
	// this part watches to see if the power button is pressed to shutdown the robot
	// i'm leaving this low-level because there's not much else you can do anyway...
	// the hardware kills power to the motors, and as far as we can tell, you can't
	// turn them back on.
	if(state->powerFlags[PowerSourceID::PauseSID]) {
		cout << "%%%%%%%  Pause button was pushed! %%%%%%%" << endl;
		OBootCondition bc(0);
		OPENR::Shutdown(bc);
	}
	//	cout << "done" << endl;
}

void
MMCombo::GotMotionMsg(const ONotifyEvent& event){
	if(isStopped) {
		//cout << "BAH!GotMotionMsg" << endl;
		return;
	}

	//	cout << "RECEIVE..."<<flush;
	if(motman!=NULL)
		motman->receivedMsg(event);
	else
		cout << "*** WARNING Main dropping MotionCommand (motman not ready) " << endl;
	observer[obsMotionManagerComm]->AssertReady();
	//	cout << "done" << endl;
}

void
MMCombo::GotSoundManager(const ONotifyEvent& event) {
	cout << objectName << "-GOTSOUNDMANAGER..." << flush;
	//	PROFSECTION("GotMemRegion()",state->mainProfile);
	ASSERT(event.NumOfData()==1,"Too many SoundManagers");
	soundManagerMemRgn = event.RCData(0);
	soundManagerMemRgn->AddReference();
	sndman = reinterpret_cast<SoundManager*>(soundManagerMemRgn->Base());
  observer[obsReceiveSoundManager]->AssertReady();
	sndman->InitAccess(subject[sbjSoundManagerComm]);
	addRunLevel();
	cout << "done" << endl;
}

void
MMCombo::OpenPrimitives()
{
	for(unsigned int i=0; i<NumOutputs; i++)
		if(open[i]) {
			OStatus result = OPENR::OpenPrimitive(PrimitiveName[i], &primIDs[i]);
			if (result != oSUCCESS)
				OSYSLOG1((osyslogERROR, "%s : %s %d","MMCombo::DoInit()","OPENR::OpenPrimitive() FAILED", result));
		}
}

void
MMCombo::SetupOutputs(const bool to_open[NumOutputs])
{
	char robotDesignStr[orobotdesignNAME_MAX];
	memset(robotDesignStr, 0, sizeof(robotDesignStr));
	if (OPENR::GetRobotDesign(robotDesignStr) != oSUCCESS) {
		cout << objectName << "::SetupOutputs - OPENR::GetRobotDesign() failed." << endl;
		return;
	} else {
		if(strcmp(robotDesignStr,"ERS-210")==0) {
			for(unsigned int j=0; j<NumOutputs; j++)
				open[j]=to_open[j] && ERS210Info::IsRealERS210[j];
		} else if(strcmp(robotDesignStr,"ERS-220")==0) {
			for(unsigned int j=0; j<NumOutputs; j++)
				open[j]=to_open[j] && ERS220Info::IsRealERS220[j];
		} else {
			cout << "MMCombo::SetupOutputs - ERROR: Unrecognized model: "<<robotDesignStr<<"\nSorry..."<<endl;
			return;
		}
	}
	
	// count how many we're opening
	for(unsigned int j=0; j<NumOutputs; j++)
		if(open[j])
			num_open++;

	if(num_open==0) //If we don't have any joints to open, leave now. (i.e. MainObj on a 220, has no ears)
		return;

	OpenPrimitives();

	OStatus result;
	MemoryRegionID      cmdVecDataID;
	OCommandVectorData* cmdVecData;

	// request memory regions
	for (unsigned int i = 0; i < NUM_COMMAND_VECTOR; i++) {
		result = OPENR::NewCommandVectorData(num_open,&cmdVecDataID,&cmdVecData);
		if (result != oSUCCESS)
			OSYSLOG1((osyslogERROR, "%s : %s %d","MMCombo::NewCommandVectorData()","OPENR::NewCommandVectorData() FAILED", result));
		region[i] = new RCRegion(cmdVecData->vectorInfo.memRegionID,cmdVecData->vectorInfo.offset,(void*)cmdVecData,cmdVecData->vectorInfo.totalSize);
		cmdVecData->SetNumData(num_open);

		// initialize the outputs we just opened
		unsigned int used=0;
		ASSERT(cmdVecData==reinterpret_cast<OCommandVectorData*>(region[i]->Base())," should be equal!?");
		for(unsigned int j=PIDJointOffset; j<PIDJointOffset+NumPIDJoints; j++)
			if(open[j]) {
				OCommandInfo* info = cmdVecData->GetInfo(used++);
				info->Set(odataJOINT_COMMAND2, primIDs[j], NumFrames);
			}
		for(unsigned int j=LEDOffset; j<LEDOffset+NumLEDs; j++)
			if(open[j]) {
				OCommandInfo* info = cmdVecData->GetInfo(used);
				info->Set(odataLED_COMMAND2, primIDs[j], NumFrames);
				OLEDCommandValue2* jval = reinterpret_cast<OLEDCommandValue2*>(cmdVecData->GetData(used)->value);
				for(unsigned int frame=0; frame<NumFrames; frame++)
					jval[frame].period = 1;
				used++;
			}			
		for(unsigned int j=BinJointOffset; j<BinJointOffset+NumBinJoints; j++)
			if(open[j]) {
				OCommandInfo* info = cmdVecData->GetInfo(used++);
				info->Set(odataJOINT_COMMAND3, primIDs[j], NumSlowFrames);
			}			
	}
}

/*! Will round up size to the nearest page */
RCRegion*
MMCombo::InitRegion(unsigned int size) {
	unsigned int pagesize=4096;
	sError err=GetPageSize(&pagesize);
	ASSERT(err==sSUCCESS,"Error "<<err<<" getting page size");
	unsigned int pages=(size+pagesize-1)/pagesize;
	return new RCRegion(pages*pagesize);
}

void
MMCombo::addRunLevel() {
	runLevel++;
	if(runLevel==readyLevel) {
		cout << "START UP BEHAVIOR..." << flush;
		startupBehavior.DoStart();
		cout << "START UP BEHAVIOR-DONE" << endl;
	}
}

void
MMCombo::RPOPENR_notify(const ONotifyEvent& event) {
  const char *buf = (const char *)event.Data(0);
  observer[event.ObsIndex()]->AssertReady();

  /* usercode: message received
   * an example could be- */
  // RPOPENR_send("yeah i received your message", 29);

  /* create TextMsg event which can contain strings
   * comment this out if you want to handle the received data exclusively
   */
  erouter->postEvent(new TextMsgEvent(buf));
}

int
MMCombo::RPOPENR_send(char *buf, int bufsize) {
  if (RPOPENR_isready && bufsize>0) {
    RPOPENR_isready=false;
    subject[sbjRPOPENRSendString]->SetData(buf, bufsize);
    subject[sbjRPOPENRSendString]->NotifyObservers();
    return bufsize;
  }
  return 0;
}

/*! @file
 * @brief Implements MMCombo, the OObject which "forks" (sort of) into Main and Motion processes
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_3 $
 * $Revision: 1.36 $
 * $State: Exp $
 * $Date: 2003/06/12 18:06:11 $
 */


