//-*-c++-*-
#ifndef INCLUDED_EventTranslator_h_
#define INCLUDED_EventTranslator_h_

#include "Shared/SharedQueue.h"
#include "Events/EventTrapper.h"

//! EventTranslator receives events from EventRouters in non-Main processes and adds them into a SharedQueue for Main to pick up
/*! The SharedQueue which the processes should set up and then pass to this object
 *  is defined by TranslatorSharedQueue_t - it allows 100 entries totalling 3KB of space.
 *  You can modify the type to change the capacity, but if you're making that many events that
 *  between Main's calls to translateEvents() you might want to rethink a few things...
 * 
 *  EventTranslator only handles LocomotionEvent, VisionEvent, and TextMsgEvent subtypes for
 *  the moment.  Anything else is only handled as an EventBase class. (so extra fields aren't
 *  going to be stored) It's easy to add more types if you need to send them from other
 *  processes (that's the whole point of this class!)
 *
 *  Reason for doing it this way: Avoids OPENR message lag time (4-8ms in our testing), also
 *  avoids problems with RTTI stored in classes from different processes. */
class EventTranslator : public EventTrapper {
public:
	//!constructor
	EventTranslator() : queue(NULL) {}

	//!Use this type to set up the shared queue between processes
	typedef SharedQueue<3*1024,100> Queue_t;
	
	//!an ID is inserted before the event data in the queue so we can tell which subclass it is
	/*!(quickly, could look at the creator code which is stored, but that's a text string)*/
	enum TypeID_t { EventBase_ID, LocomotionEvent_ID, VisionEvent_ID, TextMsgEvent_ID };

	//!sets the SharedQueue which should be used
	void setQueue(Queue_t * q) { queue=q; }

	//!called by the event router when something in this process sends an event
	virtual bool trapEvent(const EventBase& event);
	
	//!called by trapEvent to do all the work, needed so MotionCommands can enqueue directly
	static void enqueue(const EventBase& event, Queue_t * q);

	//!called whenever Main gets some processor time to check for events from other processes
	void translateEvents();

protected:
	//!called by translateEvents for each event to be sent
	static void sendEvent(const void* buf, unsigned int size);
	
	//!pointer to queue of events to be sent
	Queue_t * queue;

private:
	EventTranslator(const EventTranslator&); //!< don't call
	EventTranslator operator=(const EventTranslator&); //!< don't call
};

/*! @file
 * @brief Describes EventTranslator, which receives events from EventRouters in non-Main processes and adds them into a SharedQueue for Main to pick up
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_3 $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2003/06/12 23:41:40 $
 */

#endif
