#include "Controller.h"
#include "Motion/EmergencyStopMC.h"
#include "Motion/LedMC.h"
#include "Motion/MMAccessor.h"
#include "Shared/SharedObject.h"
#include "Shared/WorldState.h"
#include "Shared/get_time.h"
#include "SoundPlay/SoundManager.h"
#include "Events/TextMsgEvent.h"
#include "Shared/ERS210Info.h"
#include "Shared/ERS220Info.h"

Controller * Controller::theOneController=NULL;

//these are given appropriate values in init once we know which model we're running on
EventBase Controller::nextItem;
EventBase Controller::prevItem;
EventBase Controller::nextItemFast;
EventBase Controller::prevItemFast;
EventBase Controller::selectItem;
EventBase Controller::cancel;


void Controller::DoStart() {
	BehaviorBase::DoStart();
	sndman->LoadFile(config->controller.select_snd);
	sndman->LoadFile(config->controller.next_snd);
	sndman->LoadFile(config->controller.prev_snd);
	sndman->LoadFile(config->controller.read_snd);
	sndman->LoadFile(config->controller.cancel_snd);
	erouter->addListener(this,EventBase::estopEGID);
	// Turn on wireless
	wireless->setReceiver(gui_comm->sock, gui_comm_callback);
	wireless->listen(gui_comm->sock, config->controller.gui_port);
	theOneController=this;
	reset();
}

void Controller::DoStop() {
	sndman->ReleaseFile(config->controller.select_snd);
	sndman->ReleaseFile(config->controller.next_snd);
	sndman->ReleaseFile(config->controller.prev_snd);
	sndman->ReleaseFile(config->controller.read_snd);
	sndman->ReleaseFile(config->controller.cancel_snd);
	erouter->forgetListener(this);
	reset();
	wireless->close(gui_comm);
	theOneController=NULL;
	BehaviorBase::DoStop();
}

bool Controller::trapEvent(const EventBase& e) {
	if(!chkCmdStack())
		return false;
	last_time=cur_time;
	cur_time=get_time();
	if(nextItem.sameGenSource(e)) {
		nextEv_val=e.getMagnitude();
		nextEv_dur=e.getDuration();
		if(nextEv_val==0 && prevEv_val==0)
			alreadyGotBoth=false;
		if(nextEv_val>.75 && prevEv_val>.75 && nextEv_dur<666 && prevEv_dur<666)
			if(alreadyGotBoth)
				return true;
			else {
				alreadyGotBoth=true;
				return setNext(cmdstack.top()->doReadStdIn());
			}
		if(e.getTypeID()==nextItem.getTypeID() && e.getDuration()<666)
			return setNext(cmdstack.top()->doNextItem());
		if(e.getTypeID()==nextItemFast.getTypeID() && e.getDuration()>666 && calcPulse(cur_time,last_time,static_cast<unsigned int>(50/e.getMagnitude())))
			return setNext(cmdstack.top()->doNextItem());
	}
	if(prevItem.sameGenSource(e)) {
		prevEv_val=e.getMagnitude();
		prevEv_dur=e.getDuration();
		if(nextEv_val==0 && prevEv_val==0)
			alreadyGotBoth=false;
		if(nextEv_val>.75 && prevEv_val>.75 && nextEv_dur<666 && prevEv_dur<666)
			if(alreadyGotBoth)
				return true;
			else {
				alreadyGotBoth=true;
				return setNext(cmdstack.top()->doReadStdIn());
			}
		if(e.getTypeID()==prevItem.getTypeID() && e.getDuration()<666)
			return setNext(cmdstack.top()->doPrevItem());
		if(e.getTypeID()==prevItemFast.getTypeID() && e.getDuration()>666 && calcPulse(cur_time,last_time,static_cast<unsigned int>(50/e.getMagnitude())))
			return setNext(cmdstack.top()->doPrevItem());
	}
	if(e.getDuration()>250) {
		if(e==selectItem)
			return setNext(cmdstack.top()->doSelect());
		if(e==cancel)
			return setNext(cmdstack.top()->doCancel());
	}
	return true;
}

void Controller::processEvent(const EventBase& event) {
	if(event.getTypeID()==EventBase::activateETID) { //estop just turned on
		if(display==MotionManager::invalid_MC_ID)
			activate();
	}	else { //estop just turned off
		if(display!=MotionManager::invalid_MC_ID)
			deactivate();
	}
}

void Controller::reset() {
	while(cmdstack.size()>1)
		pop();
	if(!cmdstack.empty()) {
		cmdstack.top()->deactivate();
		cmdstack.pop();
	}
}

void Controller::refresh() {
	if(!chkCmdStack())
		return;
	cmdstack.top()->refresh();
}

void Controller::push(ControlBase* c) {
	if(!chkCmdStack())
		return;
	cmdstack.top()->pause();
	cmdstack.push(c);
	setNext(cmdstack.top()->activate(display,gui_comm));
}

void Controller::pop() {
	cmdstack.top()->deactivate();
	cmdstack.pop();
	refresh();
}

Controller& Controller::setRoot(ControlBase* r) {
	reset();
	root=r;
	refresh();
	return *this;
}

Controller& Controller::setEStopID(MotionManager::MC_ID estopid) {
	estop_id=estopid;
	if(static_cast<EmergencyStopMC*>(motman->peekMotion(estopid))->getStopped()) {
		if(display==MotionManager::invalid_MC_ID)
			activate();
	} else {
		if(display!=MotionManager::invalid_MC_ID)
			deactivate();
	}		
	return *this;
}

int Controller::gui_comm_callback(char *buf, int bytes) {
	std::string s(buf,bytes);
	//	cout << "Controller Received: " << s << endl;
	if(theOneController==NULL)
		return 0;

	static std::string incomplete;

	//pass a line at a time to the controller
	while(s.size()>0) {
		unsigned int endline=s.find('\n');
		if(endline==std::string::npos) {
			incomplete+=s;
			return 0;
		}
		incomplete+=s.substr(0,endline);
		theOneController->takeLine(incomplete); //is now complete
		incomplete.erase();
		s=s.substr(endline+1);
	}
	
	return 0;
}

int Controller::console_callback(char *buf, int bytes) {
	std::string s(buf,bytes);
	//	cout << "Console Received: " << s << endl;
	if(theOneController==NULL)
		return 0;

	static std::string incomplete;

	//pass a line at a time to the controller
	while(s.size()>0) {
		unsigned int endline=s.find('\n');
		if(endline==std::string::npos) {
			incomplete+=s;
			return 0;
		}
		incomplete+=s.substr(0,endline);
		//is now complete:
		if(wireless->isConnected(theOneController->gui_comm->sock))
			erouter->postEvent(new TextMsgEvent(incomplete));
		else
			theOneController->takeLine(incomplete); 
		incomplete.erase();
		s=s.substr(endline+1);
	}
	
	return 0;
}

void Controller::init() {
	if(state->robotDesign & WorldState::ERS210Mask) {
		nextItem=EventBase(EventBase::buttonEGID,ERS210Info::HeadFrButOffset,EventBase::deactivateETID,0);
		prevItem=EventBase(EventBase::buttonEGID,ERS210Info::HeadBkButOffset,EventBase::deactivateETID,0);
		nextItemFast=EventBase(EventBase::buttonEGID,ERS210Info::HeadFrButOffset,EventBase::statusETID,666);
		prevItemFast=EventBase(EventBase::buttonEGID,ERS210Info::HeadBkButOffset,EventBase::statusETID,666);
		selectItem=EventBase(EventBase::buttonEGID,ERS210Info::ChinButOffset,EventBase::deactivateETID,250);
		cancel=EventBase(EventBase::buttonEGID,ERS210Info::BackButOffset,EventBase::deactivateETID,250);
	} else if(state->robotDesign & WorldState::ERS220Mask) {
		nextItem=EventBase(EventBase::buttonEGID,ERS220Info::TailLeftButOffset,EventBase::deactivateETID,0);
		prevItem=EventBase(EventBase::buttonEGID,ERS220Info::TailRightButOffset,EventBase::deactivateETID,0);
		//the 220 doesn't really support the next two because it's using boolean buttons
		//i'm using a "hack" on the 210 because the pressure sensitivity causes status
		//events to continually be sent but since this is just on/off, it only gets the
		//activate/deactivate.  To fix this, make these timers and do timer management
		//in processEvents()
		nextItemFast=EventBase(EventBase::buttonEGID,ERS220Info::TailLeftButOffset,EventBase::statusETID,666);
		prevItemFast=EventBase(EventBase::buttonEGID,ERS220Info::TailRightButOffset,EventBase::statusETID,666);
		selectItem=EventBase(EventBase::buttonEGID,ERS220Info::TailCenterButOffset,EventBase::deactivateETID,50);
		cancel=EventBase(EventBase::buttonEGID,ERS220Info::BackButOffset,EventBase::deactivateETID,50);
	}
}

void Controller::takeLine(const std::string& s) {
	//	cout << "RECEIVED: " << s << endl;
	if(s.size()==0) {
		return;
	} else if(s[0]!='!') {
		setNext(cmdstack.top()->takeInput(s));
	} else {
		//find the first non-whitespace character after the '!'
		std::string msg;
		{ unsigned int i=1;	while(isspace(s[i]) && i<s.size()) i++;	msg=s.substr(i); }
		if(msg.find("refresh")==0) {
			refresh();
		} else if(msg.find("reset")==0) {
			reset();
		} else if(msg.find("cancel")==0) {
			setNext(cmdstack.top()->doCancel());
		} else if(msg.find("select")==0) {
			setNext(cmdstack.top()->doSelect());
		} else if(msg.find("next")==0) {
			setNext(cmdstack.top()->doNextItem());
		} else if(msg.find("prev")==0) {
			setNext(cmdstack.top()->doPrevItem());
		} else if(msg.find("msg ")==0) {
			erouter->postEvent(new TextMsgEvent(msg.substr(strlen("msg "))));
		} else if(msg.find("hilight")==0) {
			std::vector<unsigned int> hilights;
			unsigned int i=strlen("hilight");
			while(i<msg.size()) {
				while(i<msg.size() && isspace(msg[i])) i++;
				if(i<msg.size())
					 hilights.push_back(atoi(&msg.c_str()[i]));
				while(i<msg.size() && !isspace(msg[i])) i++;
			}
			cmdstack.top()->setHilights(hilights);
		} else if(msg.find("input ")==0) {
			const std::vector<unsigned int>& hilights=cmdstack.top()->getHilights();
			const std::vector<ControlBase*>& slots=cmdstack.top()->getSlots();
			std::string in=msg.substr(strlen("input "));
			for(unsigned int i=0; i<hilights.size(); i++) {
				if(hilights[i]<slots.size() && slots[hilights[i]]!=NULL)
					slots[hilights[i]]->takeInput(in);
				/* if(ret!=NULL) {
					 push(ret);
					 break;
					 }*/
			}
			refresh();
		} else
			setNext(cmdstack.top()->takeInput(s));
	}
}

bool Controller::setNext(ControlBase* next) {
	if(next==NULL)
		pop();
	else if(next!=cmdstack.top())
		push(next);
	return true;
}

void Controller::activate() {
	SharedObject<LedMC> leds;
	leds->setWeights(~FaceLEDMask,0);
	leds->setWeights(FaceLEDMask,.75);
	display=motman->addMotion(leds,MotionManager::kEmergencyPriority);
	erouter->addTrapper(this,EventBase::buttonEGID);
	if(!cmdstack.empty())
		cmdstack.top()->activate(display,gui_comm);
	else
		chkCmdStack();
}

void Controller::deactivate() {
	motman->removeMotion(display);
	//these two lines help prevent residual display in case that was the only MotionCommand using LEDs
	for(unsigned int i=LEDOffset; i<LEDOffset+NumLEDs; i++)
		motman->setOutput(NULL,i,0.f);
	display=MotionManager::invalid_MC_ID;
	erouter->removeTrapper(this);
	cmdstack.top()->pause();
}

bool Controller::chkCmdStack() {
	if(cmdstack.empty()) {
		if(root==NULL)
			return false;
		cmdstack.push(root);
		ControlBase * next = cmdstack.top()->activate(display,gui_comm);
		if(next==NULL)
			cout << "*** WARNING Controller root returned NULL on activate!" << endl;
		else if(next!=root)
			push(next);
	}
	return true;
}

std::string Controller::makeLower(const std::string& s) {
	std::string ans;
	ans.reserve(s.size());
	unsigned int i=s.size();
	while(i--!=0)
		ans+= ::tolower(s[i]);
return ans;
}

std::string Controller::removePrefix(const std::string& str, const std::string& pre) {
	if(str.compare(0,pre.size(),pre)==0)
		return str.substr(pre.size());
	return std::string();
}


/*! @file
 * @brief Implements Controller class, a behavior that should be started whenever the emergency stop goes on to provide menus for robot control
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_3 $
 * $Revision: 1.18 $
 * $State: Exp $
 * $Date: 2003/06/12 18:06:07 $
 */
