/*
 * almMain.cc -- defines the main class for interfacing with the AIBO
 *   Local Map. See almMain.h for details.
 *
 * Started 12-13-2002, tss
 */

#include "Configuration.h"
#include "almMain.h"
#include "almUtility.h"
#include "almStructures.h"
#include "agmMain.h"

#include "../Common/WalkMotionModel.h"
#include "../FastSLAM/afsParticle.h"
#include "../WorldModel2.h"

#ifndef UNIT_TEST_ALM_MA
#include "Shared/WorldState.h"
#include "Vision/Vision.h"
#include "Vision/Visiondefines.h"
#endif

#include <cmath>
#include <iostream>

#ifdef UNIT_TEST_ALM_MA
#warning Compiling unit test code for almMain.cc
#include <fstream>
#include <cstdlib>
#include <ctime>

#define IROORDIST	900
#else
#define IROORDIST	WorldState::IROORDist
#endif

  // Defines for HM_CELL_COUNT, DM_CELL_COUNT, and GM_CELL_COUNT
  // have been moved to Configuration.h

  // Handy math constant: sqrt(2*M_PI)
#define SQRT_2_PI	2.5066282746310002416

  // Here we keep the map data itself, which is declared as a global because
  // the AIBO has a buggy malloc, and anyway we only need one map. I know this
  // is bad, but I'll try to write the code so that it doesn't matter so much.
  //   Here is the horizonal height map. We actually have two so that we can
  // do motion updates with ease--a double buffering scheme, if you will. The
  // pointer HM will refer to the "current" height map
hm_cell HMs[2][HM_CELL_COUNT];
hm_cell *HM;
  //   Here is the spherical depth map. We actually have two so that we can
  // do motion updates with ease--a double buffering scheme, if you will. The
  // pointer DM will refer to the "current" depth map
dm_cell DMs[2][DM_CELL_COUNT];
dm_cell *DM;

  // This method initializes the static tables used to hold the map data.
  // Yes, you have to manually run an initializer. I apologize!
void ALM::init(void)
{
  // clear out both sets of maps and leave the first set as the current one
  // set #2
  DM = DMs[1];
  HM = HMs[1];
  nukeAndPaveCurrentMap();
  // set #1
  DM = DMs[0];
  HM = HMs[0];
  nukeAndPaveCurrentMap();
}

  // This method applies motion to the maps, updating them so that they
  // remain egocentric and so that uncertain regions are duly noted.
  // The parameters are the same as those applied to the motion commands.
void ALM::move(double dx, double dy, double da, unsigned int time)
{
  dm_cell *old_DM = DM;
  hm_cell *old_HM = HM;

  // Make the other set of maps the current map
  if(old_DM == DMs[0]) {
    DM = DMs[1];
    HM = HMs[1];
  }
  else {
    DM = DMs[0];
    HM = HMs[0];
  }

  // Clear out the current map. This clearing out does make some big
  // assumptions for us--whatever we don't fill in from the other map set
  // will be assumed to be level ground beneath us or walls out of range over
  // the horizon.
  nukeAndPaveCurrentMap();

  // We now find the relative motion that the robot underwent during this
  // maneuver
  double rel_x = 0;
  double rel_y = 0;
  double rel_theta = 0;

  WalkMotionModel(&rel_x, &rel_y, &rel_theta, dx, dy, da, time);

  // Good. Now we construct transformation matricies we'll use to translate
  // and rotate the data in the maps according to that relative motion.
  //   For transforming the HM data, we use the following matrix, whose
  // entries correspond to local variables defined below:
  //     [  cosrtheta  sinrtheta  t13_or14 ]
  //     [ -sinrtheta  cosrtheta  t23_or24 ]
  //     [      0          0          1    ]
  //  For transforming the DM data, we use this matrix. Same deal with the
  // entries:
  //     [  cosrtheta  sinrtheta  0  t13_or14 ]
  //     [ -sinrtheta  cosrtheta  0  t23_or24 ]
  //     [      0          0      1      0    ]
  //     [      0          0      0      1    ]
  double sinrtheta = sin(rel_theta);
  double cosrtheta = cos(rel_theta);
  double t13_or14  = -cosrtheta*rel_x - sinrtheta*rel_y;
  double t23_or24  =  sinrtheta*rel_x - cosrtheta*rel_y;

  // Now we actually do the transformation. Let's start with the height map.
  for(int index=0; index<HM_CELL_COUNT; ++index) {
    // Find the xy location of this cell
    double cell_x, cell_y;
    hm_index2xy(index, cell_x, cell_y);

    // Transform!
    double new_cell_x =  cosrtheta*cell_x + sinrtheta*cell_y + t13_or14;
    double new_cell_y = -sinrtheta*cell_x + cosrtheta*cell_y + t23_or24;

    // Find the new index for this transformed location.
    // Make certain the index is in bounds
    int new_index;
    if(!xy2hm_index(new_cell_x, new_cell_y, new_index)) continue;

    // TODO--bring in GM data for this cell?

    // Check the certainty value at the new index. If it's higher than the
    // cell we're moving, then abort--that one is more likely to be correct.
    if(HM[new_index].confidence > old_HM[index].confidence) continue;

    // OK, we're good; copy the cell!
    HM[new_index] = old_HM[index];

    // Diminish the confidence value to reflect motion uncertainty
    HM[new_index].confidence *= ALM_HM_TAX;
  }

  // Now let's transform the depth map. It's basically the same deal as
  // with the height map.
  for(int index=0; index<HM_CELL_COUNT; ++index) {
    // If the depth of this cell is at or greater than IROORDIST, leave
    // it alone--there's effectively no data there, since whatever's off
    // on that radial is out of range
    if(old_DM[index].depth >= IROORDIST) continue;

    // Find the azimuth and altitude of this cell.
    double cell_az, cell_alt;
    dm_index2angles(index, cell_az, cell_alt);

    // Find the XYZ values for the azimuth, altitude, and depth of this cell
    double cell_x, cell_y, cell_z;
    neck_range2xyz(old_DM[index].depth, cell_az, cell_alt,
		   cell_x, cell_y, cell_z);

    // Transform!
    double new_cell_x =  cosrtheta*cell_x + sinrtheta*cell_y + t13_or14;
    double new_cell_y = -sinrtheta*cell_x + cosrtheta*cell_y + t13_or14;
    // z does not change!

    // Find the azimuth, altitude, and range of the new xyz values
    double new_cell_az, new_cell_alt, new_cell_depth;
    xyz2neck_range(new_cell_x, new_cell_y, cell_z,
		   new_cell_depth, new_cell_az, new_cell_alt);

    // Constrain new cell depth to IROORDIST--we really probably shouldn't
    // rely on information that far out. We'll remain confident that it's
    // far away, though.
    if(new_cell_depth > IROORDIST) new_cell_depth = IROORDIST;

    // Now find the corresponding index
    // Make sure the index is in bounds.
    int new_index;
    if(!angles2dm_index(new_cell_az, new_cell_alt, new_index)) continue;

    // Check the depth values at the new index. If it's lower than the
    // depth of the cell we're moving, then abort--occlusion!
    if(DM[new_index].depth < new_cell_depth) continue;

    // OK, we're good, copy the cell and update the depth!
    DM[new_index] = old_DM[index];
    DM[new_index].depth = new_cell_depth;

    // Diminish the confidence value to reflect motion uncertainty
    DM[new_index].confidence = old_DM[index].confidence * ALM_DM_TAX;
  }
}

  // This method clears the current maps to simulate an empty room: the ground
  // will appear in the proper place in the spherical map with the rest empty;
  // the height map will be zeroed everywhere. Actually, in addition to other
  // administrativa, init just calls this function.
void ALM::nukeAndPaveCurrentMap(void)
{
  int i;
  hm_cell hmc;
  dm_cell dmc;

  dmc.confidence = 0.0;

  for(i=0; i<DM_CELL_COUNT; ++i) {
    // We have to specifically set the depth for each cell. If the cell is
    // above the horizon, we say that the depth is out of range. If it's
    // below the borizon, we plot out a flat plane at AIBO height.
    double azimuth, altitude;

    // Find angles for cell i.
    dm_index2angles(i, azimuth, altitude);

    // If altitude > 0, we're peering off into the distance.
    if(altitude > 0) {
      dmc.depth = IROORDIST; // really WorldState::IROORDist--see #define at top
#ifndef UNIT_TEST
      dmc.color = COLOR_BLUE;
#else
      dmc.color = BLUE;	// wall of the pool
#endif
    }
    // Otherwise we're staring at the ground. Find out how far away the
    // ground is, anyway.
    else {
      dmc.depth = AIBO_TILT_PIVOT_HEIGHT / -sin(altitude);
      if(dmc.depth > IROORDIST) dmc.depth = IROORDIST;
#ifndef UNIT_TEST
      dmc.color = COLOR_GREEN; // the ground
#else
      dmc.color = GREEN; // the ground
#endif
    }

    DM[i] = dmc;
  }

  hmc.height = 0.0;
  hmc.trav = 0.5;		// maybe traversable? maybe...
  hmc.confidence = 0.0;
#ifndef UNIT_TEST
  hmc.color = COLOR_GREEN;	// assumption: grass as far as the eye can see
#else
  hmc.color = GREEN;		// assumption: grass as far as the eye can see
#endif

  for(i=0; i<HM_CELL_COUNT; ++i) HM[i] = hmc;
}

  // This method registers a depth measurement in the local maps
void ALM::registerDepth(double depth, double tilt, double pan)
{
  registerDepth(depth, tilt, pan, 0);
} 

  // This method does the same, but it allows the user to specify kludges
  // that can give "better" performance in certain limited environments.
  // See the WM2Kludge namespace definition in WorldModel2.h for details.
void ALM::registerDepth(double depth, double tilt, double pan,
			unsigned int kludges)
{
  double x, y, z;
  double n_depth, n_azimuth, n_altitude;
  int index;

  // The first thing we do is convert the AIBO's depth measurement to a
  // more accurate measurement based on calibration data. We only do this
  // if the measurement is in range. Fortunately, we seem to only need to
  // modify the measurement linearly.
  if(depth < IROORDIST)
    depth = AIBO_IR_CAL_MULTIPLIER*depth + AIBO_IR_CAL_OFFSET;

  // Turn head depth measurement into XYZ coordinates
  head_range2xyz(depth, pan, tilt, x, y, z);

  // If we're ignoring measurements with z < 0, then, well, ignore away.
  if((kludges & WM2Kludge::IgnoreZLessThanZero) && (z < 0.0)) return;

  // Turn XYZ coordinates into neck depth measurement
  xyz2neck_range(x, y, z, n_depth, n_azimuth, n_altitude);

  // Find out how many pixels away from center of the depth measurement on
  // the height map we'll be filling in points. Two standard deviations
  // ought to be plenty.
  int splat_width = (int)(n_depth * 2.0*ALM_IR_SPLAT_STDDEV);

  // Find DM array index for depth measurement. Skip if this measurement is
  // out of DM bounds. TODO--not optimal, since some measurements that are
  // no good for the DM may be good for the HM.
  if(!angles2dm_index(n_azimuth, n_altitude, index)) return;

  // Find a color for this cell
#ifndef UNIT_TEST_ALM_MA
  // TODO-- get rid of this temporary, inefficient center code, replace with
  // something in vision.c that gets the color of the central image cell.
  colortype center_cell_color;
  {
    int i;
    int vx = vision->width / 2;
    int vy = vision->height / 2;
    for(i=0; i<vision->num_runs; ++i)
      if((vision->rmap[i].y == vy) &&
	 (vision->rmap[i].x + vision->rmap[i].width >= vx)) break;
    center_cell_color = vision->rmap[i].color;
  }

  // If we're ignoring measurements of green objects, just quit now.
  if((kludges & WM2Kludge::IgnoreGreenItems) &&
     (center_cell_color == COLOR_GREEN))
    return;
#else
  colortype center_cell_color = GREEN;
#endif

  // Fill in depth measurements in "splat" area around the depth measurement
  for(int dx=-splat_width; dx<=splat_width; ++dx) {
    for(int dy=-splat_width; dy<=splat_width; ++dy) {
      // Compute distance from this index to the center of the splat
      double splat_var = (double)(dx*dx + dy*dy);

      // Ignore any dx or dy greater than two standard deviations away from
      // the center of the splat
      if(splat_var > 4.0*(double)(splat_width*splat_width)) continue;

      // Make certain the dx won't wrap over to the other side of the
      // array
      int x_check = (index % ALM_DM_H_SIZE) + dx;
      if((x_check < 0) || (x_check >= ALM_DM_H_SIZE)) continue;

      // Which part of the splat are we in?
      int splat_index = index + dy*ALM_DM_H_SIZE + dx;
      // Make certain we're not going out of array bounds
      if((splat_index < 0) || (splat_index >= DM_CELL_COUNT)) continue;
      // What's the confidence value for this part of the splat?
      // It's whatever this Gaussian says it is.
      double splat_confidence =
	1/((double)splat_width*SQRT_2_PI)*exp(-splat_var/2.0*(double)splat_var);

      // Compute azimuth, altitude, X, Y, and Z for this cell index. We don't
      // need this for the DM portion of the splat functionality, but we do
      // need it for the HM part. We're just computing it here to see if any
      // of the kludges apply.
      double splat_az, splat_alt;
      double splat_x, splat_y, splat_z;
      dm_index2angles(splat_index, splat_az, splat_alt);
      neck_range2xyz(n_depth, splat_az, splat_alt, splat_x, splat_y, splat_z);

      // See if a kludge mandates us ignoring this part of the splat because it
      // corresponds to a z value less than 0
      if((kludges & WM2Kludge::IgnoreZLessThanZero) && (splat_z < 0.0))
	continue;

      // Start with DM. Make sure this part of the splat is in bounds first
      if((splat_index >= 0) && (splat_index < DM_CELL_COUNT))
	// Only modify this place if we're confident about our measurement
	// being better
	if(splat_confidence > DM[splat_index].confidence) {
	  DM[splat_index].depth		= n_depth;
	  DM[splat_index].confidence	= splat_confidence;
	  DM[splat_index].color		= center_cell_color;
	}

      // We now apply the splat to the horizontal height map. We use the
      // X, Y, and Z values we computed previously. Please note the reuse
      // of the splat_index variable.
      // Only work if this part of the splat is in bounds
      if(xy2hm_index(splat_x, splat_y, splat_index)) {
	// Only modify this place if we're confident about our measurement
	// being better
	if(splat_confidence > HM[splat_index].confidence) {
	  HM[splat_index].height	= splat_z;
	  HM[splat_index].confidence	= splat_confidence;
	  // TODO -- come up with a more intelligent way of setting
	  // traversability than this
	  if((splat_z > AIBO_MAX_BUMP) && (splat_z < AIBO_MIN_CLEARANCE))
	    HM[splat_index].trav	= 0.0;
	  else
	    HM[splat_index].trav	= 1.0;
	  HM[splat_index].color		= center_cell_color;
	}
      }
    }
  }

  // TODO--replace with intelligent sampling
}

  // This method uses the ground plane assumption to fill the local maps
  // with information about the location of the ground based on the location
  // of the head and the image coming from the camera. Head and image
  // information are obtained from WorldState, etc.

  // For now we just assume that the AIBO is staring straight ahead, a
  // condition checked for by the main WorldModel2 code. This is
  // a cop-out, but it gives us a starting place.
void ALM::registerGround()
{
#ifndef UNIT_TEST_ALM_MA
  static int curr_img_width  = 0;
  static int curr_img_height = 0;
  static double cam_depth_wpix = 0;
  static double cam_depth_hpix = 0;
  static int ciw_2 = 0; // curr_img_* / 2
  static int cih_2 = 0;
  static double z = AIBO_TILT_PIVOT_HEIGHT+AIBO_NECK_HEIGHT;

  // To do ground plane assumption calculations, we assume (naively, perhaps)
  // that the AIBO's camera is like a pinhole camera. This may be a safe-ish
  // assumption to make since the FOV is so narrow. Anyway, in any pinhole
  // camera the distance of the film to the pinhole determines the FOV of
  // that particular piece of film--the smaller the distance, the bigger
  // the FOV (try drawing it if you're having trouble seeing it).
  if((vision->width != curr_img_width) || (vision->height != curr_img_height)) {
    curr_img_width  = vision->width;
    curr_img_height = vision->height;

    ciw_2 = curr_img_width/2;
    cih_2 = curr_img_height/2;

    // now, we need to know the distance of the "film" here in the AIBO's
    // camera system. We measure this depth in pixels instead of actual
    // physical units, since pixels can be scaled. To compute the depth,
    // simply do (img_width/2)*tan(horiz_angle_of_view/2) or the analogous
    // computation for image height.
    cam_depth_wpix = ((double) ciw_2)*tan(AIBO_CAM_H_FOV/2);
    // So why do we compute it for both? It turns out that the AIBO's camera
    // doesn't quite have square pixels, at least if we assume it's a pinhole
    // camera. So the depth in horizontal pixels is slightly different from the
    // depth in vertical pixels.
    cam_depth_hpix = ((double) cih_2)*tan(AIBO_CAM_V_FOV/2);
  }

  for(int i=0; i<vision->num_runs; ++i) {
    int dy = vision->rmap[i].y - cih_2; // y distance from center of image
      // Skip if this run is on or above the horizon
    if(dy <= 0) continue;
      // Skip unless this run is green
    if(vision->rmap[i].color != COLOR_GREEN) continue;

      // Find the x distance value for this scanline
      // NOTE: this x is a quite different x from the x in dx!
    double x = cam_depth_hpix / ((double) dy)*z;

    int dx = vision->rmap[i].x - ciw_2;
    int xend = dx+vision->rmap[i].width;
    for(; dx < xend; ++dx) {
      // Find the y value for this pixel
      // NOTE: this y is a quite different y from the y in dy
      double y = ((double) dx)*x / cam_depth_wpix;

      // OK, we've got x, y, and z for this pixel. Now to use it.
      // Fill in height map information
      hm_cell hmc;
      hmc.height = 0.0;
      hmc.trav = 1.0;
      hmc.confidence = ALM_GPA_CONFIDENCE;
      hmc.color = COLOR_GREEN;

      int index;
      if(xy2hm_index(x, y, index)) HM[index] = hmc;

      // Fill in depth map information
      double az, alt, depth;
      xyz2neck_range(x, y, z, depth, az, alt);
      dm_cell dmc;
      dmc.depth = depth;
      dmc.confidence = ALM_GPA_CONFIDENCE;
      dmc.color = COLOR_GREEN;

      if(angles2dm_index(az, alt, index)) DM[index] = dmc;
    }
  }
#endif
}

  // This method copies the information from the HM onto the global map
  // (or, more poetically, turns the HM into a rubber stamp).
  // You need to furnish it with the current pose information, which you
  // get from FastSLAM
void ALM::stampHM(afsPose &pose)
{
#ifdef WANT_GLOBAL_MAP
  // For transforming the HM data from egocentric to allocentric coordinates,
  // we use this matrix
  //     [  cosntheta  sinntheta  pose.x ]
  //     [ -sinntheta  cosntheta  pose.y ]
  //     [      0          0         1   ]
  double cosntheta = cos(-pose.theta);
  double sinntheta = sin(-pose.theta);

  for(int i=0; i<HM_CELL_COUNT; ++i) {
    double lx, ly;

    hm_index2xy(i, lx, ly);
      // make certain we're in the circle guaranteed to contain valid
      // local HM data
    if((lx*lx + ly*ly) > ALM_HM_RADIUS*ALM_HM_RADIUS) continue;

      // we're in. transform x and y values to global coordinates.
    double gx =  cosntheta*lx + sinntheta*ly + pose.x;
    double gy = -sinntheta*lx + cosntheta*ly + pose.y;

    AGM::carryOver(gx, gy, HM[i]);
  }
#endif
}

  // This method generates motion requests for WorldModel2. It's the reason
  // WorldModel2 is our friend.
  // For the time being, the method used here is K-means clustering.
  // Hope you like loops!
void ALM::genRequests(MRvector &requests)
{
    // useful storage space
  MotionRequest request;
    // find handy constants
  static const double d_az   = ALM_DM_LEFT - ALM_DM_RIGHT;
  static const double d_alt  = ALM_DM_TOP - ALM_DM_BOTTOM;
  static const double d_edge = 2*ALM_HM_SIZE;

    ////////////////// Do K-means for the DM first
    //////////////////
  { // BEGIN of enclosing block, not indented to preserve screen space
  double dm_centers[ALM_DM_NUMCLUSTERS][2]; // would use new, but bad malloc
    // Randomly distribute centers through DM
  for(int i=0; i<ALM_DM_NUMCLUSTERS; ++i) {
    dm_centers[i][0] = ALM_DM_LEFT + (double) rand() * d_az / (double) RAND_MAX;
    dm_centers[i][1] = ALM_DM_TOP + (double) rand() * d_alt / (double) RAND_MAX;
  }

    // The actual K-means bit
  for(int iteration=0; iteration<AM_KMEANS_ITERATIONS; ++iteration) {
      // Array keeps track of how popular each cluster center is in terms
      // of member uncertainty
    double popularity[ALM_DM_NUMCLUSTERS];
    for(int i=0; i<ALM_DM_NUMCLUSTERS; ++i) popularity[i] = 0.0;

      // Tag all of the points as belonging to a specific cluster
    for(int i=0; i<DM_CELL_COUNT; ++i) {
      double az, alt, bestdist = HUGE_VAL;
      dm_index2angles(i, az, alt);  // get az and alt for this index

	// Find out which cluster center is best for this point
      for(int cindex=0; cindex<ALM_DM_NUMCLUSTERS; cindex++) {
	double raz, ralt, dist;

	raz  = az  - dm_centers[cindex][0]; // get relative azimuth
	ralt = alt - dm_centers[cindex][1]; // get relative altitude
        dist = raz*raz + ralt*ralt;         // get relative distance squared

	  // is this point the best seen yet?
	if(dist < bestdist) { bestdist = dist; DM[i].cluster = cindex; }
      }
	// increment chosen cluster center popularity--more uncertain
	// members mean greater popularity!
      popularity[DM[i].cluster] += 1 - DM[i].confidence;
    }

      // Now shift the cluster centers to the weighted center of their clusters.
      // Start by resetting everything to 0
    for(int cindex=0; cindex<ALM_DM_NUMCLUSTERS; cindex++) {
      dm_centers[cindex][0] = 0.0;
      dm_centers[cindex][1] = 0.0;
    }
      // Now add the influence of all the members of the DM
    for(int i=0; i<DM_CELL_COUNT; ++i) {
      double az, alt, uncertainty;
      dm_index2angles(i, az, alt);  // get az and alt for this index
      uncertainty = 1 - DM[i].confidence; // get uncertainty for this index

      dm_centers[DM[i].cluster][0] +=
	 az * uncertainty / popularity[DM[i].cluster];
      dm_centers[DM[i].cluster][1] +=
	alt * uncertainty / popularity[DM[i].cluster];
    }
  }

    // K-means is finished for the DM. Store results in the requests vector
  request.type = MotionRequest::LOOK_AT;
  for(int cindex=0; cindex<ALM_DM_NUMCLUSTERS; cindex++) {
    request.azalt.azimuth  = dm_centers[cindex][0];
    request.azalt.altitude = dm_centers[cindex][1];
    requests.push_back(request);
  }
  } // END of enclosing block


    ////////////////// Do K-means for the HM next
    //////////////////
#ifdef UNIT_TEST_ALM_MA
  // explicitly clean out HM cluster assignments first to get rid of
  // bad values outside of ALM_HM_RADIUS. Only need to do this for the
  // sake of frazzled unit testers--everyone else is on their own!
  for(int i=0; i<HM_CELL_COUNT; ++i) HM[i].cluster = -1;
#endif

  { // BEGIN of enclosing block, not indented to preserve screen space
  double hm_centers[ALM_HM_NUMCLUSTERS][2]; // would use new, but bad malloc
    // Randomly distribute centers through HM
  for(int i=0; i<ALM_HM_NUMCLUSTERS; ++i) {
    double x, y;
    do {
      x = -ALM_HM_SIZE + (double)rand() * d_edge / (double) RAND_MAX;
      y = -ALM_HM_SIZE + (double)rand() * d_edge / (double) RAND_MAX;
      // constrain to circle. there are better ways to do this.
    } while((x*x + y*y) > ALM_HM_RADIUS*ALM_HM_RADIUS);
    hm_centers[i][0] = x;
    hm_centers[i][1] = y;
  }

    // The actual K-means bit
  for(int iteration=0; iteration<AM_KMEANS_ITERATIONS; ++iteration) {
      // Array keeps track of how popular each cluster center is in terms
      // of member uncertainty
    double popularity[ALM_HM_NUMCLUSTERS];
    for(int i=0; i<ALM_HM_NUMCLUSTERS; ++i) popularity[i] = 0.0;

      // Tag all of the points as belonging to a specific cluster
    for(int i=0; i<HM_CELL_COUNT; ++i) {
      double x, y, bestdist = HUGE_VAL;
      hm_index2xy(i, x, y);	// get x and y for this index
        // ignore points outside of circle
      if((x*x + y*y) > ALM_HM_RADIUS*ALM_HM_RADIUS) continue;

	// Find out which cluster center is best for this point
      for(int cindex=0; cindex<ALM_HM_NUMCLUSTERS; cindex++) {
	double rx, ry, dist;

	rx   = x - hm_centers[cindex][0]; // get relative x value
	ry   = y - hm_centers[cindex][1]; // get relative y value
        dist = rx*rx + ry*ry;             // get relative distance squared

	  // is this point the best seen yet?
	if(dist < bestdist) { bestdist = dist; HM[i].cluster = cindex; }
      }
	// increment chosen cluster center popularity--more uncertain
	// members mean greater popularity!
      popularity[HM[i].cluster] += 1 - HM[i].confidence;
    }

      // Now shift the cluster centers to the weighted center of their clusters.
      // Start by resetting everything to 0
    for(int cindex=0; cindex<ALM_HM_NUMCLUSTERS; cindex++) {
      hm_centers[cindex][0] = 0.0;
      hm_centers[cindex][1] = 0.0;
    }
      // Now add the influence of all the members of the HM
    for(int i=0; i<HM_CELL_COUNT; ++i) {
      double x, y, uncertainty;
      hm_index2xy(i, x, y);	// get x and y for this index
        // ignore points outside of circle
      if((x*x + y*y) > ALM_HM_RADIUS*ALM_HM_RADIUS) continue;
      uncertainty = 1 - HM[i].confidence; // get uncertainty for this index

      hm_centers[HM[i].cluster][0] +=
	x * uncertainty / popularity[HM[i].cluster];
      hm_centers[HM[i].cluster][1] +=
	y * uncertainty / popularity[HM[i].cluster];
    }
  }

    // K-means is finished for the HM. Store results in the requests vector
  request.type = MotionRequest::LOOK_DOWN_AT;
  for(int cindex=0; cindex<ALM_HM_NUMCLUSTERS; cindex++) {
    request.xy.x = hm_centers[cindex][0];
    request.xy.y = hm_centers[cindex][1];
    requests.push_back(request);
  }
  } // END of enclosing block
}

void ALM::dumpDM(dmPicker &p, std::ostream &out)
{
  for(int y=0; y<ALM_DM_V_SIZE; ++y) {
    int yi = y*ALM_DM_H_SIZE;
    out << p(DM[yi]);
    for(int x=1; x<ALM_DM_H_SIZE; ++x) out << '\t' << p(DM[++yi]);
    out << std::endl;
  }
}

void ALM::dumpHM(hmPicker &p, std::ostream &out)
{
  for(int y=0; y<2*ALM_HM_SIZE; ++y) {
    int yi = y*2*ALM_HM_SIZE;
    out << p(HM[yi]);
    for(int x=1; x<2*ALM_HM_SIZE; ++x) out << '\t' << p(HM[++yi]);
    out << std::endl;
  }
}

  // These methods provide direct access to the local map data arrays. You
  // should use DM_CELL_COUNT and HM_CELL_COUNT to index these arrays.
  // These methods are PRIVATE and are intended only for use by WorldModel2
  // objects.
dm_cell *ALM::getDM() { return DM; }
hm_cell *ALM::getHM() { return HM; }


  // What this thing does with all this information remains to be seen...

#ifdef UNIT_TEST_ALM_MA
int main(int argc, char **argv)
{
  using namespace std;

  cout << "Puppies!!" << endl;
  /* Now that we've got that out of our system... */

  // 10 RANDOMIZE TIMER
  srand(time(NULL));
  srand48(time(NULL));

  /* ignition! */
  ALM::init();

  /* try some depth measurements */
  for(float t=ALM_DM_LEFT/2; t>ALM_DM_RIGHT/2;
      t += (ALM_DM_RIGHT-ALM_DM_LEFT)/2/1000)
    ALM::registerDepth(100, 0, t);

  // cluster!
  cout << "Clustering..." << endl;
  MRvector v;
  ALM::genRequests(v);

  //ALM::move(0, 50, 0, 1000);
  ALM::move(0, 0, 1.12, 1000);

  // Stamp it onto the HM
  afsPose p;
  p.x = 0;
  p.y = 100;
  p.theta = 0;
  ALM::stampHM(p);

  // DUMP
  hmPickHeight hp;
  hmPickTrav ht;
  hmPickConfidence hn;
  hmPickColor hc;
  hmPickCluster hl;
  dmPickDepth dd;
  dmPickConfidence dn;
  dmPickColor dc;
  dmPickCluster dl;
  ofstream gh("gHeight.mat"); AGM::dump(hp, gh); gh.close();
  ofstream ih("hHeight.mat"); ALM::dumpHM(hp, ih); ih.close();
  ofstream it("hTrav.mat"); ALM::dumpHM(ht, it); it.close();
  ofstream in("hConfidence.mat"); ALM::dumpHM(hn, in); in.close();
  ofstream ic("hColor.mat"); ALM::dumpHM(hc, ic); ic.close();
  ofstream il("hCluster.mat"); ALM::dumpHM(hl, il); il.close();
  ofstream jd("dDepth.mat"); ALM::dumpDM(dd, jd); jd.close();
  ofstream jn("dConfidence.mat"); ALM::dumpDM(dn, jn); jn.close();
  ofstream jc("dColor.mat"); ALM::dumpDM(dc, jc); jc.close();
  ofstream jl("dCluster.mat"); ALM::dumpDM(dl, jl); jl.close();

  return 0;
}
#endif
