/*
 * agmMain.cc -- defines the main class for interfacing with the AIBO
 *   Global (height) Map. See agmMain.h for details.
 *
 * Started 1-21-2002, tss
 */

#include "Configuration.h"
#include "agmMain.h"
#include "almStructures.h"
#include "almUtility.h"

#include "../WorldModel2.h"

#include <cmath>
#include <iostream>

#ifdef UNIT_TEST_AGM_MA
#warning Compiling unit test code for agmMain.cc
#include <fstream>
#include <cstdlib>
#include <ctime>
#endif

  // Here we keep the map data itself, which is declared as a global because
  // the AIBO has a buggy malloc, and anyway we only need one map. I know this
  // is bad, but I'll try to write the code so that it doesn't matter so much.
  //   Here is the big global map.
hm_cell GM[GM_CELL_COUNT];

  // This method initializes the static table used to hold the map data.
  // Yes, you have to manually run an initializer. I apologize!
void AGM::init(void)
{
  int i;
  hm_cell gmc;

  gmc.height = 0.0;
  gmc.trav = 0.5;		// maybe traversable? maybe...
  gmc.confidence = 0.0;
#ifndef UNIT_TEST
  gmc.color = COLOR_GREEN;	// assumption: grass as far as the eye can see
#else
  gmc.color = GREEN;		// assumption: grass as far as the eye can see
#endif
  gmc.cluster = 0;		// just set cluster to 0

  for(i=0; i<GM_CELL_COUNT; ++i) GM[i] = gmc;
}

  // Carry over information from the local height map. More literally,
  // see about placing height map cell at x,y. We may not do it if data
  // there are from a measurement we're more confident about.
void AGM::carryOver(double x, double y, hm_cell &cell)
{
  int i;

    // Get an index into the global height map.
    // Skip if we're out of bounds
  if(!xy2gm_index(x, y, i)) return;

    // yeppers. Go with the higher confidence cell
  if(cell.confidence > GM[i].confidence) GM[i] = cell;
}

  // Tax the certainty of the global map after a move
void AGM::decay()
{
  for(int i=0; i<GM_CELL_COUNT; ++i) GM[i].confidence *= ALM_GM_TAX;
}

  // This method generates motion requests for WorldModel2. It's the reason
  // WorldModel2 is our friend.
  // For the time being, the method used here is K-means clustering.
  // Hope you like loops!
void AGM::genRequests(MRvector &requests)
{
    // useful storage space
  MotionRequest request;
    // find handy constants
  static const double d_x = AGM_LEFT - AGM_RIGHT;
  static const double d_y = AGM_TOP  - AGM_BOTTOM;
    // Store the cluster centers
  double gm_centers[AGM_NUMCLUSTERS][2]; // would use new, but bad malloc

    // Randomly distribute centers through GM
  for(int i=0; i<AGM_NUMCLUSTERS; ++i) { 
    gm_centers[i][0] = AGM_LEFT + (double)rand() * d_x / (double) RAND_MAX;
    gm_centers[i][1] = AGM_TOP  + (double)rand() * d_y / (double) RAND_MAX;
  }

    // The actual K-means bit
  for(int iteration=0; iteration<AM_KMEANS_ITERATIONS; ++iteration) {
      // Array keeps track of how popular each cluster center is in terms
      // of member uncertainty
    double popularity[AGM_NUMCLUSTERS];
    for(int i=0; i<AGM_NUMCLUSTERS; ++i) popularity[i] = 0.0;

      // Tag all of the points as belonging to a specific cluster
    for(int i=0; i<GM_CELL_COUNT; ++i) {
      double x, y, bestdist = HUGE_VAL;
      gm_index2xy(i, x, y);     // get x and y for this index

        // Find out which cluster center is best for this point
      for(int cindex=0; cindex<AGM_NUMCLUSTERS; cindex++) {
	double rx, ry, dist;

	rx   = x - gm_centers[cindex][0]; // get relative x value
	ry   = y - gm_centers[cindex][1]; // get relative y value
	dist = rx*rx + ry*ry;             // get relative distance squared

	  // is this point the best seen yet?
	if(dist < bestdist) { bestdist = dist; GM[i].cluster = cindex; }
      }
	// increment chosen cluster center popularity--more uncertain
	// members mean greater popularity!
      popularity[GM[i].cluster] += 1 - GM[i].confidence;
    }

      // Now shift the cluster centers to the weighted center of their clusters.      // Start by resetting everything to 0
    for(int cindex=0; cindex<AGM_NUMCLUSTERS; cindex++) {
      gm_centers[cindex][0] = 0.0;
      gm_centers[cindex][1] = 0.0;
    }
      // Now add the influence of all the members of the GM
    for(int i=0; i<GM_CELL_COUNT; ++i) {
      double x, y, uncertainty;
      gm_index2xy(i, x, y);     // get x and y for this index
      uncertainty = 1 - GM[i].confidence; // get uncertainty for this index

      gm_centers[GM[i].cluster][0] +=
	x * uncertainty / popularity[GM[i].cluster];
      gm_centers[GM[i].cluster][1] +=
	y * uncertainty / popularity[GM[i].cluster];
    }
  }

    // K-means is finished for the GM. Store results in the requests vector
  request.type = MotionRequest::GO_TO;
  for(int cindex=0; cindex<AGM_NUMCLUSTERS; cindex++) {
    request.xy.x = gm_centers[cindex][0];
    request.xy.y = gm_centers[cindex][1];
    requests.push_back(request);
  }
}

void AGM::dump(hmPicker &p, std::ostream &out)
{
  for(int y=0; y<AGM_V_SIZE; ++y) {
    int yi = y*AGM_H_SIZE;
    out << p(GM[yi]);
    for(int x=1; x<AGM_H_SIZE; ++x) out << '\t' << p(GM[++yi]);
    out << std::endl;
  }
}

  // This method provides direct access to the global map data array. You
  // should use GM_CELL_COUNT to index the array. This method is PRIVATE
  // and is intended only for use by WorldModel2 objects.
hm_cell *AGM::getGM() { return GM; }


#ifdef UNIT_TEST_AGM_MA
int main(int argc, char **argv)
{
  using namespace std;

  // 10 RANDOMIZE TIMER
  srand(time(NULL));
  srand48(time(NULL));

  // initialize AGM
  AGM::init();

  // OK, now randomly spatter around some uncertainty
  hm_cell gmc;
  for(int i=0; i<3000000; ++i) {
    float x = drand48()*4000 - 2000;
    float y = drand48()*4000 - 2000;

    gmc.height = drand48()*4;
    gmc.trav = 1 - drand48()*0.2;
    gmc.confidence = drand48()*0.2 + 0.2;
    gmc.color = ORANGE;

    AGM::carryOver(x, y, gmc);
  }

  // OK, put certainty in select spots
  for(int i=0; i<800000; ++i) {
    float x = drand48()*1333 - 2000;
    float y = drand48()*1333 - 2000;
    gmc.confidence = drand48()*0.2 + 0.8;
    AGM::carryOver(x, y, gmc);
  }
  for(int i=0; i<800000; ++i) {
    float x = drand48()*1333 + 666;
    float y = drand48()*1333 - 2000;
    gmc.confidence = drand48()*0.2 + 0.8;
    AGM::carryOver(x, y, gmc);
  }
  for(int i=0; i<800000; ++i) {
    float x = drand48()*1333 - 2000;
    float y = drand48()*1333 + 666;
    gmc.confidence = drand48()*0.2 + 0.8;
    AGM::carryOver(x, y, gmc);
  }
  for(int i=0; i<800000; ++i) {
    float x = drand48()*1333 + 666;
    float y = drand48()*1333 + 666;
    gmc.confidence = drand48()*0.2 + 0.8;
    AGM::carryOver(x, y, gmc);
  }
  for(int i=0; i<800000; ++i) {
    float x = drand48()*1333 - 666;
    float y = drand48()*1333 - 666;
    gmc.confidence = drand48()*0.2 + 0.8;
    AGM::carryOver(x, y, gmc);
  }

  // cluster!
  cout << "Clustering..." << endl;
  MRvector v;
  AGM::genRequests(v);

  // write files to ensure everything's in place off the bat
  cout << "Saving..." << endl;
  cout.flush();
  hmPickHeight hp;
  hmPickTrav ht;
  hmPickConfidence hn;
  hmPickColor hc;
  hmPickCluster hl;
  ofstream ih("iHeight.mat"); AGM::dump(hp, ih); ih.close();
  ofstream it("iTrav.mat"); AGM::dump(ht, it); it.close();
  ofstream in("iConfidence.mat"); AGM::dump(hn, in); in.close();
  ofstream ic("iColor.mat"); AGM::dump(hc, ic); ic.close();
  ofstream il("iCluster.mat"); AGM::dump(hl, il); il.close();

  return 0;
}

#endif
