/*
 * Triangulator code for AIBO FastSLAM. Takes in the locations and
 * values of two bearings-only measurements and constructs a Gaussian for
 * the probable location of the landmark.
 */

#ifdef __cplusplus
extern "C" {
#endif

#include "afsParticle.h"
#include "afsUtility.h"

#include <math.h>

#ifdef UNIT_TEST
#include <stdio.h>
#endif

/* To actually generate the gaussians, we need a way to perform triangulations
 * based on two bearing measurements from different locations. The arguments
 * to this routine consist of the measurements (first six args) and the
 * place to store the triangulated location information (last two).
 * We assume that the calling function has already ensured that the angles
 * given to us will be easy to triangulate. */
void triangulate(double x1, double y1, double theta1,
		 double x2, double y2, double theta2,
		 double *pos_x, double *pos_y)
{
  double ux1, uy1, ux2, uy2, t;

  /* The first thing we need to do is convert the ray form of these lines
   * into vector form. The u(x|y)(1|2) variables define linear subspaces
   * along our lines of sight. */
  ux1 = cos(theta1);
  uy1 = sin(theta1);
  ux2 = cos(theta2);
  uy2 = sin(theta2);

  /* Paul Bourke rules. His webpage can tell you how to find out anything
   * interesting. Go to http://astronomy.swin.edu.au/~pbourke/ and see.
   * This code computes how far from x1, y1 in terms of ux1, uy1 the
   * intersection is. */
  t = (ux2*(y1-y2) - uy2*(x1-x2)) / (uy2*ux1 - ux2*uy1);

  /* From this, we get the actual intersection */
  *pos_x = x1 + t*ux1;
  *pos_y = y1 + t*uy1;
}

/* The whole triangulator routine itself. See the header file for more
 * information */
int afsTriangulator(double x1, double y1, double theta1,
		    double x2, double y2, double theta2,
		    afsLandmarkLoc *landmark)
{
  /* Our stack frame is BIG. Uh-uh, it ain't ashamed... */
  double dtheta;
  double mean_x, mean_y;
  double variance_x, variance_y, variance_xy;
  double corner_x, corner_y;

  /* Make certain the two observations are far enough apart for us to
   * triangulate meaningfully. */
  dtheta = find_dtheta(theta1, theta2);
  if(dtheta < AFS_MIN_TRIANG_ANGLE) return 0;
  if(dtheta > AFS_MAX_TRIANG_ANGLE) return 0;

  /* Now we do that thing that we do. Find the mean of the Gaussian made
   * by the two measurements. */
  triangulate(x1,y1,theta1, x2,y2,theta2, &mean_x, &mean_y);

  /* The variance of this Gaussian is derived from the intersections of
   * the variance lines of the two measurements. We take the points of
   * these intersections and use them as training data for a maximum
   * likelihood Gaussian with a mean at the intersection calculated above.
   * This is a fancy way of saying that we take the x and y locations of
   * these points relative to the mean, multiply them by themselves to
   * make a 2x2 matrix, and averages the matrices made for all four points.
   * to get the final covariance matrix. */
  triangulate(x1,y1,theta1+AFS_MEASURE_VARIANCE,
	      x2,y2,theta2-AFS_MEASURE_VARIANCE, &corner_x, &corner_y);
  corner_x -= mean_x;
  corner_y -= mean_y;
  variance_x = corner_x * corner_x;
  variance_y = corner_y * corner_y;
  variance_xy = corner_x * corner_y;
#ifdef UNIT_TEST
printf("plot(%f, %f, 'bx')\n", corner_x, corner_y);
#endif

  triangulate(x1,y1,theta1-AFS_MEASURE_VARIANCE,
	      x2,y2,theta2+AFS_MEASURE_VARIANCE, &corner_x, &corner_y);
  corner_x -= mean_x;
  corner_y -= mean_y;
  variance_x += corner_x * corner_x; /* note addition, not assignment */
  variance_y += corner_y * corner_y;
  variance_xy += corner_x * corner_y;
#ifdef UNIT_TEST
printf("plot(%f, %f, 'bx')\n", corner_x, corner_y);
#endif

  triangulate(x1,y1,theta1+AFS_MEASURE_VARIANCE,
	      x2,y2,theta2+AFS_MEASURE_VARIANCE, &corner_x, &corner_y);
  corner_x -= mean_x;
  corner_y -= mean_y;
  variance_x += corner_x * corner_x;
  variance_y += corner_y * corner_y;
  variance_xy += corner_x * corner_y;
#ifdef UNIT_TEST
printf("plot(%f, %f, 'bx')\n", corner_x, corner_y);
#endif

  triangulate(x1,y1,theta1-AFS_MEASURE_VARIANCE,
	      x2,y2,theta2-AFS_MEASURE_VARIANCE, &corner_x, &corner_y);
  corner_x -= mean_x;
  corner_y -= mean_y;
  variance_x += corner_x * corner_x;
  variance_y += corner_y * corner_y;
  variance_xy += corner_x * corner_y;
#ifdef UNIT_TEST
printf("plot(%f, %f, 'bx')\n", corner_x, corner_y);
#endif

  /* Check for degeneracy and fail if we see it. */
  if((variance_x < 0.0001) || (variance_y < 0.0001)) return 0;

  /* Fill in the variance entries in the landmark and do division. */
  landmark->variance.x = variance_x * AFS_COVARIANCE_FUDGE / 4;
  landmark->variance.y = variance_y * AFS_COVARIANCE_FUDGE / 4;
  landmark->variance.xy = variance_xy * AFS_COVARIANCE_FUDGE / 4;

  /* Fill in the mean entries in the landmark. */
  landmark->mean.x = mean_x;
  landmark->mean.y = mean_y;

  /* Success! */
  return 1;
}

#ifdef UNIT_TEST

int main(int argc, char **argv)
{
  double x1, y1, x2, y2, theta1, theta2;
  afsLandmarkLoc l;

  fputs("x1: ", stderr);
  scanf("%lf", &x1);
  fputs("y1: ", stderr);
  scanf("%lf", &y1);
  fputs("theta1: ", stderr);
  scanf("%lf", &theta1);
  fputs("x2: ", stderr);
  scanf("%lf", &x2);
  fputs("y2: ", stderr);
  scanf("%lf", &y2);
  fputs("theta2: ", stderr);
  scanf("%lf", &theta2);

  theta1 *= M_PI/180;
  theta2 *= M_PI/180;

  puts("hold on");
  if(afsTriangulator(x1, y1, theta1, x2, y2, theta2, &l)) {
    printf("C = [%f %f; %f %f];\n",
      l.variance.x, l.variance.xy, l.variance.xy, l.variance.y);
    puts("plot_ellipse(C)");
  }
  else puts("Failure.");

  return 0;
}

#endif

#ifdef __cplusplus
}
#endif
