/*
 * Main interface to the AIBO FastSLAM.
 */

#ifndef _AFS_MAIN_H_
#define _AFS_MAIN_H_

#include "afsParticle.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Initialize everything */
void afsInit();

/* Manually set a landmark location estimate. landmark is the landmark
 * to set, x and y are the location, and covariance is the spherical
 * covariance to set about the landmark. You should probably set it to
 * something small, like 500. */
void afsSetLandmark(int landmark, double x, double y, double covariance);

/* Distribute the particle robot location estimates uniformly throughout
 * the specified bounding box. Recommended for use immediately after
 * afsSetLandmark is used to set all landmark locations and before any
 * measurement updates has taken place. The tighter the bounding box,
 * the better the performance. This function with the prior function
 * essentially reduces FastSLAM to Monte Carlo Localization. */
/* lx = left x, ty = top y, rx = right x, by = bottom y */
void afsDistribute(double lx, double ty, double rx, double by);

/* Report a movement */
void afsMotion(double dx, double dy, double da, unsigned int time);

/* Report a sensor measurement. This routine will not resample the particles!
 * Use afsResample() for that. */
void afsMeasurement(int landmark, double theta);

/* Resamples the particles. */
void afsResample();

/* Get the particle that is most representative of where we are and what the
 * world looks like. This particle may be synthesized or it may be taken
 * directly out of the particle list. Don't modify it, please. The error for
 * this particle will be put into the variable pointed to by error, so long
 * as the address isn't NULL. */
afsParticle *afsWhatsUp(double *error);

/* Get an indication of how certain the FastSLAM algorithm is about its
 * location. The lower the number, the better. Particles with unprimed
 * landmarks will tend to lower the error, but with more priming, things
 * become more reasonable */
double afsCertainty();

#ifdef __cplusplus
}
#endif
#endif

