//-*-c++-*-
#ifndef INCLUDED_MotionSequenceMC_h_
#define INCLUDED_MotionSequenceMC_h_

#include "MotionCommand.h"
#include "MotionManager.h"
#include "Shared/LoadSave.h"
#include "Shared/ListMemBuf.h"
#include "PostureEngine.h"

#include <iostream>

//! A handy little (or not so little) class for switching between a sequence of postures
/*! Outputs are handled independently.  It's easy to add keyframes
 *  which modify all of the outputs, but since each output is tracked
 *  individually, OutputCmd's with 0 weight can be used to not affect
 *  other motions.  For instance, pan the head left to right while
 *  moving the right leg up and down several times, you won't have to
 *  specify the position of the head in its motion at each of the leg
 *  motion keyframes.
 *
 *  Be aware that the 0 time frame will be replaced on a call to
 *  play() with the current body posture.  However, this only applies
 *  to  outputs which have a non-zero weighted frame defined at some
 *  point.  The weights, of the 0 time frame will remain unchanged.  
 *  These weights are initially set to 0, so that it's
 *  possible to 'fade in' the first frame of the motion sequence from
 *  whereever the body happens to be (or already doing)
 *
 *  To fade out at the end, set a frame with 0 weight for everything.
 *  Otherwise it will simply die suddenly.  (Currently, will still jerk
 *  between priority level shifts - can only fade within a priority level)
 *
 *  Currently, MotionSequence's are intended mainly for building, 
 *  not editing.  It's easy to add keyframes, but hard/impossible to
 *  delete them.
 *
 *  The MotionSequence base class is an abstract class so that you can
 *  create memory efficient motion sequences and simply refer to them
 *  by the common base class instead of having to worry about the
 *  actual size allocated in the template, MotionSequenceMC.
 *  
 *  The file format used is as follows: ('<' and '>' are not meant literally)
 *  @verbatim
 *       First line: #MSq
 *  Zero or more of: delay <time-delta>              (moves playhead forward, in milliseconds)
 *               or: settime <time>                  (sets play time to specified value, in ms)
 *               or: <outputname> <value> [<weight>] (sets the specified output to the value - assumes 1 for weight)
 *               or: load <filename>                 (file is a posture, sets position)
 *               or: overlay <filename>              (file can be a posture or another motion sequence)
 *               or: degrees                         (following <value>s will be interpreted as degrees [default])
 *               or: radians                         (following <value>s will be interpreted as radians)
 *        Last line: #END
 *  @endverbatim
 *  After loading a motion sequence, the playtime is left at the end.
 *  This is to make it easy to append/overlay motion sequences
 *  
 *  Lines beginning with '#' are ignored.  Output names are defined in RobotInfo.h, RobotInfo::outputNames.
 */
class MotionSequence : public MotionCommand, public LoadSave {
public:
	//!constructor, will start playing immediately
	MotionSequence() : MotionCommand(), playtime(1), lasttime(0), endtime(0), playspeed(1.0), playing(true), loadSaveMode(M_PI/180) {}
	//!destructor
	virtual ~MotionSequence() {}

	//!@name Inherited from MotionCommand
	virtual int updateOutputs();
	virtual int isDirty() { return isPlaying(); }
	virtual int isAlive() { return (playspeed>0) ? (playtime<=endtime) : (playtime>0); }
	//@}
	
	//!@name LoadSave related
	virtual unsigned int getBinSize() const; //!< inherited, returns the size used to save the sequence
	virtual unsigned int LoadBuffer(const char buf[], unsigned int len); //!< inherited, doesn't clear before loading - call clear yourself if you want to reset, otherwise it will overlay.  Leaves playtime at end of load.
	virtual unsigned int SaveBuffer(char buf[], unsigned int len) const; //!< inherited, saves the motion sequence - will save a flat file - doesn't remember references to other files which were loaded
	void setSaveDegrees() { loadSaveMode=M_PI/180; }       //!< will store angles as degrees on future saves
	bool isSaveDegrees() const { return loadSaveMode!=1; } //!< returns true if will store angles as degrees on future saves
	void setSaveRadians() { loadSaveMode=1; }              //!< will store angles as radians on future saves
	bool isSaveRadians() const { return loadSaveMode==1; } //!< returns true if will store angles as degrees on future saves
	//@}
	
	//!@name Sequence Construction
	virtual void clear()=0; //!< empties out the sequence (constant time operation - faster than a series of pops)
	void setPlayTime(unsigned int x); //!< set the time for both playback and editing (in milliseconds)
	void setOutputCmd(unsigned int i, const OutputCmd& cmd); //!< will insert a keyframe for the given output, or change an existing one
	const OutputCmd& getOutputCmd(unsigned int i); //!< gets the value of output @a i at the playhead
	void setPose(const PostureEngine& pose); //!< calls setOutputCmd on each of the OutputCmds in @a pose
	void overlayPose(const PostureEngine& pose); //!< calls setOutputCmd on non-zero weighted OutputCmds in @a pose
	void compress(); //!< compresses the sequence by eliminating sequences of moves which are identical
	virtual unsigned int getMaxFrames() const=0; //!< returns the maximum number of key frames (Move's) which can be stored, determined by the instantiating MotionSequenceMC's template parameter
	virtual unsigned int getUsedFrames() const=0; //!< returns the number of used key frames (Move's) which have been stored by the instantiation MotionSequence subclass
	//@}

	//!@name Playback Control
	bool isPlaying() { return playing && isAlive(); }       //!< returns true if currently playing
	void play();                                            //!< restarts playback from beginning
	void pause() { playing=false; }                         //!< pauses playback until another call to play() or resume()
	void resume();                                          //!< begins playback from the current playtime
	unsigned int getPlayTime() const { return playtime; }   //!< returns the current position of the playback (in milliseconds), see setPlayTime()
	unsigned int getEndTime() const { return endtime; }     //!< returns the length of the motion sequence (in milliseconds)
	void setPlaySpeed(float x) { playspeed=x; }            //!< sets the playback speed (e.g. 1=regular, 0.5=half speed, -1=@b backwards)
	float getPlaySpeed() const { return playspeed; }       //!< returns the playback speed
	//@}

protected:
	// TYPES:
	typedef unsigned short Move_idx_t; //!< type for indexes to move structures in #moves
	static Move_idx_t invalid_move; //!< used to mark the ends of the Move linked lists

	//! This struct holds all the information needed about a frame for a particular output
	struct Move {
		//!constructor
		Move() : cmd(), next(), prev(), starttime(0) {}
		OutputCmd cmd;           //!< the actual command to use
		Move_idx_t next;        //!< the next frame
		Move_idx_t prev;        //!< the previous frame
		unsigned int starttime; //!< the time (relative to first frame) this frame should be expressed at
	};

	// MEMBERS:
	Move_idx_t starts[NumOutputs]; //!< the beginning frame for each output animation
	Move_idx_t prevs[NumOutputs];  //!< the previous frame (the starttime for this frame will always be less than or equal to playtime)
	Move_idx_t nexts[NumOutputs];  //!< the upcoming frame (the starttime for this frame will always be greater than playtime)
	OutputCmd curs[NumOutputs];          //!< merely a cache of current values (if computed, see #curstamps)
	unsigned int curstamps[NumOutputs]; //!< timestamp of corresponding value in #curs
	unsigned int playtime;              //!< the current time of playback, 0 is start of sequence
	unsigned int lasttime;              //!< the time of the last update
	unsigned int endtime;               //!< max of #moves's Move::starttime's
	float playspeed;                    //!< multiplies the difference between current time and starttime, negative will cause play backwards
	bool playing;                       //!< true if playing, false if paused
	
	float loadSaveMode;                 //!< 1 to use radians, M_PI/180 for degrees during a save

	virtual Move& getKeyFrame(Move_idx_t x) =0;            //!< returns the Move struct corresponding to @a x in the subclass's actual data structure
	virtual const Move& getKeyFrame(Move_idx_t x) const=0; //!< returns the Move struct corresponding to @a x in the subclass's actual data structure
	virtual Move_idx_t newKeyFrame()=0;                    //!< causes subclass to create a new Move structure, returns its index
	virtual void eraseKeyFrame(Move_idx_t x)=0;            //!< causes subclass to mark the corresponding Move structure as free

	//!Does the actual calculation of position information.  Perhaps replace with a Bezier or spline or something?
	void calcOutput(OutputCmd& ans, unsigned int t, const Move& prev, const Move& next) const {
		float prevweight=(float)(next.starttime-t)/(float)(next.starttime-prev.starttime);
		ans.set(prev.cmd,next.cmd,prevweight);
	}
	
	//!Sets prev and next to the appropriate values for the given time and output index
	virtual void setRange(unsigned int t,Move_idx_t& prev, Move_idx_t& next) const=0;

	//!used by LoadBuffer()/SaveBuffer(), checks to see if the amount read/written (@a res) is nonzero, increments @a buf, decrements @a len, or displays @a msg if @a is zero
	static bool ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg);

	//!used by LoadBuffer()/SaveBuffer(), checks to see if the amount read/written (@a res) is nonzero, increments @a buf, decrements @a len, or displays @a msg with @a arg1 if @a is zero
	static bool ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg, int arg1);

	//!sets playtime to next time for which any output has a keyframe, -1 if none exists
	unsigned int setNextFrameTime(Move_idx_t p[NumOutputs], Move_idx_t n[NumOutputs]) const;
	
	//!reads a line from a file, parsing it into variables, returns ending position
	static unsigned int readWord(const char buf[], const char * const buflen, char word[], const unsigned int wordlen);

	//!returns the index for the output named in the string or NumOutputs if not found, begins search through RobotInfo::outputName's at index i
	static unsigned int getOutputIndex(const char name[], unsigned int i);
};

//! Instantiates MotionSequences - when you want to make a new MotionSequence, make one of these
/*! Allows a (compile-time) variable amount of data storage through its template parameter.
 *  See MotionSequence for documentation on its members */
template<unsigned int MAXMOVE>
class MotionSequenceMC : public MotionSequence {
public:
	//!constructor
	MotionSequenceMC() : MotionSequence(), moves() {clear();}
	//!constructor, loads from a file and then resets the playtime to beginning and begins to play
	explicit MotionSequenceMC(const char* filename) : MotionSequence(), moves() {clear();LoadFile(filename);setPlayTime(1);}
	//!destructor
	virtual ~MotionSequenceMC() {}

	
	/*struct logent {
		logent(unsigned int t, unsigned int p, unsigned int i, OutputCmd f[NumFrames])
			: time(t), play(p), index(i)
		{
			for(unsigned int j=0;j<NumFrames;j++)
				frames[j]=f[j];
		}
		unsigned int time,play,index;
		OutputCmd frames[NumFrames];
	};

	std::vector<logent> log;

	virtual int isAlive() {
		if(MotionSequenceMC::isAlive())
			return true;
		cout << "LOG:" << endl;
		for(unsigned int i=0; i<log.size(); i++) {
			cout << '#' << '\t' << log[i].time << '\n';
			for(unsigned int j=0; j<NumFrames; j++)
				cout << log[i].play+j*FrameTime << ' ' << log[i].frames[j].value << ' ' << log[i].frames[j].weight << ' ' << log[i].index << '\n';
		}
		cout << endl;
		log.clear();
		return false;
		}*/

	// I put this here because i want direct access to moves so it'll be faster
	virtual int updateOutputs() {
		MotionSequence::updateOutputs();
		if(!isPlaying()) {
			for(unsigned int i=0; i<NumOutputs; i++) //just copies getOutputCmd(i) across frames
				motman->setOutput(this,i,getOutputCmd(i));
		} else {
			for(unsigned int i=0; i<NumOutputs; i++) { //fill out the buffer of commands for smoother movement
				Move_idx_t prev=prevs[i],next=nexts[i];
				OutputCmd frames[NumFrames];
				frames[0]=getOutputCmd(i);
				for(unsigned int t=playtime+FrameTime,j=1;j<NumFrames;j++,t+=FrameTime) {
					setRange(t,prev,next);
					if(next!=invalid_move)
						calcOutput(frames[j],t,moves[prev],moves[next]);
					else
						frames[j].unset();
				}
				motman->setOutput(this,i,frames);
			}
		}
		return NumOutputs;
		//		if(i<NumLegJointss)
		//			log.push_back(logent(get_time(),playtime,i,frames));
	}
	
	virtual void clear() {
		moves.clear();
		for(unsigned int i=0; i<NumOutputs; i++) {
			prevs[i]=starts[i]=moves.new_back();
			moves.back().cmd.unset();
			moves.back().next=invalid_move;
			moves.back().prev=invalid_move;
			nexts[i]=invalid_move;
		}
		setPlayTime(1);
	}
	virtual unsigned int getMaxFrames() const { return moves.getMaxCapacity(); }
	virtual unsigned int getUsedFrames() const { return moves.size(); }

protected:
	// TYPES:
	typedef ListMemBuf<Move,MAXMOVE,Move_idx_t> list_t; //!< shorthand for the ListMemBuf that stores all of the movement frames

	// MEMBERS:
	list_t moves;                       //!< stores all of the movement keyframes

	virtual Move& getKeyFrame(Move_idx_t x) { return moves[x]; }
	virtual const Move& getKeyFrame(Move_idx_t x) const { return moves[x]; }
	virtual Move_idx_t newKeyFrame() { return moves.new_front(); }
	virtual void eraseKeyFrame(Move_idx_t x) { moves.erase(x); }
	void setRange(unsigned int t,Move_idx_t& prev, Move_idx_t& next) const {
		if(next!=invalid_move && moves[next].starttime<=t) {
			do {
				prev=next;
				next=moves[prev].next;
			} while(next!=invalid_move && moves[next].starttime<=t);
		} else if(t<moves[prev].starttime) {
			do {
				next=prev;
				prev=moves[next].prev;
			} while(t<moves[prev].starttime);
		}
	}
};

/*! @file
 * @brief Describes MotionSequence and defines MotionSequenceMC, handy little (or not so little) classes for switching between a sequence of postures
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_2 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2003/04/25 18:22:40 $
 */

#endif
