#include "LedEngine.h"
#include "MotionManager.h"

LEDBitMask_t LedEngine::numMask[11] = {
	BotRLEDMask|BotLLEDMask|TopBrLEDMask, //0
	BotLLEDMask|MidLLEDMask|TopLLEDMask,  //1
	BotRLEDMask|BotLLEDMask|TopLLEDMask|TopBrLEDMask, //2
	BotRLEDMask|BotLLEDMask|MidRLEDMask|TopLLEDMask|TopBrLEDMask, //3
	BotLLEDMask|MidLLEDMask|TopRLEDMask|TopLLEDMask,  //4
	BotRLEDMask|BotLLEDMask|TopRLEDMask|TopBrLEDMask, //5
	BotRLEDMask|BotLLEDMask|MidRLEDMask|MidLLEDMask|TopRLEDMask|TopBrLEDMask, //6
	BotLLEDMask|MidLLEDMask|TopLLEDMask|TopBrLEDMask,  //7
	BotRLEDMask|BotLLEDMask|MidRLEDMask|MidLLEDMask|TopRLEDMask|TopLLEDMask|TopBrLEDMask, //8
	BotLLEDMask|MidLLEDMask|TopRLEDMask|TopLLEDMask|TopBrLEDMask,  //9
	BotLLEDMask //.
};
											 

LedEngine::LedEngine() : numCycling(0), dirty(true), dirtyTime((unsigned int)-1) {
	for(unsigned int i=0; i<NumLEDs; i++) {
		infos[i].flashtime=0;
		infos[i].starttime=0;
	}
	clear();
}

int LedEngine::isDirty() {
	unsigned int t = get_time();
	if(t>dirtyTime) {
		dirty=true;
		dirtyTime=(unsigned int)-1;
		for(unsigned int i=0; i<NumLEDs; i++)
			if(infos[i].flashtime>t && dirtyTime>infos[i].flashtime)
				dirtyTime=infos[i].flashtime;
	}
	return (dirty || numCycling>0);
}

int LedEngine::updateLEDs(const MotionCommand* caller, LEDBitMask_t mask/*=AllLEDMask*/) {
	unsigned int t = get_time();
	for(unsigned int i=0; i<NumLEDs; i++)
		if((mask>>i)&1)
			for(unsigned int f=0; f<NumFrames; f++)
				motman->setOutput(caller, i+LEDOffset,calcValue(i,t+f*FrameTime),f);
	bool tmp=dirty;
	dirty=false;
	return tmp;
}

int LedEngine::updateLEDs(OutputCmd cmds[NumLEDs]) {
	unsigned int t = get_time();
	for(unsigned int i=0; i<NumLEDs; i++)
			cmds[i].value=calcValue(i,t);
	bool tmp=dirty;
	dirty=false;
	return tmp;
}

int LedEngine::updateLEDFrames(OutputCmd cmds[NumLEDs][NumFrames]) {
	unsigned int t = get_time();
	for(unsigned int i=0; i<NumLEDs; i++)
		for(unsigned int f=0; f<NumFrames; f++)
			cmds[i][f].value=calcValue(i,t+f*FrameTime);
	bool tmp=dirty;
	dirty=false;
	return tmp;
}

void LedEngine::invert(LEDBitMask_t leds) {
	if(leds!=0) {
		dirty=true;
		for(unsigned int i=0; i<NumLEDs; i++)
			if((leds>>i)&1)
				if(infos[i].isCycling)
					infos[i].amp*=-1;
				else
					infos[i].value=1-infos[i].value;
	}
}
void LedEngine::set(LEDBitMask_t leds, float value) {
	if(leds!=0) {
		dirty=true;
		for(unsigned int i=0; i<NumLEDs; i++)
			if((leds>>i)&1) {
				infos[i].value=value;
				if(infos[i].isCycling) {
					numCycling--;
					infos[i].isCycling=false;
				}
			}
	}
}
void LedEngine::cflash(LEDBitMask_t leds, float value, unsigned int ms) {
	if(leds!=0) {
		dirty=true;
		unsigned int t = get_time();
		if(t+ms<dirtyTime)
			dirtyTime=t+ms;
		for(unsigned int i=0; i<NumLEDs; i++) {
			infos[i].flashvalue=((leds>>i)&1)*value;
			infos[i].flashtime=t+ms;
		}
	}
}
void LedEngine::flash(LEDBitMask_t leds, unsigned int ms) {
	if(leds!=0) {
		dirty=true;
		unsigned int t = get_time();
		if(t+ms<dirtyTime)
			dirtyTime=t+ms;
		for(unsigned int i=0; i<NumLEDs; i++)
			if((leds>>i)&1) {
				infos[i].flashvalue=calcFlash(calcValue(i,t));
				infos[i].flashtime=t+ms;
			}
	}
}
/*!@param leds the bitmask of leds to apply this to
 * @param period the period of the cycle (milliseconds), includes an on and off
 * @param amp the amplitude of the cycle - note that this is clipped at 0 and 1.
 * @param offset the vertical offset of the cycle - simply shifts the baseline of the cycle up or down
 * @param phase the phase within the cycle to start at (specify in milliseconds)
 *
 * When this function is called, the starting time is stored as current time + phase.
 *
 * The equation used is \f[\cos(\frac{2\pi(t-starttime)}{period})*(\frac{-amp}{2})+.5+offset\f]
 * 
 * The idea is that with a amplitude=1 and offset=0, it will start at
 * 0, ramp up to 1, and then ramp down again.  The arguments to this
 * function will let you control all parameters of the cycle.
 *
 * You can get a blink-on/off instead of cycle on/off by using a very large amplitude.
 */
void LedEngine::cycle(LEDBitMask_t leds, unsigned int period, float amp, float offset, int phase) {
	//	cout << "cycle("<<leds<<","<<period<<","<<amp<<","<<offset<<","<<phase<<")"<<endl;
	if(leds!=0) {
		dirty=true;
		unsigned int start = get_time()+phase;
		for(unsigned int i=0; i<NumLEDs; i++)
			if((leds>>i)&1) {
				if(!infos[i].isCycling)
					numCycling++;
				infos[i].isCycling=true;
				infos[i].amp=amp;
				infos[i].period=period;
				infos[i].starttime=start;
				infos[i].offset=offset;
			}
	}
}
void LedEngine::clear() {
	for(unsigned int i=0; i<NumLEDs; i++) {
		infos[i].value=0;
		infos[i].isCycling=false;
	}
	numCycling=0;
	dirty=true;
}

void LedEngine::displayNumber(int x, numStyle_t style) {
	switch(style) {
	case onedigit:
		if(x>9 || x<-9) {
			ccycle(FaceLEDMask&~TopBrLEDMask,333,10,-5);
			infos[TopBrLEDOffset-LEDOffset].value=x<0?1:0;
		} else {
			clear();
			if(x<0) {
				set(numMask[-x],1);
				infos[TopBrLEDOffset-LEDOffset].value=infos[TopBrLEDOffset-LEDOffset].value*.5+.25;
			} else
				set(numMask[x],1);
		}
		break;
	case twodigit:
		if(x>99 || x<-99) {
			ccycle(FaceLEDMask&~TopBrLEDMask,333,10,-5);
			infos[TopBrLEDOffset-LEDOffset].value=x<0?1:0;
		} else {
			clear();
			if(x<0) {
				infos[TopBrLEDOffset-LEDOffset].value=1;
				x=-x;
			}
			setOneOfTwo(x%10,BotLLEDOffset-LEDOffset,MidLLEDOffset-LEDOffset,TopLLEDOffset-LEDOffset);
			setOneOfTwo(x/10,BotRLEDOffset-LEDOffset,MidRLEDOffset-LEDOffset,TopRLEDOffset-LEDOffset);
		}
		break;
	}
}
void LedEngine::setOneOfTwo(unsigned int x, unsigned int low, unsigned int mid, unsigned int high) {
	if(x==0)
		return;
	float bg = ((x-1)/3)/3.0;
	float fg = bg+.333333333;
	switch(x%3) {
	case 1:
		infos[high].value=bg;
		infos[mid].value=bg;
		infos[low].value=fg;
		break;
	case 2:
		infos[high].value=bg;
		infos[mid].value=fg;
		infos[low].value=bg;
		break;
	case 0:
		infos[high].value=fg;
		infos[mid].value=bg;
		infos[low].value=bg;
		break;
	}
}

void LedEngine::displayPercent(float x, percentStyle_t left_style, percentStyle_t right_style) {
	clear();
	if(x<0) {
		set(FaceLEDMask,.25);
		return;
	}
	if(x>1) {
		set(FaceLEDMask,.75);
		return;
	}
	if(left_style==major)
		setColumn(x,BotLLEDMask,MidLLEDMask,TopLLEDMask,TopBrLEDMask);
	if(right_style==major)
		setColumn(x,BotRLEDMask,MidRLEDMask,TopRLEDMask,TopBrLEDMask);
	x*=4;
	x-=(int)x;
	if(left_style==minor)
		setColumn(x,BotLLEDMask,MidLLEDMask,TopLLEDMask,TopBrLEDMask);
	if(right_style==minor)
		setColumn(x,BotRLEDMask,MidRLEDMask,TopRLEDMask,TopBrLEDMask);
}

void LedEngine::setColumn(float x, unsigned int low, unsigned int mid, unsigned int high, unsigned int top) {
	LEDBitMask_t solid=0;
	LEDBitMask_t partial=0;
	switch((int)(4*x)) {
	case 4:
		solid|=top;
	case 3:
		solid|=high;
	case 2:
		solid|=mid;
	case 1:
		solid|=low;
	}
	switch((int)(4*x)) {
	case 3:
		partial=top; break;
	case 2:
		partial=high; break;
	case 1:
		partial=mid; break;
	case 0:
		partial=low; break;
	}
	float partialvalue=(x*4)-(int)(x*4);
	set(partial,partialvalue);
	set(solid,1);
}


/*! @file
 * @brief Implements LedEngine, which provides basic LED effects to anything that inherits or instantiates it
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_2 $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2003/03/09 02:45:22 $
 */
