//-*-c++-*-
#ifndef INCLUDED_ControlBase_h
#define INCLUDED_ControlBase_h

#include "Events/EventBase.h"
#include "Motion/MotionManager.h"
#include "SoundPlay/SoundManager.h"
#include "Shared/Config.h"
#include <string>

//! Allows creation of items for the control system
/*! These are similar to behaviors in that they are given events, and are told when to start and stop\n
 *  However, the important difference is that these have to follow a much tighter set of guidelines for a
 *  more refined purpose.  Controls are expected to fit into a heirarchical scheme, where each control
 *  (except the root) has a parent which created it, and may return its own children where appropriate.\n
 *  This base class will do most of the work of processing events for you, and will call appropriate
 *  virtual functions which you can override.  Controls generally live in Main/Controls/ \n
 *  The ControlBase pointers which are returned at various points are the responsibility of the creator.
 *  Controller will not delete them upon deactivation
 * @todo ControlBase's should use ReferenceCounter so that's not an issue
 * @see Controller @see MenuControl */
class ControlBase {
 public:
	
	ControlBase() : name("(null name)"), last_time(0), cur_time(0), nextEv_val(0), nextEv_dur(0), prevEv_val(0), prevEv_dur(0), alreadyGotBoth(false) {} //!< Contructor
	ControlBase(const std::string& n) : name(n), last_time(0), cur_time(0), nextEv_val(0), nextEv_dur(0), prevEv_val(0), prevEv_dur(0), alreadyGotBoth(false) {} //!< Contructor, initializes with a name
	virtual ~ControlBase() {} //!< Destructor

	//@{
	//! event masks used by processEvent()
	static const EventBase nextItem; 
	static const EventBase prevItem;
	static const EventBase nextItemFast;
	static const EventBase prevItemFast;
	static const EventBase selectItem;
	static const EventBase cancel; //@}

	virtual ControlBase * activate(MotionManager::MC_ID) { return this; } //!< called when the control is activated (or the control system is reactivating)
	virtual void pause() {} //!< called when a control is being overriden by a child, or the control system is deactivating (e-stop being turned off)
	virtual void refresh() {} //!< called when the child has died and this control should refresh its display
	virtual ControlBase * processControlEvent(const EventBase& e); //!< called with incoming events - Controller only passes it button events.  For return values, see activate()
	virtual void deactivate() {} //!< called when this control is being popped from the control stack

	//@{
	//! you probably want to override some of these, call the ControlBase functions from your code if you want default sound effects
	virtual ControlBase * doSelect()   {sndman->PlayFile(config->controller.select_snd); return this;}
	virtual ControlBase * doNextItem() {sndman->PlayFile(config->controller.next_snd); return this;}
	virtual ControlBase * doPrevItem() {sndman->PlayFile(config->controller.prev_snd); return this;}
	virtual ControlBase * doReadStdIn(){sndman->PlayFile(config->controller.read_snd); return this;}
	virtual ControlBase * doCancel()   {sndman->PlayFile(config->controller.cancel_snd); return NULL;} //@}
	
	virtual ControlBase& setName(const std::string& n) { name=n; return *this; } //!< sets the name of the control, used by such as MenuControl
	virtual const char* getName() const { return name.c_str(); } //!< returns the name of the control
 protected:
	//! returns true when the current time and last time are in different periods
	static bool calcPulse(unsigned int t, unsigned int last, unsigned int period) {
		bool nextclock=(t/period)&1;
		bool lastclock=(last/period)&1;
		return (lastclock!=nextclock);
	}
	std::string name; //!< the name of this control
	unsigned int last_time; //!< the time of the last event
	unsigned int cur_time; //!< the time of the current event (do*() can check this instead of calling get_time() )
	float nextEv_val; //!< the magnitude of the last next event (::nextItem)
	unsigned int nextEv_dur; //!< the duration of the last next event (::nextItem)
	float prevEv_val; //!< the magnitude of the last prev event (::prevItem)
	unsigned int prevEv_dur; //!< the duration of the last prev event (::prevItem)
	bool alreadyGotBoth; //!< if doReadStdIn() was already called, but the buttons are both still down
};

/*! @file
 * @brief Defines ControlBase from which all items in the control system should inherit
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_2 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2003/04/30 03:55:49 $
 */
#endif
