#ifndef INCLUDED_BehaviorBase_h_
#define INCLUDED_BehaviorBase_h_

#include "Events/EventListener.h"
#include "Shared/ReferenceCounter.h"

//! The basis from which all other Behaviors should inherit
/*! Makes use of ReferenceCounter so that behaviors can automatically delete themselves if wanted\n
 *  Make sure your own DoStart and DoStop call BehaviorBase::DoStart (or Stop) to allow this behavior... otherwise you'll get memory leaks */
class BehaviorBase : public ReferenceCounter, public EventListener {
 public:
	//! constructor
	BehaviorBase() : ReferenceCounter(), started(false) {}
	//! destructor
	virtual ~BehaviorBase()=0;
	//! By default, merely adds to the reference counter (through AddReference()) @note you should still call this from your overriding methods
	virtual void DoStart() { AddReference(); started=true;}
	//! By default, subtracts from the reference counter, and deletes if zero @note you should still call this when you override this @warning call this at the end of your DoStop(), not beginning (it might @c delete @c this )
	virtual void DoStop() { started=false; RemoveReference(); }
	//! Allows you to get away with not supplying a processEvent() function for the EventListener interface.  By default, does nothing.
	virtual void processEvent(const EventBase& /*event*/) {};

	
	//! Used to identify the behavior in menus and such
	virtual const char* getName() const =0;

	//! Returns true if the behavior is currently running
	virtual bool isActive() const { return started; }

 protected:
	bool started; //!< true when the behavior is active
};

/*! @file
 * @brief Defines BehaviorBase from which all Behaviors should inherit
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_2 $
 * $Revision: 1.3 $
 * $State: Exp $
 * $Date: 2003/04/18 05:15:05 $
 */

#endif
