//-*-c++-*-
#ifndef INCLUDED_SharedQueue_h_
#define INCLUDED_SharedQueue_h_

#include "MutexLock.h"
#include "ProcessID.h"
#include "LockScope.h"

//! SharedQueue is a shared memory message buffer for interprocess communication
/*! This makes no assumptions about the data to be stored - it's all just an array of bytes
 *
 *  Assumptions are made about reception usage: all entries will be processed and then
 *  cleared so that the queue is emptied.  You cannot pop the front, only clear the whole
 *  thing.  This assumption allows much simpler/faster access. (Otherwise we need to
 *  implement a circular buffer and worry about wrap around.  *shrug* Not too hard, but
 *  unnecessary complication for now.) */
template<unsigned int maxsize, unsigned int maxentries>
class SharedQueue {
public:
	//! maximum capacity of data storage for the queue
	static const unsigned int MAX_SIZE=maxsize;
	//! maximum number of entries that can be queued
	static const unsigned int MAX_ENTRIES=maxentries;
	
	//! constructor
	SharedQueue();

	//! inserts a class into the queue
	template<class T> bool SharedQueue<maxsize,maxentries>::push(const T& obj);

	//! reserves a number of bytes in the queue, LEAVES QUEUE LOCKED, call done when finished
	void* reserve(unsigned int size);
	
	//! call this when you're finished filling in a buffer you received from reserve
	void done() { lock.release(); }

	//! checks to see if the number taken by the reader is the number added, and then clears
	/*! This will allow you to process entries without locking the entire queue, but still
	 *  not worry about missing one (or more) if it was added while you were processing the
	 *  last one. */
	bool clear(unsigned int taken);
	
	//! @return current number of entries
	unsigned int size() const {	return len; }
	
	//! @return size of entry @a i
	unsigned int size(unsigned int i) const { return entries[i].size; }

	//! @return data of entry @a i
	const void* data(unsigned int i) const { return &buf[entries[i].offset]; }

protected:
	//! @return the first free byte in buf
	unsigned int buf_end() { return len==0?0:entries[len-1].offset+entries[len-1].size; }
	
	//! @return @a sz rounded up to the nearest word (makes @a sz divisible by sizeof(int))
	unsigned int round_up(unsigned int sz) { return ((sz-1)/sizeof(int)+1)*sizeof(int); }

	//! provides mutual exclusion on non-const functions
	MutexLock<ProcessID::NumProcesses> lock;
	//! for convenience in locking functions
	typedef LockScope<ProcessID::NumProcesses> AutoLock;

	//! data storage
	char buf[MAX_SIZE];
	
	//! entry information
	struct entry_t {
		unsigned int offset; //!< offset within SharedQueue::buf
		unsigned int size;   //!< size of entry
	};

	//! holds number of entries
	unsigned int len;
	
	//! holds entry information
	entry_t entries[MAX_ENTRIES];
};

template<unsigned int maxsize, unsigned int maxentries>
SharedQueue<maxsize,maxentries>::SharedQueue()
	: lock(), len(0)
{}

template<unsigned int maxsize, unsigned int maxentries> template<class T>
bool
SharedQueue<maxsize,maxentries>::push(const T& obj) {
	AutoLock(lock,ProcessID::getID());
	if(len>=MAX_ENTRIES)
		return false;
	entries[len].offset=buf_end();
	entries[len].size=round_up(sizeof(obj));
	if(entries[len].offset+entries[len].size>=MAX_SIZE)
		return false;
	len++;
	return true;
}

template<unsigned int maxsize, unsigned int maxentries>
void*
SharedQueue<maxsize,maxentries>::reserve(unsigned int sz) {
	AutoLock(lock,ProcessID::getID());
	if(len>=MAX_ENTRIES)
		return NULL;
	entries[len].offset=buf_end();
	entries[len].size=round_up(sz);
	if(entries[len].offset+entries[len].size>=MAX_SIZE)
		return NULL;
	return &buf[entries[len++].offset];
}

template<unsigned int maxsize, unsigned int maxentries>
bool
SharedQueue<maxsize,maxentries>::clear(unsigned int taken) {
	AutoLock(lock,ProcessID::getID());
	if(size()!=taken)
		return false;
	len=0;
	return true;
}

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_1 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2003/04/06 20:57:46 $
 */

#endif
