//-*-c++-*-
#ifndef INCLUDED_Config_h
#define INCLUDED_Config_h

#include <vector>
#include <string>

//!provides global access to system configuration information
class Config {
 public:
	//!constructor
	Config(const char* filename)
		: wireless(), vision(), main(), behaviors(), controller(), motion(),
		worldmodel2(), sound()
		{ readConfig(filename); }
	//!destructor
	~Config() {}

	//!section IDs
	enum section_t {
		sec_wireless=0,  //!< denotes wireless section of config file
		sec_vision,      //!< denotes vision section of config file
		sec_main,        //!< denotes main section of config file, for misc. settings
		sec_behaviors,   //!< denotes behaviors section of config file
		sec_controller,  //!< denotes controller section of config file
		sec_motion,      //!< denotes motion section of config file
		sec_worldmodel2, //!< denotes worldmodel section of config file
		sec_sound,       //!< denotes sound section of config file
		sec_invalid      //!< denotes an invalid section of config file
	};

	//!wirless information
	struct wireless_config {
		int id; //!< id number (in case you have more than one AIBO)
      
		wireless_config () : id(1) {} //!< constructor
	} wireless;

	//!vision information
	struct vision_config {
		int white_balance;    //!< white balance
		int gain;             //!< gain
		int shutter_speed;    //!< shutter speed
		int resolution;       //!< resolution
		char thresh[30];      //!< thresholds
		char colors[30];      //!< colors
		int raw_port;         //!< port to send raw frames on
		int rle_port;         //!< port to send RLE frames on
		int obj_port;         //!< port to send object info on
      
		//!constructor
		vision_config() : white_balance(3), gain(2), shutter_speed(2), resolution(2), thresh(), colors(), raw_port(0), rle_port(0), obj_port(0) {}
	} vision;
	
	//!core functionality information
	struct main_config {
		int console_port;  //!< port to send/receive "console" information on (separate from system console)
		int stderr_port;   //!< port to send error information to
		int error_level;   //!< controls amount of info to error port
		int debug_level;   //!< controls amount of debug info
		int verbose_level; //!< controls verbosity of info
		int wsjoints_port; //!< port to send joint positions on
		int wspids_port;   //!< port to send pid info on

		//!constructor
		main_config()
			: console_port(0), stderr_port(0), error_level(0), debug_level(0),
				 verbose_level(0),wsjoints_port(0),wspids_port(0)
		{ }
	} main;

	//!placeholder
	struct behaviors_config {
	} behaviors;
    
	//!controller information
	struct controller_config {
		char select_snd[50]; //!< sound file to use for "select" action
		char next_snd[50];   //!< sound file to use for "next" action
		char prev_snd[50];   //!< sound file to use for "prev" action
		char read_snd[50];   //!< sound file to use for "read from std-in" action
		char cancel_snd[50]; //!< sound file to use for "cancel" action

		//!constructor
		controller_config() {
			select_snd[0]=next_snd[0]=prev_snd[0]=read_snd[0]=cancel_snd[0]='\0';
		}
	} controller;
    
	//!motion information
	struct motion_config {
		char root[50]; //!< path on memory stick to "motion" files - for instance, position (.pos) and motion sequence (.mot)
		char estop_on_snd[50];  //!< sound file to use when e-stop turned on
		char estop_off_snd[50]; //!< sound file to use when e-stop turned off
		motion_config() {
			estop_on_snd[0]=estop_off_snd[0]='\0';
		}
	} motion;

	//!world model information
	struct worldmodel2_config {
		//@{
		//!ports to use for sending world model information
		int dm_port, hm_port, gm_port, fs_port; //@}

		//!constructor
		worldmodel2_config() : dm_port(0), hm_port(0), gm_port(0), fs_port(0) {}
	} worldmodel2;
	
	//!sound information
	struct sound_config {
		char root[50]; //!< path to sound clips
		unsigned int sample_rate; //!< sample rate to send to system, currently only 8000 or 16000 supported
		unsigned int sample_bits; //!< sample bit depth, either 8 or 16
		std::vector<std::string> preload; //!< list of sounds to preload at boot
			
		//!constructor
		sound_config() : sample_rate(0), sample_bits(0), preload() {}
	} sound;

	//! call this function when it's time to read the configuration file
	void readConfig(const char* filename);
};

//!allows global access to current settings
extern Config* config;

/*! @file
 * @brief Describes Config, which provides global access to system configuration information
 * @author alokl (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_1 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2003/04/09 04:48:18 $
 */

#endif
