//-*-c++-*-
#ifndef INCLUDED_EventRouter_h
#define INCLUDED_EventRouter_h

#include <vector>
#include <list>
#include <map>
#include <algorithm>
#include "EventListener.h"
#include "EventTrapper.h"
#include "Shared/get_time.h"
#include "Shared/debuget.h"

//! This class will handle distribution of events as well as management of timers
/*! Classes must inherit from EventListener and/or EventTrapper in order to
 *  receive events.\n
 *  Use the global @c ::erouter EventRouter to post and subscribe to events\n
 *
 *  When multiple listeners are subscribed, the order in which an event is
 *  distributed among them looks like this:
 *  -# "Specific" listeners: any listener which specifies a particular source id.
 *     (It doesn't matter if they specify a type id or not.)
 *    - older listeners get events before younger listeners
 *  -# "General" listeners: those that subscribe to an entire generator
 *    - older listeners get events before younger listeners
 *
 *  ...but if you're relying on that ordering, there should be a cleaner
 *  way to do whatever you're doing.
 *
 *  If one behaviors unsubscribes another one during a processEvent(), that
 *  behavior will still get the "current" event before the unsubscription
 *  takes place.
 *
 *  Buffering events has not been tested thoroughly...
 *
 *  @see EventBase::EventGeneratorID_t for a complete listing of all generators,
 *  as well as instructions on how to add new generators.
 */
class EventRouter : public EventListener {
 public:
	EventRouter(); //!< Constructs the router, #buffertime defaults to 1
	virtual ~EventRouter() { reset(); removeAllTimers(); } //!< just calls reset and removeAllTimers()
	void reset() { listeners.clear(); trappers.clear(); removeAllTimers(); } //!< erases all listeners, trappers and timers, resets EventRouter
	
	void setBufferTime(unsigned int t) { buffertime=t; } /*!< @brief sets the time to wait between buffer clears, see EventRouter::buffertime. @param t time to wait between buffer clears @see buffertime */
	unsigned int getBufferTime() { return buffertime; } /*!< @brief returns the time to wait between buffer clears, see EventRouter::buffertime. @return time to wait between buffer clears @see buffertime */

	//!@name Posting/Processing Events
	/*!@brief recommended to create and post an event using current buffer setting */
	void postEvent(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid, unsigned int dur) { if(buffertime>0) events.push_back(new EventBase(egid,sid,etid,dur)); else processEvent(EventBase(egid,sid,etid,dur)); }
	void postEvent(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid, unsigned int dur, const std::string& n, float m) { if(buffertime>0) events.push_back(new EventBase(egid,sid,etid,dur,n,m)); else processEvent(EventBase(egid,sid,etid,dur,n,m)); }
	void postEvent(EventBase* e) { if(buffertime>0) events.push_back(e); else { processEvent(*e); delete e; } }

	//! determines if timers need to be posted, and posts them if so.
	/*! Call this often to ensure accurate timers.  Also, will clear event buffer before sending
			timer events in order to ensure correct ordering of event reception */
	void processTimers();
	void processEventBuffer(); //!< clears the event buffer, deletes events as it does so.
	void processEvent(const EventBase& e); //!< forces unbuffered - sends event *now*.  Will clear the buffer first if needed to ensure proper event ordering
	//@}

	//!@name Listener Detection
	/*!@brief <b>counts both listeners and trappers</b>, so stuff can tell if it even needs to bother generating an event...*/
	/* ... if a tree falls in a forest, and there's no one around to see it, does it make a sound?\n
		 ... if Vision sees a ball in an image, and there's no listeners, does it make an event? ;) */
	bool hasListeners(EventBase::EventGeneratorID_t egid) { return trappers.hasMapping(egid) || listeners.hasMapping(egid); }
	bool hasListeners(EventBase::EventGeneratorID_t egid, unsigned int sid) { return trappers.hasMapping(egid,sid) || listeners.hasMapping(egid,sid); }
	bool hasListeners(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { return trappers.hasMapping(egid,sid,etid) || listeners.hasMapping(egid,sid,etid); }
	//@}

	//!@name Timer Management
	void addTimer(EventListener* el, unsigned int sid, unsigned int delay, bool repeat=true); //!< adds a timer or sets a timer if it doesn't already exist.
	void addTimer(EventListener* el, const EventBase& e, bool repeat=true) { addTimer(el,e.getSourceID(),e.getDuration(),repeat); } //!< calls the other addTimer() with the event's source id and duration, doesn't check to see if the generator is timerEGID
	void removeTimer(EventListener* el); //!< clears all pending timers for listener @a el
	void removeTimer(EventListener* el, unsigned int sid); //!< clears any pending timers with source id @a sid for listener @a el
	void removeAllTimers(); //!< clears all timers for all listeners
	//@}

	//!@name Listener Management
	void addListener(EventListener* el, const EventBase& e); //!< Adds a listener for a specific source id and type from a given event generator, adding a Timer event will invoke addTimer(@a el, @a e.getSourceID(), @a e.getDuration(), @c true)
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid); //!< Adds a listener for all events from a given event generator
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid); //!< Adds a listener for all types from a specific source and generator
	void addListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a listener for a specific source id and type from a given event generator

	//! stops sending specified events from the generator to the listener.  If a timer is passed it will invoke removeTimer(@a el, @a e.getSourceID())
	void removeListener(EventListener* el, const EventBase& e);
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid) { listeners.removeMapping(el,egid); listeners.clean(); } //!< stops sending specified events from the generator to the listener.
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { for(unsigned int et=0; et<EventBase::numETIDs; et++) listeners.removeMapping(el,egid,sid,(EventBase::EventTypeID_t)et); listeners.clean(); } //!< stops sending specified events from the generator to the listener.
	void removeListener(EventListener* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { listeners.removeMapping(el,egid,sid,etid); listeners.clean(); } //!< stops sending specified events from the generator to the listener.

	void removeListener(EventListener* el) { for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++) listeners.removeMapping(el,(EventBase::EventGeneratorID_t)eg); listeners.clean(); } //!< stops sending ALL events to the listener
	void forgetListener(EventListener* el) { removeTimer(el); removeListener(el); } //!< clears timers and removes listener from all events
	//@}

	//!@name Trapper Management
	void addTrapper(EventTrapper* el, const EventBase& e) { trappers.addMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); } //!< Adds a trapper for a specific source id and type from a given event generator
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid) { trappers.addMapping(el,egid); } //!< Adds a trapper for all events from a given event generator
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { for(unsigned int et=0; et<EventBase::numETIDs; et++) trappers.addMapping(el,egid,sid,(EventBase::EventTypeID_t)et); }  //!< Adds a trapper for all types from a specific source and generator
	void addTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { trappers.addMapping(el,egid,sid,etid); } //!< Adds a trapper for a specific source id and type from a given event generator

	void addTrapper(EventTrapper* el) { for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++) trappers.addMapping(el,(EventBase::EventGeneratorID_t)eg); } //!< adds a trapper for ALL events

	//! stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, const EventBase& e) { trappers.removeMapping(el,e.getGeneratorID(),e.getSourceID(),e.getTypeID()); trappers.clean(); }
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid) { trappers.removeMapping(el,egid); trappers.clean(); } //!< stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid) { for(unsigned int et=0; et<EventBase::numETIDs; et++) trappers.removeMapping(el,egid,sid,(EventBase::EventTypeID_t)et); trappers.clean(); } //!< stops sending specified events from the generator to the trapper.
	void removeTrapper(EventTrapper* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid) { trappers.removeMapping(el,egid,sid,etid); trappers.clean(); } //!< stops sending specified events from the generator to the trapper.

	void removeTrapper(EventTrapper* el) { for(unsigned int eg=0; eg<EventBase::numEGIDs; eg++) trappers.removeMapping(el,(EventBase::EventGeneratorID_t)eg); trappers.clean(); } //!< stops sending ALL events to the trapper
	//@}

 protected:
	void doSendBuffer(); //!< does the work of clearing the buffer
	void doSendEvent(const EventBase& e); //!< does the work of sending an event

	//! Contains all the information needed to maintain a timer by the EventRouter
	struct TimerEntry {
		//! constructors an entry using the given value for next - useful for with TimerEntryPtrCmp
		explicit TimerEntry(unsigned int nxt) : el(NULL), sid(0), delay(0), next(nxt), repeat(false) {}
		//! constructs with the given values, sets next field automatically @see next
		TimerEntry(EventListener* e, unsigned int s, unsigned int d, bool r) : el(e), sid(s), delay(d), next(get_time()+delay), repeat(r) {}
		//! just does the default, i'm just being explicit since there's a pointer (no deep copy!)
		TimerEntry(const TimerEntry& t) : el(t.el), sid(t.sid), delay(t.delay), next(t.next), repeat(t.repeat) {}
		//! just does the default, i'm just being explicit since there's a pointer (no deep copy!)
		TimerEntry operator=(const TimerEntry& t) { el=t.el; sid=t.sid; delay=t.delay; next=t.next; repeat=t.repeat; return *this; }
		//! will reset timer
		/*! @param d the time from now when the timer should go off (in milliseconds)
		 *  @param r true if the timer should automatically repeat */
		void Set(unsigned int d, bool r) { delay=d; repeat=r; next=get_time()+delay; }
		EventListener* el;  //!< the listener to fire at
		unsigned int sid;   //!< the source id to fire with
		unsigned int delay; //!< the delay until firing
		unsigned int next;  //!< the time at which this timer will go off next
		bool repeat;        //!< if true, will reset after firing, else will be deleted
	};
	/*! @brief Used by STL to sort the timer list in order of activation time
	 *  @see EventRouter::timers */
	class TimerEntryPtrCmp {
	public:
		//! Used by STL to sort the timer list in order of activation time @see timers
		/*! Since we remove NULLs before sorting, shouldn't need to check here (and I want to know if i'm wrong)
		 *  @return (a->next<b->next) */
		bool operator()(const TimerEntry* const a, const TimerEntry* const b) const { return (a->next<b->next); }
	};
	typedef std::vector<TimerEntry*>::iterator timer_it_t; //!< makes code more readable
	std::vector<TimerEntry*> timers;         //!< the list of timer entries being maintained, kept sorted by time they go active

	//! just for debugging
	void chkTimers() {
		unsigned int last=0;
		for(timer_it_t it=timers.begin(); it!=timers.end(); it++) {
			if(last>(*it)->next) {
				dispTimers();
				return;
			}
			last=(*it)->next;
		}
	}
	//! just for debugging
	void dispTimers() {
		std::cout << "out of order timers " << get_time() << " :\t";
		unsigned int last=0;
		for(timer_it_t it=timers.begin(); it!=timers.end(); it++) {
			if(last>(*it)->next)
				std::cout << "##";
			std::cout << (last=(*it)->next) << '\t';
		}
		std::cout << std::endl;
	}

	std::vector<EventBase*> events;     //!< used to store buffered events
	bool doSendBufferLock;              //!< in case of recursive calls to processEventBuffer()/doSendBuffer()
	unsigned int lastBufClear;          //!< time of last event buffer clear
	unsigned int buffertime;            //!< The time between clearings of the buffer
																			/*!< <li>0 will not use the buffer, events are routed upon posting
																			 *   <li>1 will clear the buffer at next call to processTimers() or processEventBuffer()
																			 *   <li>a larger value will cause a delay of that number of milliseconds since the last clearing */
	
	//! Does the actual storage of the mapping between EventBase's and the EventListeners/EventTrappers who should receive them
	/*! Actually only stores void*'s, so it's more general than just Listeners or Trappers */
	class EventMapper {
	public:
		//! constructor
		EventMapper();

		void addMapping(void* el, EventBase::EventGeneratorID_t egid) { allevents[egid].push_back(el); } //!< Adds a listener for all events from a given event generator
		void addMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid); //!< Adds a listener for a specific source id and type from a given event generator

		//! Removes a listener for all events from a given event generator
		/*! Doesn't necessarily remove the vector or mapping if this was the last listener, use clean() to do that */
		void removeMapping(void* el, EventBase::EventGeneratorID_t egid); 

		//! Removes a listener for a specific source id and type from a given event generator
		/*! Doesn't necessarily remove the vector or mapping if this was the last listener, use clean() to do that */
		void removeMapping(void* el, EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid);

		void clean(); //!<compresses empty data structures
		void clear(); //!<Resets the mapping

		//@{
		//! so stuff can tell if it even needs to bother generating an event...
		/* ... if a tree falls in a forest, and there's no one around to see it, does it make a sound?
			 ... if Vision sees a ball in an image, and there's no listeners, does it make an event? ;) */
		/*! @return true if it has any listeners, false otherwise */
		bool hasMapping(EventBase::EventGeneratorID_t egid);
		bool hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid);
		bool hasMapping(EventBase::EventGeneratorID_t egid, unsigned int sid, EventBase::EventTypeID_t etid);
		//@}

		//! builds a list of all listeners which should receive the event, templated to typecast the pointers for you
		/*! @param e the key event
		 *  @param listeners upon return, the resulting list of listeners @a e maps to\n
		 *  @a listeners is not cleared prior to building, new listeners are pushed on end\n
		 *  Results are in the order: all specific matches first, all generator listeners second, in order they were added to the EventMapper.*/
		template<class T>
		void getMapping(const EventBase& e, std::vector<T*>& listeners);

	protected:
		//! a mapping from source IDs (unsigned ints), each to a vector of pointers to listeners
		/*! main use in filteredevents @see filteredevents */
		typedef std::map<unsigned int,std::vector<void*>,std::less<unsigned int> > SIDtoListenerVectorMap_t;
		
		//! an array of vectors of pointers to listeners... in other words, a vector of listener pointers for each generator
		std::vector<void*> allevents[EventBase::numEGIDs];
		//! not for the faint of heart: a matrix of mappings to vectors of pointers to listeners
		SIDtoListenerVectorMap_t* filteredevents[EventBase::numEGIDs][EventBase::numETIDs];

	private:
		EventMapper(const EventMapper&);           //!< this shouldn't be called...
		EventMapper operator=(const EventMapper&); //!< this shouldn't be called...
	};

	EventMapper trappers;  //!< A mapping of which EventTrapper's should get a chance to trap the event
	EventMapper listeners; //!< A mapping of which EventListener's should receive events

};

//! a global router for cross communication, probably the most common usage, although perhaps there may be times you'd rather have multiple event routers for smaller sections
extern EventRouter * erouter;

/*! @file
 * @brief Describes EventRouter class, for distribution and trapping of events to listeners
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_1 $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2003/04/06 20:57:43 $
 */

#endif
