//-*-c++-*-
#ifndef INCLUDED_BatteryMonitorBehavior_h_
#define INCLUDED_BatteryMonitorBehavior_h_

#include "Behaviors/BehaviorBase.h"
#include "Shared/debuget.h"
#include "Shared/WorldState.h"
#include "Events/EventRouter.h"
#include "Shared/SharedObject.h"
#include "Motion/MotionManager.h"
#include "Motion/PostureMC.h"
#include "Motion/LedMC.h"

//! A background behavior which will monitor the power level and flip the ears when appropriate
/*! Think of this as a simple example class.  For exercise, try using a MotionSequenceMC instead
 *  of switching the ears back manually using a PostureMC */
class BatteryMonitorBehavior : public BehaviorBase {
public:
	//! constructor
	BatteryMonitorBehavior() : BehaviorBase(), pose(NULL), pose_id(MotionManager::invalid_MC_ID), led_id(MotionManager::invalid_MC_ID) {}
	//! destructor
	virtual ~BatteryMonitorBehavior() {}

	//! Listens for the PowerSourceID::LowPowerWarnSID
	virtual void DoStart() {
		BehaviorBase::DoStart();
		erouter->addListener(this,EventBase::powerEGID,PowerSourceID::LowPowerWarnSID);
		erouter->addListener(this,EventBase::powerEGID,PowerSourceID::ExternalPowerSID);
		erouter->addListener(this,EventBase::powerEGID,PowerSourceID::BatteryConnectSID);
		//if the low power warning is *already* on, better forge an event and send it to myself
		if(shouldWarn())
			processEvent(EventBase(EventBase::powerEGID,PowerSourceID::UpdatedSID,EventBase::statusETID));
	}
	//! Stops listening for events
	virtual void DoStop() {
		if(pose!=NULL)
			stopWarning();
		erouter->forgetListener(this);
		BehaviorBase::DoStop();
	}
	//! Adds a BatteryMonitorMC to motman if power goes low
	virtual void processEvent(const EventBase &event) {
		if(event.getGeneratorID()==EventBase::powerEGID) {
			//just check for low power status
			bool shouldwarn=shouldWarn();
			if(pose!=NULL && !shouldwarn)
				stopWarning();
			else if(pose==NULL && shouldwarn)
				startWarning();
		} else {
			ASSERTRET(event.getGeneratorID()==EventBase::timerEGID,"Unrequested event "<<event.getName());
			switch(event.getSourceID()) {
			case 1: { // toggle the ears (signals low battery), show battery level on LEDs
				ASSERTRET(pose!=NULL,"Extra timer 1");
				for(unsigned int i=EarOffset; i<EarOffset+NumEarJoints; i++)
					pose->setOutputCmd(i,!state->outputs[i]);
				unsigned int flipdelay=calcFlipDelay();
				// if we're just constantly flipping the ears, a slight change is needed so the battery
				// level isn't obscuring the LED settings
				if(flipdelay<=NumFrames*FrameTime) {
					static bool on=false;
					on=!on;
					if(on) {
						motman->setPriority(led_id,MotionManager::kEmergencyPriority+1);
						MMAccessor<LedMC> led(led_id);
						led->displayPercent(state->sensors[PowerRemainOffset],LedEngine::major,LedEngine::major);
					} else
						motman->setPriority(led_id,MotionManager::kIgnoredPriority);
					erouter->addTimer(this,1,128+flipdelay,false);
				} else {
					motman->setPriority(led_id,MotionManager::kEmergencyPriority+1);
					MMAccessor<LedMC> led(led_id);
					led->displayPercent(state->sensors[PowerRemainOffset],LedEngine::major,LedEngine::major);
					erouter->addTimer(this,2,128,false);
				}
			} break;
			case 2: { // release ear until next flap, hide LEDs display
				ASSERTRET(pose!=NULL,"Extra timer 1");
				for(unsigned int i=EarOffset; i<EarOffset+NumEarJoints; i++)
					pose->setOutputCmd(i,OutputCmd());
				motman->setPriority(led_id,MotionManager::kIgnoredPriority);
				erouter->addTimer(this,1,calcFlipDelay(),false);
			} break;
			default:
				ASSERTRET(false,"Unrequested timer " << event.getName());
				break;
			}
		}
	}
	virtual const char* getName() const { return (isActive()?"#BatteryMonitorBehavior":"-BatteryMonitorBehavior"); }

	//! returns true if the warning should be active (power remaining less than 40%, no external power, but also checks that a power update has been received)
	static bool shouldWarn() { return state!=NULL && state->powerFlags[PowerSourceID::BatteryConnectSID] && (state->sensors[PowerRemainOffset]<=.15 || state->powerFlags[PowerSourceID::LowPowerWarnSID]) && !state->powerFlags[PowerSourceID::ExternalPowerSID]; }

protected:
	//! adds a pose and a timer to get the ears flipping
	void startWarning() {
		pose_id=motman->addMotion(SharedObject<PostureMC>(),MotionManager::kEmergencyPriority+1);
		pose=(PostureMC*)motman->peekMotion(pose_id);
		SharedObject<LedMC> led;
		led->displayPercent(state->sensors[PowerRemainOffset],LedEngine::major,LedEngine::major);
		led_id=motman->addMotion(led,MotionManager::kEmergencyPriority+1);
		for(unsigned int i=EarOffset; i<EarOffset+NumEarJoints; i++)
			pose->setOutputCmd(i,!state->outputs[EarOffset]);
		erouter->addTimer(this,2,128,false);
	}
	//! removes pose, in case battery magically charges
	void stopWarning() {
		motman->removeMotion(pose_id);
		motman->removeMotion(led_id);
		led_id=pose_id=MotionManager::invalid_MC_ID;
		pose=NULL;
		erouter->removeTimer(this,1);
		erouter->removeTimer(this,2);
	}
	//! makes the ears flip more rapidly as power declines.  Flips back and forth once every 15 seconds at 15%, down to flipping constantly at 5%.
	unsigned int calcFlipDelay() {
		unsigned int max_t=5000;
		double high_power=.15;
		double no_power=.05;
		double cur_power=state->sensors[PowerRemainOffset];
		if(cur_power<no_power)
			return 0;
		return (unsigned int)(max_t*(cur_power-no_power)/(high_power-no_power));
	}
	PostureMC* pose; //!< if we are currently warning of low battery, holds a pose, NULL otherwise
	MotionManager::MC_ID pose_id; //!< id of pose if we are currently warning, MotionManager::invalid_MC_ID otherwise
	MotionManager::MC_ID led_id; //!< id of LedMC if we are currently warning, MotionManager::invalid_MC_ID otherwise

private:
	BatteryMonitorBehavior(const BatteryMonitorBehavior&); //!< don't copy behaviors
	BatteryMonitorBehavior operator=(const BatteryMonitorBehavior&); //!< don't assign behaviors
};

/*! @file
 * @brief Defines BatteryMonitorBehavior, a background behavior to trigger BatteryMonitorMC to warn when the power is low
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_1 $
 * $Revision: 1.3 $
 * $State: Exp $
 * $Date: 2003/03/09 02:45:20 $
 */

#endif
