//-*-c++-*-
#ifndef INCLUDED_Controller_h
#define INCLUDED_Controller_h

#include "Controls/ControlBase.h"
#include "Behaviors/BehaviorBase.h"
#include "Events/EventBase.h"
#include "Events/EventRouter.h"
#include "Motion/MotionManager.h"
#include <stack>

//! Handles the menu/command system... when it detects the EmergencyStopMC is activated, it'll kick into high priority.
/*! Keeps track of a command stack.  A Control can designate another sub-control, which will receive events until it finishes\n
 *  Events will then be sent to the parent again */
class Controller : public BehaviorBase, public EventTrapper {
 public:
	Controller() : display(MotionManager::invalid_MC_ID), estop_id(MotionManager::invalid_MC_ID), root(NULL), cmdstack() {}	//!< Constructor
	Controller(ControlBase* r) : display(MotionManager::invalid_MC_ID), estop_id(MotionManager::invalid_MC_ID), root(r), cmdstack() {} //!< Constructor, sets a default root control
	virtual ~Controller() {
		delete root;
	} //!< Destructor

	virtual void DoStart(); //!< register for events and resets the cmdstack
	virtual void DoStop(); //!< stop listening for events and resets the cmdstack
	virtual bool trapEvent(const EventBase& e); //!< passes an event to the top control
	virtual void processEvent(const EventBase& e); //!< just for timers - checks to see if e-stop is active
	
	void reset();   //!< will take the command stack back down to the root
	void refresh(); //!< refreshes the display, for times like sub-control dying, the previous control needs to reset it's display

	void push(ControlBase* c); //!< puts a new control on top
	void pop();                //!< kills the top control, goes to previous
	ControlBase* top() { return cmdstack.top(); } //!< returns the current control

	Controller& setRoot(ControlBase* r); //!< sets the root level control

	Controller& setEStopID(MotionManager::MC_ID estopid); //!< Sets the emergency stop MC to monitor for pausing
	
	virtual const char* getName() const { return "Controller"; }

 protected:
	//! maintains top Control
	/*! @param next one of: @li NULL: pop() ::cmdstack @li ::cmdstack.top(): nothing @li other address: ::push(@a next) */
	void setNext(ControlBase* next);

	//! called when the estop switches on
	/*!  causes the top control to activate, registers for button events */
	void activate();

	//! called when the estop switches off\n
	/*!  causes the top control to deactivate, stops listening for buttons */
	void deactivate();
	
	//! @brief returns true if a valid control is available on the stack
	/*!  if the stack is empty, will push root if it's non-null */
	bool chkCmdStack();

	//! invalid_MC_ID if not active, otherwise id of high priority LEDs
	MotionManager::MC_ID display;

	//! the EmergencyStopMC MC_ID that this Controller is monitoring
	MotionManager::MC_ID estop_id;

	//! the base control, if cmdstack underflows, it will be reset to this
	ControlBase * root;

	/*! @brief the stack of the current control hierarchy\n
	 *  should never contain NULL entries */
	std::stack< ControlBase* > cmdstack;

private:
	Controller(const Controller&); //!< shouldn't be called...
	Controller& operator=(const Controller&); //!< shouldn't be called...
};

/*! @file
 * @brief Describes Controller class, a behavior that should be started whenever the emergency stop goes on to provide menus for robot control
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-1_1 $
 * $Revision: 1.5 $
 * $State: Exp $
 * $Date: 2003/04/06 20:57:42 $
 */

#endif
