//-*-c++-*-
#ifndef INCLUDED_WorldState_h
#define INCLUDED_WorldState_h

#include <OPENR/core_macro.h>
#include <OPENR/ObjcommTypes.h>
#include <OPENR/OPENR.h>
#include <OPENR/OPENRAPI.h>
#include <OPENR/OPENRMessages.h>

#include "Shared/RobotInfo.h"
#include "Events/EventBase.h"
#include "Shared/Profiler.h"
#include <math.h>
#include "WorldStateSerializer.h"

class EventRouter;
class WorldStateSerializer;

//The following SourceIDs are for events created by WorldState's event generators

//! holds source ID types for button events @see EventBase @see ButtonSourceID_t
namespace ButtonSourceID {
	//! holds source ID types for button events
	/*! Should be a straight mapping to the ButtonOffset_t @see ButtonOffset_t*/
	enum ButtonSourceID_t {
		LFrPawSID = LFrPawOffset,
		RFrPawSID = RFrPawOffset,
		LBkPawSID = LBkPawOffset,
		RBkPawSID = RBkPawOffset,
		ChinButSID = ChinButOffset,
		BackButSID,
		HeadFrButSID,
		HeadBkButSID
	};
}

//! holds source ID types for sensor events @see EventBase @see SensorSourceID_t
namespace SensorSourceID {
	//commented out SIDs aren't actually being generated
	//! holds source ID types for sensor events
	enum SensorSourceID_t {
		/*	 IRDistSID = NumButtons+IRDistOffset
				 BAccelSID = NumButtons+BAccelOffset,
				 LAccelSID = NumButtons+LAccelOffset,
				 DAccelSID = NumButtons+DAccelOffset,
				 ThermoSID = NumButtons+ThermoOffset, */
		UpdatedSID //!< sends status event as last event after processing a frame
	};
}

//! holds source ID types for power events @see EventBase @see PowerSourceID_t
namespace PowerSourceID {
	//! holds source ID types for power events
	/*! Also serve as offsets into ::powerFlags[]
	 *  I've never seen a lot of these events thrown by the OS.  NS means never-seen, which
	 *  could simply be because i haven't put it in that situation (don't have a station-type power
	 *  charger) or because the OS doesn't actually support sending that flag.
	 *
	 *  Under normal conditions, you'll see MotorPowerSID, BatteryConnectSID, DischargingSID, and
	 *  PowerGoodSID always active with occasional VibrationSID and UpdateSID. When the chest
	 *  button is pushed, PauseSID is activated and MotorPowerSID is deactivated.
	 *
	 *  The BatteryMonitorBehavior will give a warning once power begins getting low.  The OS won't
	 *  boot off a battery with less than 15% power remaining (which is when the LowPowerWarnSID is
	 *  thrown)
	 *
	 *  @note there's not a one-to-one correspondance of the events from the
	 *  OPENR power system... i map several OPENR events to fewer Tekkotsu events, check the name
	 *  if you want to know the specific source (say if low battery is low current and/or low voltage)
	 *  Status ETIDS are only generated when one of a related group goes on/off but others are still active*/
	enum PowerSourceID_t {
		PauseSID=0, //!< the chest button was pushed (this is not a normal button, it kills power to the motors in hardware)
		MotorPowerSID, //!< active while the motors have power
		VibrationSID, //!< triggered when the OS decides a large acceleration has occured, like falling down (or more specifically, hitting the ground afterward)
		BatteryEmptySID, //!< battery is dead
		LowPowerWarnSID, //!< triggered when sensors[PowerRemainOffset] <= 0.15 (PowerGoodSID stays on)
		BatteryFullSID,  //!< battery is full
		ExternalPowerSID, //!< receiving power from an external source (such as AC cable, may or may not include the "station", see StationConnectSID)
		ExternalPortSID,  //!< an external power source is plugged in (does not imply current is flowing however)
		BatteryConnectSID, //!< a battery is plugged in
		BatteryInitSID, //!< ? NS
		DischargingSID, //!< using power from the battery (although still stays on after hooked up to external power)
		ChargingSID, //!< you used to be able to charge while running, tho that has changed in more recent versions of OPEN-R.  In any case, I never saw this even when it did work.
		OverheatingSID, //!< in case the robot starts getting too hot NS
		PowerGoodSID, //!< there is power, either from external or battery
		ChargerStatusSID, //!< ? NS
		SuspendedSID, //!< ? NS
		OverChargedSID, //!< in case the charger screws up somehow (?) NS
		TermDischargeSID, //!< end of battery (?) NS
		TermChargeSID, //!< end of charging (?) NS
		ErrorSID, //!< general power error NS
		StationConnectSID, //!< connected to a station NS
		BatteryOverCurrentSID, //!< similar to OverChargedSID (?) NS
		DataFromStationSID, //!< ? NS
		RegisterUpdateSID, //!< ? NS
		RTCSID, //!< ? NS
		SpecialModeSID, //!< ? NS
		BMNDebugModeSID, //!< ? NS
		PlungerSID, //!< I think this is in reference to having a memorystick (?) NS
		UpdatedSID, //!<sent as last event after processing a frame
		NumPowerSIDs
	};
}
	
//! The state of the robot and its environment\n
/* Contains sensor readings, current joint positions, etc.\n
 * This is a shared memory region between MainObj, MotoObj, and possibly others in the future\n
 * Be very careful about including structures that use pointers in this class... they will only be valid
 * from the OObject that created them, others may cause a crash if they try to access them.
 *
 * WorldState takes power and sensor updates from the system and maintains the last known
 * values in its member fields.  It throws events when some of these values change, listed in
 * the ButtonSourceID, SensorSourceID, and PowerSourceID namespaces.
 *
 * Status events for buttons are always generated unless the WorldState::alwaysGenerateStatus flag
 * is turned off, in which case they are only generated when a value has changed (the top head buttons)
 */
class WorldState {
public:
	//! constructor - sets everything to zeros
	WorldState();

	bool alwaysGenerateStatus; //!< controls whether status events are generated for the boolean buttons

	double outputs[NumOutputs];     //!< same format as Motion stuff, for ears, x<.5 is up, x>=.5 is down
	double buttons[NumButtons];     //!< magnitude is pressure for some, 0/1 for others
	double sensors[NumSensors];     //!< IR, Accel, Thermo, Power stuff
	double pids[NumPIDJoints][3];   //!< current PID settings
	double pidduties[NumPIDJoints]; //!< duty cycles - -1 means the motor is trying to move full power in negative direction, 1 means full power in positive direction, in practice, these values stay rather small - 0.15 is significant force.

	unsigned int robotStatus;       //!< bitmask, see OPENR/OPower.h
	unsigned int batteryStatus;     //!< bitmask, see OPENR/OPower.h
	unsigned int powerFlags[PowerSourceID::NumPowerSIDs]; //!< bitmasks of similarly-grouped items from previous two masks, corresponds to the PowerSourceID_t's

	unsigned int button_times[NumButtons]; //!< value is time of current press, 0 if not down
	
	unsigned int lastSensorUpdateTime;     //!<primarily so calcDers can determine the time difference between updates, but others might want to know this too...

  static const double g;                 //!< the gravitational acceleration of objects on earth
	static const double IROORDist;         //!< If IR returns this, we're out of range

	ProfilerOfSize<20> mainProfile;        //!< holds code profiling information for MainObject
	ProfilerOfSize<06> motionProfile;      //!< holds code profiling information for MotionObject

  void read(OSensorFrameVectorData& sensor, EventRouter* er); //!< will process a sensor reading as given by OPEN-R
	void read(const OPowerStatus& power, EventRouter* er);      //!< will process a power status update as given by OPEN-R

  WorldStateSerializer * wsser;

protected:
	unsigned int curtime; //!< set by read(OSensorFrameVectorData& sensor, EventRouter* er) for chkEvent() so each call doesn't have to

	//! Tests to see if the button status has changed and post events as needed
	void chkEvent(EventRouter* er, unsigned int off, float newval, const char* name);

	//! sets the names of the flags that will be generating events
	/*! note that this function does not actually do the event posting */
	void chkPowerEvent(unsigned int sid, unsigned int cur, unsigned int mask, const char* name, 
										 std::string actname[PowerSourceID::NumPowerSIDs],
										 std::string dename[PowerSourceID::NumPowerSIDs],
										 unsigned int summask[PowerSourceID::NumPowerSIDs]) {
		if(cur&mask) {
			actname[sid]+=name;
			summask[sid]|=mask;
		} else if(powerFlags[sid]&mask)
			dename[sid]+=name;
	}

	//! given the next value, calculates and stores the next current, the velocity, and the acceleration
	/*! @param next the new value that's about to be set
	 *  @param cur the previous value
	 *  @param vel the previous 1st derivative
	 *  @param acc the previous 2nd derivative
	 *  @param invtimediff @f$1/(curtime-prevtime)@f$ in seconds*/
	inline void calcDers(double next, double& cur, double& vel, double& acc, double invtimediff) {
		double diff=next-cur;
		cur=next;
		next=diff*invtimediff;;
		diff=next-vel;
		vel=next;
		acc=diff*invtimediff;
	}

	/*! @name CPC IDs
	 * used to interface with lower level OPEN-R code to read sensors - DOESN'T correspond to ::PrimitiveName */
	static const int CPCJointNeckTilt           =  0; //!< Head
	static const int CPCJointNeckPan            =  1;
	static const int CPCJointNeckRoll           =  2;
	static const int CPCSensorHeadBackPressure  =  3;
	static const int CPCSensorHeadFrontPressure =  4;
	static const int CPCSensorPSD               =  5;
	static const int CPCJointMouth              =  6;
	static const int CPCSensorChinSwitch        =  7;
	static const int CPCJointLFRotator          =  8; //!< Left front leg
	static const int CPCJointLFElevator         =  9;
	static const int CPCJointLFKnee             = 10;
	static const int CPCSensorLFPaw             = 11;
	static const int CPCJointLHRotator          = 12; //!< Left hind leg
	static const int CPCJointLHElevator         = 13;
	static const int CPCJointLHKnee             = 14;
	static const int CPCSensorLHPaw             = 15;
	static const int CPCJointRFRotator          = 16; //!< Right front leg
	static const int CPCJointRFElevator         = 17;
	static const int CPCJointRFKnee             = 18;
	static const int CPCSensorRFPaw             = 19;
	static const int CPCJointRHRotator          = 20; //!< Right hind leg
	static const int CPCJointRHElevator         = 21;
	static const int CPCJointRHKnee             = 22;
	static const int CPCSensorRHPaw             = 23;
	static const int CPCJointTailPan            = 24; //!< Tail
	static const int CPCJointTailTilt           = 25;
	static const int CPCSensorThermoSensor      = 26;
	static const int CPCSensorBackSwitch        = 27;
	static const int CPCSensorAccelFB           = 28; //!< Front-back @see RobotInfo::BAccelOffset 
	static const int CPCSensorAccelLR           = 29; //!< Left-right @see RobotInfo::LAccelOffset 
	static const int CPCSensorAccelUD           = 30; //!< Up-down  @see RobotInfo::DAccelOffset 
	//@}

private:
	WorldState(const WorldState&); //!< don't use
	WorldState operator=(const WorldState&); //!< don't use
};

extern WorldState * state; //!< the global state object, is a shared memory region, created by MainObject

/*! @file
 * @brief Describes WorldState, maintains information about the robot's environment, namely sensors and power status, adds to RobotInfo
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.11 $
 * $State: Exp $
 * $Date: 2003/03/03 01:18:14 $
 */

#endif
