#include <OPENR/core_macro.h>
#include <OPENR/ObjcommTypes.h>
#include <OPENR/OPENR.h>
#include <OPENR/OPENRAPI.h>
#include <OPENR/OPENRMessages.h>
#include <OPENR/OPower.h>

#include "WorldState.h"
#include "Shared/get_time.h"
#include "Events/EventRouter.h"

#define GETD(cpc) (((float)sensor.GetData(cpc)->frame[0].value) / 1.0E6f) //!<@return value from OPEN-R, converted from micro in int to base in float
#define GETB(cpc) ((bool)sensor.GetData(cpc)->frame[0].value) //!<@return value from OPEN-R, as bool
#define GETSIG(cpc) ((word)sensor.GetData(cpc)->frame[0].signal) //!<@return signal from OPEN-R as word
#define GETDUTY(cpc) ((float)((OJointValue*)&sensor.GetData(cpc)->frame[0])->pwmDuty/512.0f) //!<@return duty cycle from OPEN-R as float; -1 (full reverse) to 0 (idle) to 1 (full forward)

const double WorldState::g=9.80665;
const double WorldState::IROORDist = 900.0;

WorldState * state=NULL;

WorldState::WorldState()
	: alwaysGenerateStatus(false), robotStatus(0), batteryStatus(0),
		lastSensorUpdateTime(0), mainProfile(), motionProfile(),
    wsser(NULL), curtime(get_time()) {
	for(unsigned int i=0; i< NumOutputs; i++)
		outputs[i]=0;
	for(unsigned int i=0; i< NumButtons; i++)
		buttons[i]=0;
	for(unsigned int i=0; i< NumSensors; i++)
		sensors[i]=0;
	for(unsigned int i=0; i< NumPIDJoints; i++)
		for(unsigned int j=0; j<3; j++)
			pids[i][j]=0;
	for(unsigned int i=0; i< NumPIDJoints; i++)
		pidduties[i]=0;
	memset(powerFlags,0,sizeof(unsigned int)*PowerSourceID::NumPowerSIDs);
	memset(button_times,0,sizeof(unsigned int)*NumButtons);
  wsser=new WorldStateSerializer();
}


/*! This will cause events to be posted
 *  @todo change to use most recent instead of oldest - is a buffer! */
void WorldState::read(OSensorFrameVectorData& sensor, EventRouter* er) {
	curtime=get_time();

  outputs[LFrLegOffset + RotatorOffset   ] = GETD(CPCJointLFRotator);
  outputs[LFrLegOffset + ElevatorOffset  ] = GETD(CPCJointLFElevator);
  outputs[LFrLegOffset + KneeOffset      ] = GETD(CPCJointLFKnee);
  pidduties[LFrLegOffset + RotatorOffset ] = GETDUTY(CPCJointLFRotator);
  pidduties[LFrLegOffset + ElevatorOffset] = GETDUTY(CPCJointLFElevator);
  pidduties[LFrLegOffset + KneeOffset    ] = GETDUTY(CPCJointLFKnee);
	
  outputs[RFrLegOffset + RotatorOffset   ] = GETD(CPCJointRFRotator);
  outputs[RFrLegOffset + ElevatorOffset  ] = GETD(CPCJointRFElevator);
  outputs[RFrLegOffset + KneeOffset      ] = GETD(CPCJointRFKnee);
  pidduties[RFrLegOffset + RotatorOffset ] = GETDUTY(CPCJointRFRotator);
  pidduties[RFrLegOffset + ElevatorOffset] = GETDUTY(CPCJointRFElevator);
  pidduties[RFrLegOffset + KneeOffset    ] = GETDUTY(CPCJointRFKnee);
	
  outputs[LBkLegOffset + RotatorOffset   ] = GETD(CPCJointLHRotator);
  outputs[LBkLegOffset + ElevatorOffset  ] = GETD(CPCJointLHElevator);
  outputs[LBkLegOffset + KneeOffset      ] = GETD(CPCJointLHKnee);
  pidduties[LBkLegOffset + RotatorOffset ] = GETDUTY(CPCJointLHRotator);
  pidduties[LBkLegOffset + ElevatorOffset] = GETDUTY(CPCJointLHElevator);
  pidduties[LBkLegOffset + KneeOffset    ] = GETDUTY(CPCJointLHKnee);

  outputs[RBkLegOffset + RotatorOffset   ] = GETD(CPCJointRHRotator);
  outputs[RBkLegOffset + ElevatorOffset  ] = GETD(CPCJointRHElevator);
  outputs[RBkLegOffset + KneeOffset      ] = GETD(CPCJointRHKnee);
  pidduties[RBkLegOffset + RotatorOffset ] = GETDUTY(CPCJointRHRotator);
  pidduties[RBkLegOffset + ElevatorOffset] = GETDUTY(CPCJointRHElevator);
  pidduties[RBkLegOffset + KneeOffset    ] = GETDUTY(CPCJointRHKnee);

  // Get head tilt,pan,roll joint angles
  outputs[HeadOffset+TiltOffset] = GETD(CPCJointNeckTilt);
  outputs[HeadOffset+PanOffset ] = GETD(CPCJointNeckPan);
  outputs[HeadOffset+RollOffset] = GETD(CPCJointNeckRoll);
  pidduties[HeadOffset+TiltOffset] = GETDUTY(CPCJointNeckTilt);
  pidduties[HeadOffset+PanOffset ] = GETDUTY(CPCJointNeckPan);
  pidduties[HeadOffset+RollOffset] = GETDUTY(CPCJointNeckRoll);

	outputs[TailOffset+TiltOffset] = GETD(CPCJointTailTilt);
	outputs[TailOffset+PanOffset]  = GETD(CPCJointTailPan);
	pidduties[TailOffset+TiltOffset] = GETDUTY(CPCJointTailTilt);
	pidduties[TailOffset+PanOffset]  = GETDUTY(CPCJointTailPan);

	outputs[MouthOffset] = GETD(CPCJointMouth);
	pidduties[MouthOffset] = GETDUTY(CPCJointMouth);


  // Get foot switches
	chkEvent(er,LFrPawOffset,GETB(CPCSensorLFPaw),"LFrPaw");
  chkEvent(er,RFrPawOffset,GETB(CPCSensorRFPaw),"RFrPaw");
  chkEvent(er,LBkPawOffset,GETB(CPCSensorLHPaw),"LBkPaw");
  chkEvent(er,RBkPawOffset,GETB(CPCSensorRHPaw),"RBkPaw");

  // Get buttons
  chkEvent(er,ChinButOffset,  ((OSwitchValue)sensor.GetData(CPCSensorChinSwitch)->frame[0].value == oswitchON)?1:0,"ChinBut");
  chkEvent(er,BackButOffset,  ((OSwitchValue)sensor.GetData(CPCSensorBackSwitch)->frame[0].value == oswitchON)?1:0,"BackBut");
  chkEvent(er,HeadFrButOffset,GETD(CPCSensorHeadFrontPressure),"HeadFrBut");
  chkEvent(er,HeadBkButOffset,GETD(CPCSensorHeadBackPressure),"HeadBkBut");

  // Get IR distance sensor
	sensors[IRDistOffset]=((float)sensor.GetData(CPCSensorPSD)->frame[0].value) / 1000.0;

  // Get acceleration sensors
	sensors[BAccelOffset] = GETD(CPCSensorAccelFB);
	sensors[LAccelOffset] = GETD(CPCSensorAccelLR);
	sensors[DAccelOffset] = GETD(CPCSensorAccelUD);

	sensors[ThermoOffset] = GETD(CPCSensorThermoSensor);

	unsigned int dif=curtime-lastSensorUpdateTime;
	lastSensorUpdateTime=curtime;
	er->postEvent(EventBase::sensorEGID,SensorSourceID::UpdatedSID,EventBase::statusETID,dif);
}

/*! This will cause events to be posted */
void WorldState::read(const OPowerStatus& power, EventRouter* er) {
	std::string actnames[PowerSourceID::NumPowerSIDs];
	std::string denames[PowerSourceID::NumPowerSIDs];
	unsigned int actmasks[PowerSourceID::NumPowerSIDs];
	memset(actmasks,0,sizeof(unsigned int)*PowerSourceID::NumPowerSIDs);

	//RobotStatus
	chkPowerEvent(PowerSourceID::PauseSID,          power.robotStatus,orsbPAUSE,                        "Pause",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::MotorPowerSID,     power.robotStatus,orsbMOTOR_POWER,                  "MotorPower",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::VibrationSID,      power.robotStatus,orsbVIBRATION_DETECT,             "Vibration",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ExternalPortSID,   power.robotStatus,orsbEX_PORT_CONNECTED,            "ExternalPort",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::StationConnectSID, power.robotStatus,orsbSTATION_CONNECTED,            "StationConnect",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ExternalPowerSID,  power.robotStatus,orsbEX_POWER_CONNECTED,           "ExternalPower",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryConnectSID, power.robotStatus,orsbBATTERY_CONNECTED,            "BatteryConnect",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ChargingSID,       power.robotStatus,orsbBATTERY_CHARGING,             "BatteryCharging",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryFullSID,    power.robotStatus,orsbBATTERY_CAPACITY_FULL,        "BatteryFull",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::LowPowerWarnSID,   power.robotStatus,orsbBATTERY_CAPACITY_LOW,         "BatteryLow",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverChargedSID,    power.robotStatus,orsbBATTERY_OVER_CURRENT,         "BatteryOverCurrent",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverheatingSID,    power.robotStatus,orsbBATTERY_OVER_TEMP_DISCHARGING,"BatteryOverTempDischarge",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverheatingSID,    power.robotStatus,orsbBATTERY_OVER_TEMP_CHARGING,   "BatteryOverTempCharge",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbBATTERY_ERROR_OF_CHARGING,    "BatteryChargeError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbERROR_OF_PLUNGER,             "PlungerError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::PowerGoodSID,      power.robotStatus,orsbOPEN_R_POWER_GOOD,            "PowerGood",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbERROR_OF_FAN,                 "FanError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::DataFromStationSID,power.robotStatus,orsbDATA_STREAM_FROM_STATION,     "DataFromStation",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RegisterUpdateSID, power.robotStatus,orsbREGISTER_UPDATED_BY_STATION,  "RegisterUpdate",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ErrorSID,          power.robotStatus,orsbRTC_ERROR,                    "RTCError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RTCSID,            power.robotStatus,orsbRTC_OVERFLOW,                 "RTCOverflow",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RTCSID,            power.robotStatus,orsbRTC_RESET,                    "RTCReset",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::RTCSID,            power.robotStatus,orsbRTC_SET,                      "RTCSet",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::SpecialModeSID,    power.robotStatus,orsbSPECIAL_MODE,                 "SpecialMode",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BMNDebugModeSID,   power.robotStatus,orsbBMN_DEBUG_MODE,               "BMNDebugMode",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::ChargerStatusSID,  power.robotStatus,orsbCHARGER_STATUS,               "ChargerStatus",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::PlungerSID,        power.robotStatus,orsbPLUNGER,                      "Plunger",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::SuspendedSID,      power.robotStatus,orsbSUSPENDED,                    "Suspended",actnames,denames,actmasks);

	//BatteryStatus
	chkPowerEvent(PowerSourceID::ErrorSID,        power.batteryStatus,obsbERROR_CODE_MASK,             "BatteryError",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryEmptySID, power.batteryStatus,obsbFULLY_DISCHARGED,            "FullyDischarged",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryFullSID,  power.batteryStatus,obsbFULLY_CHARGED,               "FullyCharged",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::DischargingSID,  power.batteryStatus,obsbDISCHARGING,                 "Discharging",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::BatteryInitSID,  power.batteryStatus,obsbINITIALIZED,                 "BatteryInit",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::LowPowerWarnSID, power.batteryStatus,obsbREMAINING_TIME_ALARM,        "RemainingTimeAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::LowPowerWarnSID, power.batteryStatus,obsbREMAINING_CAPACITY_ALARM,    "RemainingCapacityAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::TermDischargeSID,power.batteryStatus,obsbTERMINATED_DISCHARGING_ALARM,"TermDischargeAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverheatingSID,  power.batteryStatus,obsbOVER_TEMP_ALARM,             "OverTempAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::TermChargeSID,   power.batteryStatus,obsbTERMINATED_CHARGING_ALARM,   "TermChargeAlarm",actnames,denames,actmasks);
	chkPowerEvent(PowerSourceID::OverChargedSID,  power.batteryStatus,obsbOVER_CHARGED_ALARM,          "OverChargeAlarm",actnames,denames,actmasks);
	
	sensors[PowerRemainOffset] = power.remainingCapacity/100.0;
	sensors[PowerThermoOffset] = power.temperature/100.0;
	sensors[PowerCapacityOffset] = power.fullyChargedCapacity;
	sensors[PowerVoltageOffset] = power.voltage/1000.0;
	sensors[PowerCurrentOffset] = power.current;

	//only generate status events when a change happens
	for(unsigned int i=0; i<PowerSourceID::NumPowerSIDs; i++) {
		if(actmasks[i]) { //now on
			if(!powerFlags[i]) //was off: activation
				er->postEvent(EventBase::powerEGID,i,EventBase::activateETID,0,actnames[i],1);
			else if(actmasks[i]!=powerFlags[i]) //already on - change? : status
				er->postEvent(EventBase::powerEGID,i,EventBase::statusETID,0,actnames[i],1);
		} else { // now off
			if(powerFlags[i]) //was on: deactivation
				er->postEvent(EventBase::powerEGID,i,EventBase::deactivateETID,0,denames[i],0);
		}
		powerFlags[i]=actmasks[i];
	}

	er->postEvent(EventBase::powerEGID,PowerSourceID::UpdatedSID,EventBase::statusETID,0);
}

void WorldState::chkEvent(EventRouter* er, unsigned int off, float newval, const char* name) {
	ButtonSourceID::ButtonSourceID_t sid=static_cast<ButtonSourceID::ButtonSourceID_t>(off); //should be a direct conversion
	if(newval>=0.1) { //now on
		if(buttons[sid]<0.1) { //was off: activation
			er->postEvent(EventBase::buttonEGID,sid,EventBase::activateETID,0,name,newval);
			button_times[sid]=curtime;
		} else if(alwaysGenerateStatus || buttons[sid]!=newval) { //already on - always or change? : status
			unsigned int dur=curtime-button_times[sid];
			er->postEvent(EventBase::buttonEGID,sid,EventBase::statusETID,dur,name,newval);
		}
	} else { //now off
		if(buttons[sid]>=0.1) { //was on: deactivation
			unsigned int dur=curtime-button_times[sid];
			button_times[sid]=0;
			er->postEvent(EventBase::buttonEGID,sid,EventBase::deactivateETID,dur,name,0);
		}
	}
	//update value
	buttons[sid]=newval;
}

/*! @file
 * @brief Implements WorldState, maintains information about the robot's environment, namely sensors and power status
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.8 $
 * $State: Exp $
 * $Date: 2003/03/03 01:18:14 $
 */
