//-*-c++-*-
#ifndef INCLUDED_MotionManager_h
#define INCLUDED_MotionManager_h

#include "MotionCommand.h"
#include "OutputCmd.h"
#include "OutputPID.h"
#include "SharedMotion.h"
#include "MotionManagerMsg.h"
#include "Shared/RobotInfo.h"
#include "Shared/MutexLock.h"
#include "Shared/ListMemBuf.h"

#include <OPENR/OPENR.h>
#include <OPENR/OPENRAPI.h>
#include <OPENR/OSubject.h>
#include <OPENR/ObjcommEvent.h>
#include <OPENR/OObject.h>

extern unsigned int _MMaccID; //!< set by InitAccess(), unique for each process (that's why it's static)

//! The purpose of this class is to serialize access to the MotionCommands and simplify their sharing between memory spaces
/*! Since MotionObject and MainObject run as separate processes, they could potentially try to access the
 *  same motion command at the same time, leading to unpredictable behavior.  The MotionManager enforces
 *  a set of locks to solve this\n
 *  The other problem is that we are sharing between processes.  MotionManager will do what's necessary to distribute new
 *  MotionCommand's to all the processes (currently just MainObj and MotoObj)\n
 *  You should be able to create and add a new motion in one line:
 *  @code
 *  motman->addMotion(SharedMotion<YourMC>([arg1,[arg2,...]]) [, ...]);
 *  @endcode
 *  But if you want to do some more initializations not handled by the constructor (the @p arg1, @p arg2, ...
 *  params) then you would want to do something like the following:
 *  @code
 *  SharedMotion<YourMC> tmpvar([arg1,[arg2,...]]);
 *  tmpvar.MCBase()->cmd1();
 *  tmpvar.MCBase()->cmd2();
 *  //...
 *  motman->addMotion(tmpvar [, ...]);
 *  @endcode
 *  @warning don't make calls to the underlying MotionCommand after it has been added... you must check it
 *  out at this point or risk concurrent access problems.
 *
 *  @see MotionCommand for information on creating new motion primitives.
 *
 *  @see MMAccessor for information on accessing motions after you've added them to MotionManager.
 *
 *  @todo clean up lookup()/numAcc mess
 */
class MotionManager {
public:
	//! This is the number of processes which will be accessing the MotionManager
	/*! Probably just MainObject and MotionObject
	 *  Isn't really a hard maximum, but should be actual expected, need to know when they're all connected */
	static const unsigned int MAX_ACCESS=2;

	static const unsigned int MAX_MOTIONS=64;   //!< This is the maximum number of Motions which can be managed, can probably be increased reasonably without trouble

	typedef MotionManagerMsg::MC_ID MC_ID;      //!< use this type when referring to the ID numbers that MotionManager hands out
	static const MC_ID invalid_MC_ID=(MC_ID)-1; //!< for errors and undefined stuff

	MotionManager();                            //!< Constructor, sets all the outputs to 0
	void InitAccess(OSubject* subj);            //!< @b LOCKS @b MotionManager Everyone who is planning to use the MotionManager needs to call this before they access it or suffer a horrible fate
	void receivedMsg(const ONotifyEvent& event); //!< @b LOCKS @b MotionManager This gets called by an OObject when it receives a message from one of the other OObject's MotionManagerComm Subject

	//!@name MotionCommand Safe
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd& cmd); //!< @b LOCKS @b MotionManager Requests a value be set for the specified output, copies cmd across frames
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd& cmd, unsigned int frame); //!< @b LOCKS @b MotionManager Requests a value be set for the specified output in the specified frame
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd cmd[NumFrames]); //!< @b LOCKS @b MotionManager Requests a value be set for the specified output across frames
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputPID& pid); //!< @b LOCKS @b MotionManager Requests a PID be set for the specified output, notice that this might be overruled by a higher priority motion
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd& cmd, const OutputPID& pid); //!< @b LOCKS @b MotionManager Requests a value and PID be set for the specified output
	void setOutput(const MotionCommand* caller, unsigned int output, const OutputCmd cmd[NumFrames], const OutputPID& pid); //!< @b LOCKS @b MotionManager Requests a value and PID be set for the specified output
	const OutputCmd& getOutputCmd(unsigned int output) const { return cmds[output]; } //!< Returns the value of the output last sent to the OS.  Note that this will differ from the sensed value in state, even when staying still.  There is no corresponding getOutputPID because this value *will* duplicate the value in state.
	void setPriority(MC_ID mcid, float p) { cmdlist[mcid].priority=p; }//!< sets the priority level of a MotionCommand
	float getPriority(MC_ID mcid) const { return cmdlist[mcid].priority; } //!< returns priority level of a MotionCommand
	//@}

	//! @todo create templated method for adding motions that don't require people to touch SharedMotion directly
	//@{
	MC_ID addMotion(const SharedMotionBase& sm); //!< @b LOCKS @b MotionManager Creates a new MotionCommand, automatically sharing it between processes (there is some lag time here)
	MC_ID addMotion(const SharedMotionBase& sm, float priority); //!< @b LOCKS @b MotionManager allows a quick way to set a priority level of a new MotionCommand
	MC_ID addMotion(const SharedMotionBase& sm, bool autoprune); //!< @b LOCKS @b MotionManager allows a quick was to set the autoprune flag
	MC_ID addMotion(const SharedMotionBase& sm, float priority, bool autoprune); //!< @b LOCKS @b MotionManager Call one of these to add a MotionCommand to the MotionManager, using the SharedMotion class
	void removeMotion(MC_ID mcid); //!< @b LOCKS @b MotionManager removes the specified MotionCommand
	//@}

	//@{
	MotionCommand * checkoutMotion(MC_ID mcid,bool block=true) { return checkoutMotion(mcid,block,_MMaccID); } //!< locks the command and possibly performs RTTI conversion; supports recursive calls
	void checkinMotion(MC_ID mcid); //!< marks a MotionCommand as unused
	MotionCommand * peekMotion(MC_ID mcid) { return cmdlist[mcid].baseaddrs[_MMaccID]; } //!< allows access to a MotionCommand without checking it out @warning @b never call a function based on this, only access member fields through it
	unsigned int checkoutLevel(MC_ID mcid) { return cmdlist[mcid].lock.get_lock_level(); } //!< returns the number of times @a mcid has been checked out minus the times it's been checked in
	//@}

	//@{
	inline MC_ID begin() const         { return begin(_MMaccID);    } //!< returns the MC_ID of the first MotionCommand
	inline MC_ID next(MC_ID cur) const { return next(cur,_MMaccID); } //!< returns the MC_ID of MotionCommand following the one that is passed
	inline MC_ID end() const           { return cmdlist.end();      } //!< returns the MC_ID of the one-past-the-end MotionCommand (like the STL)
	inline unsigned int size() const   { return cmdlist.size();     } //!< returns the number of MotionCommands being managed
	//@}

	//@{
	void lock()    { MMlock.lock(_MMaccID); } //!< gets an exclusive lock on MotionManager - functions marked @b LOCKS @b MotionManager will cause (and require) this to happen automatically
	bool trylock() { return MMlock.try_lock(_MMaccID); } //!< tries to get a lock without blocking
	void release() { MMlock.release(); } //!< releases a lock on the motion manager
	//@}

	//@{
	void getOutputs(float outputs[NumFrames][NumOutputs]);  //!< @b LOCKS @b MotionManager called by MotionObject to fill in the output values for the next ::NumFrames frames (only MotoObj should call this...)
	void updateWorldState();                                //!< call this when you want MotionManager to set the WorldState to reflect what things should be for unsensed outputs (LEDs, ears) (only MotoObj should be calling this...)
	bool updatePIDs(OPrimitiveID primIDs[NumOutputs]);      //!< call this when you want MotionManager to update modified PID values, returns true if changes made (only MotoObj should be calling this...)
	//@}

	//! holds the full requested value of an output
	class OutputState {
	public:
		//!@name Constructors
		//!Constructor
		OutputState()
			: priority(0),mcid(MotionManager::invalid_MC_ID), pid()
		{}
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd cmds[NumFrames])
			: priority(pri),mcid(mc), pid(DefaultPIDs[out])
		{
			for(unsigned int i=0; i<NumFrames; i++)
				frames[i]=cmds[i];
		}
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd& cmd)
			: priority(pri),mcid(mc), pid(DefaultPIDs[out])
		{
			for(unsigned int i=0; i<NumFrames; i++)
				frames[i]=cmd;
		}
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd& cmd, unsigned int frame)
			: priority(pri),mcid(mc), pid(DefaultPIDs[out])
		{
				frames[frame]=cmd;
		}
		OutputState(unsigned int /*out*/, float pri, MC_ID mc, const OutputPID& p)
			: priority(pri),mcid(mc), pid(p)
		{}
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd cmds[NumFrames], const OutputPID& p)
			: priority(pri),mcid(mc), pid(p)
		{
			for(unsigned int i=0; i<NumFrames; i++)
				frames[i]=cmds[i];
		}
		OutputState(unsigned int out, float pri, MC_ID mc, const OutputCmd& cmd, const OutputPID& p)
			: priority(pri),mcid(mc), pid(p)
		{
			for(unsigned int i=0; i<NumFrames; i++)
				frames[i]=cmd;
		}
		//@}
		float priority;             //!< priority level
		MC_ID mcid;                 //!< MC_ID of requester
		OutputCmd frames[NumFrames]; //!< values of output planned ahead
		OutputPID pid;               //!< pid of output
	};

	static const float kIgnoredPriority;    //!< won't be expressed, handy if you want to temporarily pause something
	static const float kBackgroundPriority; //!< will only be expressed if *nothing* else is using that joint
	static const float kLowPriority;        //!< for stuff that's not background but lower than standard
	static const float kStdPriority;        //!< for every-day commands
	static const float kHighPriority;       //!< for stuff that should over ride standard stuff
	static const float kEmergencyPriority;  //!< for really important stuff, such as the emergency stop

protected:
	//! used to request pids for a given joint
	struct PIDUpdate {
		//!constructor
	 	PIDUpdate() : joint((unsigned int)-1) {}
		//!constructor
		PIDUpdate(unsigned int j, const float p[3]) : joint(j) {
			for(unsigned int i=0; i<3; i++)
				pids[i]=p[i];
		}
		unsigned int joint; //!< the joint ID (see RobotInfo.h for offset values)
		float pids[3]; //!< the PID values to use (see ::Pid )
	};
	ListMemBuf<PIDUpdate,NumPIDJoints> pidchanges;  //!< stores PID updates, up to one per joint (if same is set more than once, it's just overwrites previous update)
	void setPID(unsigned int j, const float p[3]); //!< @b LOCKS @b MotionManager you can call this to set the PID values directly (instead of using a motion command) Be careful though

	typedef unsigned short accID_t; //!< type to use to refer to accessors of MotionManager (or its locks)

	unsigned int cleanup(accID_t id); //!< called by getOutputs() automatically, causes autopruning, assumes all MCs are checked out

	accID_t func_begin(); //!< called at the begining of many functions to lock MotionManager and get the process's accessor ID
	void func_end(); //!< called at the end of a function which called func_begin() to release it
	template<class T> T func_end(T val) { func_end(); return val; } //!< same as func_end(), except passes return value through

	MotionCommand * checkoutMotion(MC_ID mcid, bool block, accID_t accID); //!< used by internal functions to check out a motion without duplicating the work of looking up the accessor ID

	inline MC_ID begin(accID_t acc) const { return skip_ahead(cmdlist.begin(),acc); } //!< used by internal functions to scan through motion commands without duplicating the work of looking up the accessor ID each iteration
	inline MC_ID next(MC_ID cur, accID_t acc) const { return skip_ahead(cmdlist.next(cur),acc); } //!< used by internal functions to scan through motion commands without duplicating the work of looking up the accessor ID each iteration
	MC_ID skip_ahead(MC_ID mcid, accID_t acc) const; //!< during iteration, skips over motioncommands which are still in transit from on OObject to another
		
	//!All the information we need to maintain about a MotionCommand
	struct CommandEntry {
		//! Constructor, sets everything to basics
		CommandEntry() : lastAccessor((unsigned short)-1),lock(),priority(MotionManager::kStdPriority) {
			for(unsigned int i=0; i<MAX_ACCESS; i++) {
				baseaddrs[i]=NULL;
				rcr[i]=NULL;
			}
		}
		MotionCommand * baseaddrs[MAX_ACCESS]; //!< for each accessor, the base address of the motion command
		RCRegion * rcr[MAX_ACCESS];            //!< for each accessor the shared memory region that holds the motion command
		accID_t lastAccessor;                  //!< the ID of the last accessor to touch the command (which implies if it wants to touch this again, we don't have to convert again)
		MutexLock<MAX_ACCESS> lock;            //!< a lock to maintain mutual exclusion
		float priority;                        //!< MotionCommand's priority level
	private:
		CommandEntry(const CommandEntry&); //!< this shouldn't be called...
		CommandEntry& operator=(const CommandEntry&); //!< this shouldn't be called...
	};
	ListMemBuf<CommandEntry,MAX_MOTIONS,MC_ID> cmdlist;     //!< the list where MotionCommands are stored, remember, we're in a shared memory region with different base addresses - no pointers!
	MC_ID cur_cmd; //!< MC_ID of the MotionCommand currently being updated by getOutputs(), or NULL if not in getOutputs.  This is used by the setOutput()'s to tell which MotionCommand is calling


	inline MC_ID pop_free() { return cmdlist.new_front(); } //!<pulls an entry from cmdlist's free section and returns its index
	inline void push_free(MC_ID a) { cmdlist.erase(a); }    //!<puts an entry back into cmdlist's free section

	MutexLock<MAX_ACCESS> MMlock;          //!< The main lock for the class

	typedef ListMemBuf<OutputState,MAX_MOTIONS> cmdstatelist_t; //!< shorthand for a list of OutputState's
	cmdstatelist_t cmdstates[NumOutputs];  //!< requested positions by each of the MC's for each of the outputs
	float cmdSums[NumOutputs];             //!<Holds the final values for the outputs of the last frame generated
	OutputCmd cmds[NumOutputs];            //!<Holds the weighted values and total weight for the outputs of the last frame

	accID_t numAcc;                        //!<The number of accessors who have registered with InitAccess()
	//	MCRegistrar accRegs[MAX_ACCESS];       //!<The MCRegistrar for each process (accessor), to allow for conversions
	OSubject* subjs[MAX_ACCESS];           //!<The OSubject for each process (accessor) on which it should br broadcast when a command is added

private:
	MotionManager(const MotionManager&); //!< this shouldn't be called...
	MotionManager& operator=(const MotionManager&); //!< this shouldn't be called...
};

//!anyone who #includes MotionManager.h will be wanting to use the global motman... don't want multiple of these! created by MotoObj
extern MotionManager * motman;

/*! @file
 * @brief Describes MotionManager, simplifies sharing of MotionCommand's and provides mutual exclusion to their access
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.11 $
 * $State: Exp $
 * $Date: 2003/03/04 05:46:06 $
 */

#endif
