#ifndef INCLUDED_MMAccessor_h_
#define INCLUDED_MMAccessor_h_

#include "MotionManager.h"

//! This class allows convenient ways of accessing a motion command
/*! Since MotionCommands must be checked out of the motion manager and then checked back
 *  in when they are done, this is a common source of errors, leading to deadlock.  This class
 *  will check the motion out when it's created, and check it back in when it goes out of scope\n
 *  It supports recursive checkin/checkouts. \n
 *  Uses global ::motman
 *
 *  So, instead of doing things like this:
 *  @code
 *  YourMotionCommand* ymc = dynamic_cast<YourMotionCommand*>(motman->checkoutMotion(your_mc_id));
 *  //do 'stuff' with ymc, e.g.: ymc->rollOver();
 *  motman->checkinMotion(your_mc_id);
 *  @endcode
 *  ...which can be error prone in many regards - if 'stuff' returns without checking in, or you
 *  forget to check in, or you get lazy and leave it checked out longer than you should, which can
 *  cause general performance issues (or worse, deadlock)
 *  Using MMAccessor makes it much easier to solve these problems, and is easier to code:
 *  @code
 *  MMAccessor<YourMotionCommand> mma(your_mc_id);
 *  //do 'stuff' with mma, e.g.: mma->rollOver();
 *  @endcode
 *  We can call a return at any point and the motion command will automatically be checked in, and
 *  since C++ guarrantees that the destructor of mma will be called, we don't have to worry about
 *  forgetting about it.  We can limit the scope by placing {}'s around the segment in question:
 *  @code
 *  //pre-stuff
 *  {
 *    MMAccessor<YourMotionCommand> mma(your_mc_id);
 *    //do 'stuff' with mma, e.g.: mma->rollOver();
 *  }
 *  //post-stuff - has no knowledge of mma, out of its scope
 *  @endcode
 *  And, for those astute enough to notice that the theoretical @a rollOver() function is called on
 *  MMAccessor when it's actually a member of YourMotionCommand, this is because MMAccessor behaves as a
 *  'smart pointer', which overloads operator->() so it is fairly transparent to use.
 *
 *  See also the templated checkin(Ret_t ret) function for more examples of streamlined usage.
 *
 *  MMAccessor is a small class, you may consider passing it around instead of a MotionManager::MC_ID
 *  if appropriate.  (Would be appropriate to avoid multiple checkin/outs in a row from different
 *  functions, but not as appropriate for storage and reuse of the same MMAccessor.
 */
template<class MC_t>
class MMAccessor {
 public:

	//! constructor, checks out by default
	/*! @param id the motion command to check out
	 *  @param ckout if true (default) will checkout upon creation.  otherwise it just gets current address (so you can peek at member fields, which should be safe) */
	MMAccessor(MotionManager::MC_ID id,bool ckout=true) : mc_id(id), mcptr(NULL) {
		if(ckout)
			checkout();
		else
			mcptr=(MC_t*)motman->peekMotion(id);
	}

	//! copy constructor - will reference the same mc_id - checkin/checkouts are NOT independent between this and @a a - they will be linked
	MMAccessor(const MMAccessor& a) : mc_id(a.mc_id), mcptr(a.mcptr) {}

	//! destructor, checks in if needed
	~MMAccessor() {
		while(motman->checkoutLevel(mc_id)>0)
			checkin();
	}

	//! allows assignment of MMAccessor's, similar to the copy constructor - the two MMAccessor's will control the same MotionCommand
	MMAccessor<MC_t> operator=(const MMAccessor<MC_t>& a) {
		mc_id=a.mc_id;
		mcptr=a.mcptr;
		return *this;
	}
	
	//! So you can check out if not done by default (or you checked in already)
	inline MC_t* checkout() {
		return mcptr=dynamic_cast<MC_t*>(motman->checkoutMotion(mc_id)); //!< @test can this be a dynamic_cast?
	}

	//! Returns the motion command's address so you can call functions
	/*! @todo investigate operator->() and related */
	inline MC_t* mc() const { return mcptr; }

	//! Checks in the motion
	/*! Don't forget, you can also just limit the scope using extra { }'s */
	inline void checkin() {
		motman->checkinMotion(mc_id);
	}

	//! Checks in the motion, passing through the value it is passed.
	/*! @return the same value it's passed
	 *
	 *  Useful in situations like this:
	 *  @code
	 *  MMAccessor<myMC> mine(myMC_id);
	 *  if(mine.mc()->foo())
	 *    //do something with motman here
	 *  @endcode
	 *  But we want to check in @a mine ASAP - if we don't reference it
	 *  anywhere in the if statement, we're leaving the MC locked longer
	 *  than we need to.  How about instead doing this:
	 *  @code
	 *  bool cond;
	 *  {MMAccessor<myMC> mine(myMC_id); cond=mine.mc()->foo();}
	 *  if(cond)
	 *    //do something with motman here
	 *  @endcode
	 *  But that uses an extra variable... ewwww... so use this function
	 *  as a pass through to checkin the MC:
	 *  @code
	 *  MMAccessor<myMC> mine(myMC_id);
	 *  if(mine.checkin(mine.mc()->foo()))
	 *    //do something with motman here
	 *  @endcode*/
	template<class Ret_t> Ret_t checkin(Ret_t ret) {
		checkin();
		mcptr=NULL;
		return ret;
	}

	MC_t* operator->() { return mc(); } //!< smart pointer to the underlying MotionCommand
	const MC_t* operator->() const { return mc(); } //!< smart pointer to the underlying MotionCommand
	MC_t& operator*() { return *mc(); } //!< smart pointer to the underlying MotionCommand
	const MC_t& operator*() const { return *mc(); } //!< smart pointer to the underlying MotionCommand
	MC_t& operator[](int i) { return mc()[i]; } //!< smart pointer to the underlying MotionCommand
	const MC_t& operator[](int i) const { return mc()[i]; } //!< smart pointer to the underlying MotionCommand

 protected:
	MotionManager::MC_ID mc_id; //!< the MC_ID that this Accessor was constructed with
	MC_t* mcptr;                //!< a pointer to the motion command, should always be valid even when not checked out so you can access member fields (which is reasonably safe)
};

/*! @file
 * @brief Defines MMAccessor, allows convenient ways to check MotionCommands in and out of the MotionManager
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.6 $
 * $State: Exp $
 * $Date: 2003/01/22 05:33:27 $
 */

#endif
