//-*-c++-*-
#ifndef INCLUDED_CompareTrans_h_
#define INCLUDED_CompareTrans_h_

#include "Transition.h"
#include "Events/EventRouter.h"

//! causes a transition if a value (through a pointer) goes above a given value
/*! You will need to specify an event mask which will be listened for.  This event
 *  will then be listened for - each time it is received, CompareTrans will check
 *  the values for possible activation.
 *
 *  For example, if you want to transition when the IR sensor goes below, say 200,
 *  pass &state->sensors[IRDistOffset], CompareTrans::LT, 200, and
 *  EventBase(EventBase::sensorEGID,SensorSourceID::UpdatedSID,EventBase::statusETID)
 *  as the polling event.  Or a timer event to just check at a certain interval.
 *
 *  If you pass a class as the templated type, only requires that < operator is
 *  defined for comparing inequality, == for equality, and a copy constructor (CompareTrans
 *  holds a protected copy of the value)
 *  
 *  Passing NULL as the value to monitor will cause a transition on the first event received
 */
template<class T>
class CompareTrans : public Transition, public EventListener {
public:
	//! use these values to sepecify what kind of comparison should be made to test for activation
	enum Test_t {
		LT, //!< less than
		GT, //!< greater than
		LTE, //!< less than or equal
		GTE, //!< greater than or equal
		EQ, //!< equal
		NE //!< not equal
	};
	
	//! constructor, see class notes for information
  CompareTrans(StateNode* source, StateNode* destination, const T* monitor, Test_t test, const T& value, const EventBase& poll)
		: Transition(source,destination), EventListener(), mon(monitor), tst(test), val(value), poller(poll)
	{ }
	
	//!starts listening
	virtual void enable() { erouter->addListener(this,poller); }

	//!stops listening
	virtual void disable() { erouter->forgetListener(this); }

	//!don't care about the event, just a pulse to check the values
	virtual void processEvent(const EventBase&) {
		switch(tst) {
		case LT:
			if(*mon<val) activate();
			break;
		case GT:
			if(val<*mon) activate();
			break;
		case LTE:
			if(!(val<*mon)) activate();
			break;
		case GTE:
			if(!(*mon<val)) activate();
			break;
		case EQ:
			if(*mon==val) activate();
			break;
		case NE:
			if(!(*mon==val)) activate();
			break;
		}
	}

protected:
	const T* mon; //!< address of value to monitor
	Test_t tst; //!< test to make
	T val; //!< value to compare against
	EventBase poller; //!< event to listen to, when it comes, compare the values

private:
	CompareTrans(const CompareTrans& node); //!< don't call this
	CompareTrans operator=(const CompareTrans& node); //!< don't call this
};

/*! @file
 * @brief Defines CompareTrans, which causes a transition if a value (through a pointer) goes above a given value
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2003/03/01 20:53:32 $
 */

#endif
