//-*-c++-*-
#ifndef INCLUDED_StateNode_h_
#define INCLUDED_StateNode_h_

#include "Transition.h"
#include "Behaviors/BehaviorBase.h"
#include <vector>

//! Recursive data structure - both a state machine controller as well as a node within a state machine itself
/*! Override setup() to setup your own Transition and StateNode network.*/
class StateNode  : public BehaviorBase {
public:
	//!constructor
	StateNode() : BehaviorBase(), parent(NULL), transitions(), issetup(false), retain(true), nodes(), name(NULL) {
		name=new char[3];
		name[0]='-';
		name[1]='?';
		name[2]='\0';
	}

	//!constructor, pass a name to use, calls setName(n) 
	StateNode(const char* n, StateNode* p=NULL) : BehaviorBase(), parent(p), transitions(), issetup(false), retain(true), nodes(), name(NULL) {
		setName(n);
	}

	//!destructor, frees memory used by its outgoing transitions (be careful of incoming ones - they're still around!), and calls RemoveReference() on subnodes
	virtual ~StateNode();

	//!Adds the specified StateTransition to the transition table
	virtual void addTransition(Transition* trans);

	//!Returns the std::vector of transitions so you can modify them yourself if need be
	std::vector<Transition*>& getTransitions() { return transitions; }

	//!Adds a StateNode to #nodes so it can be automatically deleted later, returns what it's passed (for convenience), calls AddReference() on @a node
	virtual StateNode* addNode(StateNode* node) { nodes.push_back(node); node->AddReference(); return node; }

	//!Returns the std::vector of nodes so you can modify them yourself if need be
	std::vector<StateNode*>& getNodes() { return nodes; }

	//!Sets the retain flag - if not retained, will RemoveReference() subnodes upon DoStop() and recreate them on DoStart (by calling setup()) - may be too expensive to be worth saving memory...
	void setRetain(bool f) { retain=f; }

	//!Transitions should call this when you are entering the state, so it can enable its transitions
	void DoStart();

	//!This is called by DoStart() when you should setup the network
	virtual void setup() {issetup=true;}

	//!Transitions should call this when you are leaving the state, so it can disable its transitions
	void DoStop();
	
	//!set name to @a n.  Makes a copy of string, so you can throw away @a n later.
	void setName(const char* n);

	//!returns name of StateNode - the first character will be '#' if it's active and '-' if it's inactive.  So strcmp(getName()+1,...) if you want to check name.
	virtual const char* getName() const { return name; }

	//!Doesn't do anything, supplied here so you don't have to (since events will probably be going to Transition's, not here)
	virtual void processEvent(const EventBase&) {}

	//!called by a subnode when it is being DoStart()'ed
	void transitionTo(StateNode* n);

	//!called by a subnode when it is being DoStop()'ed
	void transitionFrom(StateNode* n);

protected:
	//Node Stuff:
	//! pointer to the machine that contains this node
	StateNode* parent;
	//! a vector of outgoing transitions
	std::vector<Transition*> transitions;
	
	//Machine Stuff:
	//! this is set to true if the network has been setup but not destroyed (i.e. don't need to call setupSubNodes again)
	bool issetup;
	//! this is set to true if the network should be retained between activations.  Otherwise it's deleted upon DoStop(). (or at least RemoveReference() is called on subnodes)
	bool retain;
	//! vector of StateNodes, just so they can be deleted again on DoStop() (unless retained) or ~StateNode()
	std::vector<StateNode*> nodes;

	//Behavior Stuff:
	//! holds the name of the Node/Machine. The first character will be '#' if it's active and '-' if it's inactive.  So strcmp(getName()+1,...) if you want to check name.
	char* name;

private:
	StateNode(const StateNode& node); //!< don't call this
	StateNode operator=(const StateNode& node); //!< don't call this
};

/*! @file
 * @brief Describes StateNode, which is both a state machine controller as well as a node within a state machine itself
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2003/03/01 20:53:26 $
 */

#endif
