#ifndef INCLUDED_RunSequenceControl_h_
#define INCLUDED_RunSequenceControl_h_

#include "ControlBase.h"
#include "Motion/MotionSequenceMC.h"
//#include "Motion/DynamicMotionSequence.h"
#include "Shared/TimeET.h"
#include <string>
#include "Shared/WorldState.h"

//! Upon activation, loads a position from a file name read from cin (stored in ms/data/motion...)
/*! The template parameter is passed to MotionSequenceTMC's template parameter in order
 *  to specify the number of keyframes to reserve - larger values use more memory, but will allow
 *  you to load more complicated sequences.
 *
 *  @see DynamicMotionSequence
 */
template<unsigned int SequenceSize>
class RunSequenceControl : public ControlBase {
 public:
	//! Constructor
	RunSequenceControl(const std::string& n) : ControlBase(n), genName(), display_id(MotionManager::invalid_MC_ID), file() {}
	//! Constructor, second parameter specifies file name to use upon activation (instead of prompting)
	RunSequenceControl(const std::string& n, const std::string& f) : ControlBase(n), genName(), display_id(MotionManager::invalid_MC_ID), file(f) {}
	//! Destructor
	virtual ~RunSequenceControl() {}

	virtual ControlBase* activate(MotionManager::MC_ID disp_id) {
		display_id=disp_id;
		if(state->button_times[HeadFrButOffset])
			file.clear();
		if(file.size()==0)
			doReadStdIn();
		else
			cout << "Loading from '" << file << "'" << endl << "Hold front head button upon activation to prompt." << endl;
		Load(file);
		return NULL;
	}

	ControlBase* doReadStdIn() {
		std::string filename;
		if(display_id!=MotionManager::invalid_MC_ID) {
			MMAccessor<LedMC> display(display_id);
			display.mc()->cset(FaceLEDMask,.5);
		}
		std::cout << "Enter posture file to load (from /ms/data/motion):" << std::endl;
		std::cout << "#> " << std::flush;
		std::cin >> filename;
		std::cout << std::endl;
		file="/ms/data/motion/"+filename;
		return NULL;
	}

	virtual const char* getName() const {
		if(file.size()==0)
			return name.c_str();
		genName=name+" (";
		genName+=file+")";
		return genName.c_str();
	}
	

 private:
	//!does the actual loading of the MotionSequence
	static void Load(std::string f) {
		TimeET timer;
		//		SharedMotion< DynamicMotionSequence > s(f.c_str());
		SharedMotion< MotionSequenceMC<SequenceSize> > s(f.c_str());
		cout << "Load Time: " << timer.Age() << endl;
		motman->addMotion(s,MotionManager::kEmergencyPriority+1,true);
	}

	mutable std::string genName; //!< the generated name (so filename can be appended)
	MotionManager::MC_ID display_id; //!< the LEDMC to use to give feedback that it's waiting for console input
	std::string file; //!< if not empty-string, will load this file upon activation instead of prompting
};

/*! @file
 * @brief Defines RunSequenceControl, which when activated, loads and runs a motion sequence from a file name read from cin (stored in ms/data/motion)
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.3 $
 * $State: Exp $
 * $Date: 2003/03/04 05:46:03 $
 */

#endif
