#include "MenuControl.h"
#include "Motion/MMAccessor.h"
#include "Motion/LedMC.h"
#include <iomanip>

ControlBase * MenuControl::activate(MotionManager::MC_ID display) {
	display_id=display;
	refresh();
	return this;
}

void MenuControl::pause() {
	if(doRewrite) {
		doRewrite=false;
	}
}

void MenuControl::refresh() {
	if(display_id!=MotionManager::invalid_MC_ID) {
		MMAccessor<LedMC> display(display_id);
		if(options.size()<=10)
			display.mc()->displayNumber(cur,LedEngine::onedigit);
		else
			display.mc()->displayNumber(cur,LedEngine::twodigit);
	}
	const char * const nosel="  ";
#ifdef CONSOLE_VT100
	const char * const slctd="\33[1m**\33[0m";
	if(doRewrite)
		clearMenu();
	cout << "\33[1m" << getName() << "\33[0m" << ':' << endl; //displays name in bold
#else
	const char * const slctd="**";
	cout << getName() << ':' << endl;
#endif
	unsigned int digits=0;
	for(unsigned int i=1; i<options.size(); i*=10)
		digits++;
	for(unsigned int i=0; i<options.size(); i++) {
		if(options[i]==NULL)
			for(unsigned int j=0; j<strlen(nosel)+digits+2; j++)
				cout << ' ';
		else
			cout << (i==cur?slctd:nosel) << std::setw(digits) << i << "> ";
		cout << getSlotName(i) << endl;
	}
	if(options.size()==0)
		cout << "  Empty menu" << endl;
	doRewrite=true;
}

void MenuControl::deactivate() {
	cur=0;
	if(display_id!=MotionManager::invalid_MC_ID) {
		MMAccessor<LedMC> display(display_id);
		display.mc()->clear();
	}
	if(doRewrite) {
#ifdef CONSOLE_VT100
		clearMenu();
#endif
		doRewrite=false;
	}
	display_id=MotionManager::invalid_MC_ID;
}

const char* MenuControl::getSlotName(unsigned int i) const {
	if(options[i]!=NULL)
		return options[i]->getName();
	else
		return "----------";
}


void MenuControl::setSlot(unsigned int i,ControlBase* o) {
	while(options.size()<=i)
		options.push_back(NULL);
	options[i]=o;
}

void MenuControl::pushSlot(ControlBase* o) {
	options.push_back(o);
}

ControlBase* MenuControl::doSelect() {
	//		cout << "MenuControl::doSelect()" << endl;
	//		cout << "cur==" << cur << endl;
	if(cur>=options.size() || options[cur]==NULL) {
		if(display_id!=MotionManager::invalid_MC_ID) {
			MMAccessor<LedMC> display(display_id);
			display.mc()->cflash(FaceLEDMask,.5,100);
		}
		if(cur>=options.size())
			cout << "Invalid choice" << endl;
		else
			cout << "NULL option" << endl;
		return this;
	}
	if(display_id!=MotionManager::invalid_MC_ID) {
		MMAccessor<LedMC> display(display_id);
		display.mc()->flash(FaceLEDMask,100);
		display.mc()->clear();
	}
	if(doRewrite) {
#ifdef CONSOLE_VT100
		clearMenu();
#endif
		doRewrite=false;
	}
	return options[cur];
}

ControlBase* MenuControl::doNextItem() {
	//		cout << "MenuControl::doNextItem()" << endl;
	do
		cur=(cur+1)%options.size();
	while(options[cur]==NULL);
	refresh();
	//		cout << "cur==" << cur << endl;
	return this;
}

ControlBase* MenuControl::doPrevItem() {
	//		cout << "MenuControl::doPrevItem()" << endl;
	do
		cur=(cur+options.size()-1)%options.size();
	while(options[cur]==NULL);
	refresh();
	//		cout << "cur==" << cur << endl;
	return this;
}

ControlBase* MenuControl::doReadStdIn() {
	unsigned int choice;
	if(display_id!=MotionManager::invalid_MC_ID) {
		MMAccessor<LedMC> display(display_id);
		display.mc()->cset(FaceLEDMask,.5);
	}
	std::cout << "#> " << std::flush;
	std::cin >> choice;
	std::cout << std::endl;
	if(choice<options.size() && options[choice]!=NULL) {
		cur=choice;
		return doSelect();
	} else {
#ifdef CONSOLE_VT100
		cout << "\r\33[1A";
		clearMenu();
		doRewrite=false;
#endif
		cout << "Invalid choice"<<endl;
		refresh();
		return this;
	}
}

#ifdef CONSOLE_VT100
void MenuControl::clearMenu() {
	cout << "\r\33[" << (options.size()+1) << 'A'; //moves cursor up to beginning of menu
	cout << "\33[J"; //clears to end of screen
}
#endif

/*! @file
 * @brief Implements MenuControl, the main Control for the heirarchical menu structure
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2003/03/03 01:18:11 $
 */

